#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use CGI;

use Lib::Logger;
use Lib::String;
use Lib::Filter;
use Lib::Mail;
use Lib::Ping;
use Lib::DateTime;
use Lib::User;
use Lib::App::Conf;
use Lib::App::Topic;
use Lib::App::Archive;

my $conf = Lib::App::Conf->new();
my $logger = Lib::Logger->new();

my $cgi = CGI->new();

my $string = Lib::String->new();

# トピックID
my $topicid = get_id($cgi);
if ($topicid eq '') {
    trackback_error('パラメーターが不正です');
}
#$logger->write("rcvtb.cgi - topicid:$topicid");# for debug

# エントリのタイトル
my $title = $cgi->param('title');
if ($title eq '') {
    trackback_error('エントリのタイトルが設定されていません');
}

# エントリの要約
my $excerpt = $cgi->param('excerpt');
if ($excerpt eq '') {
    trackback_error('エントリの要約が設定されていません');
}

# ブログ名
my $blogname = $cgi->param('blog_name');
if ($blogname eq '') {
    trackback_error('ブログ名が設定されていません');
}
# エントリのURL
my $url = $cgi->param('url');
if ($url eq '') {
    trackback_error('URLが不正です');
}

# 文字コードの種類を取得
my $code = $string->get_encode("$title $excerpt $blogname");

# 文字コードをUTF8に変換(euc,sjis,jis以外は原則utf8とみなす)
if ($code eq 'euc' || $code eq 'sjis' || $code eq 'jis') {
    $title = $string->convert($title, $code, 'utf8');
    $excerpt = $string->convert($excerpt, $code, 'utf8');
    $blogname = $string->convert($blogname, $code, 'utf8');
    $url = $string->convert($url, $code, 'utf8');
}

# 要約が半角英数のみの場合は拒否
if ($excerpt =~ /\A[\x20-\x7E\r\n]+\z/) {
    trackback_error('トラックバックが拒否されました');
}

# 拒否リストに登録されていないかチェック
my $ipaddr = $ENV{'REMOTE_ADDR'};
my $filter = Lib::Filter->new();
if ($filter->check_filter("$ipaddr $url $title $blogname")) {
    trackback_error('トラックバックが拒否されました');
}

# 要約の長さを調整
$excerpt = $string->cut($excerpt, $conf->get_excerpt_length());

# コンマをエンコード
$title =~ s/,/enc_conma/g;
$excerpt =~ s/,/enc_conma/g;
$blogname =~ s/,/enc_conma/g;
$url =~ s/,/enc_conma/g;

my $topic = Lib::App::Topic->new();
$topic->load($topicid, 1);
my $topic_status = $topic->get_status();
if ($topic_status != 1 && $topic_status != 3) {
    trackback_error('トラックバックが拒否されました');
}

my $editor = $topic->get_editor();

# トラックバックIDの設定
my @trackbacks = $topic->get_trackbacks();
my $trackbackid = 0;
if (scalar(@trackbacks) > 0) {
    $trackbackid = (split(/\,/, $trackbacks[$#trackbacks]))[0];
}
$trackbackid++;

my $dt = Lib::DateTime->new();
my $recvdate = $dt->local_datetime(0);

my $status = 1;
if ($topic_status == 3) {
    $status = 2; # 承認待ちステータスを設定
}

my $trackback = "$trackbackid,$status,$recvdate,$url,$blogname,$title,$excerpt,$ipaddr";
push @trackbacks, $trackback;
$topic->set_trackbacks(@trackbacks);

my $point = $topic->get_point() + $conf->get_normal_point();
$topic->set_point($point);

$topic->update();

# トピックのHTMLファイルを生成
$topic->get_html($conf->get_static_html(), 0);

my $docrootdir = $conf->get_docroot_dir();
my $topictitle = $topic->get_title();
my $topicurl = "$docrootdir/?act=view_topic&id=$topicid";

my $archive = Lib::App::Archive->new();
$archive->add($topicid);

# 更新Ping送信
my $ping = Lib::Ping->new();
$ping->weblogupdates($docrootdir, $conf->get_site_title());

# コンマをデコード
$title =~ s/enc_conma/,/g;
$excerpt =~ s/enc_conma/,/g;
$blogname =~ s/enc_conma/,/g;
$url =~ s/enc_conma/,/g;

my $subject = "[$topictitle] にトラックバックがつきました";

my $body = <<"END_BODY";
[$topictitle] にトラックバックがつきました。
$topicurl

----------

IPアドレス：$ipaddr
ブログ名  ：$blogname
タイトル  ：$title
URL       ：$url

要約：
$excerpt

----------
END_BODY

my $mail = Lib::Mail->new();
my $mailaddr = '';

# トピック投稿者へのメール通知
if ($editor ne '') {
    my $user = Lib::User->new();
    $user->load($editor);
    $mailaddr = $user->get_mailaddr();
    $mail->send($subject, $body, $mailaddr);
}

# サイト管理者へのメール通知
my $adminaddr = $conf->get_admin_address();
if ($conf->get_notify_mail() && $adminaddr ne $mailaddr) {
    $mail->send($subject, $body, $adminaddr);
}

my $result =  <<"END_RESULT";
<?xml encoding="utf-8"?>
<response>
<error>0</error>
<message>Thanks for your ping to $topicurl</message>
</response>
END_RESULT

print $cgi->header('text/xml'), $result;

exit;

sub get_id {
    my $cgi = shift;

    my $path_info = $cgi->path_info();
    if ($path_info eq '') {
        $path_info = $ENV{REQUEST_URI};
    }
    $path_info && $path_info =~ m!(\w+)$! && return $1;

    return;
}

sub trackback_error {
    my $msg = shift;

    my $result = <<"END_RESULT";
<?xml encoding="utf-8"?>
<response>
<error>1</error>
<message>$msg</message>
</response>
END_RESULT

    print $cgi->header('text/xml'), $result;

    exit;

}
