#include "stdafx.h"
#include "gl_handler.hpp"
#include "../timer.hpp"
#include <gpuppur/mesh/index_buffer.hpp>

using namespace gpuppur;
using namespace std;

void gl_handler::set_gpuppurut()
{
	tmpler::tmpl_set_gpuppurut(this->get_gpuppurut(), this);
}

void gl_handler::uninit(gpuppur::gpuppurut&)
{
	this->common_handler.tmpl_uninit(*this);
	this->uninit_render_sphere_resource();
	this->texture.uninitialize();
	this->triangle_buffer.uninitialize();
}

namespace
{
	void draw_sphere
	(
		std::vector<vector3>& vertices,
		std::vector<vector3>& normals,
		const int num_div_theta,
		const int num_div_phi
	)
	{
		glBegin(GL_TRIANGLE_FAN);
			glNormal3fv(normals[0].raw_data());
			glVertex3fv(vertices[0].raw_data());
			std::size_t base=1;
			std::size_t j=num_div_theta+1;
			do
			{
				--j;
				std::size_t index=base+j%num_div_theta;
				glNormal3fv(normals[index].raw_data());
				glVertex3fv(vertices[index].raw_data());
			}while(j!=0);
		glEnd();

		glBegin(GL_TRIANGLE_STRIP);
		for(int i=1; i<num_div_phi-1; ++i, base+=num_div_theta)
		for(int j=0; j<num_div_theta+1; ++j)
		{
			std::size_t index=base+j%num_div_theta;
			glNormal3fv(normals[index+num_div_theta].raw_data());
			glVertex3fv(vertices[index+num_div_theta].raw_data());
			glNormal3fv(normals[index].raw_data());
			glVertex3fv(vertices[index].raw_data());
		}
		glEnd();

		glBegin(GL_TRIANGLE_FAN);
			glNormal3fv(normals.back().raw_data());
			glVertex3fv(vertices.back().raw_data());
			for(int j=0; j<num_div_theta+1; ++j)
			{
				std::size_t index=base+j%num_div_theta;
				glNormal3fv(normals[index].raw_data());
				glVertex3fv(vertices[index].raw_data());
			}
		glEnd();
	}

	void usage()
	{
		std::cout << "------Usage-----------------------" << std::endl;
		std::cout << "'1' key: Rendering GPU Raytracing spheres" << std::endl;
		std::cout << "'2' key: Rendering spheres using small display list." << std::endl;
		std::cout << "'3' key: Rendering spheres using big display list." << std::endl;
		std::cout << "'C'(shift+c) key: Switch rendering physics simulation box." << std::endl;
		std::cout << "Esc key: Quit this program" << std::endl;
	}
}

bool gl_handler::init(gpuppurut& gut)
{
	GLenum err = glewInit();
	if (GLEW_OK != err)
	{
		cerr << "Failed to glewInit()" << endl;
		return this->find_error();
	}

	if (!GLEW_VERSION_2_0)
	{
		cerr << "OpenGL2.0 or higher is needed." << endl;
		return this->find_error();
	}

	if(!GLEW_ARB_texture_rectangle)
	{
		cerr << "GL_ARB_texture_rectangle is needed." << endl;
		return this->find_error();
	}

	if(!GLEW_ARB_pixel_buffer_object)
	{
		cerr << "GL_ARB_pixel_buffer_object is needed." << endl;
		return this->find_error();
	}
	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
//	glPixelStorei(GL_UNPACK_ALIGNMENT, 8);
//	glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
//	glShadeModel(GL_SMOOTH);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_CULL_FACE);
	glDisable(GL_DITHER);

	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	float AmbientColor[]	= { 0.4f, 0.4f, 0.4f, 1.0f };
	glLightfv(GL_LIGHT0, GL_AMBIENT, AmbientColor);
	float DiffuseColor[]	= { 0.4f, 0.7f, 0.7f, 1.0f };
	glLightfv(GL_LIGHT0, GL_DIFFUSE, DiffuseColor);
	NxVec3 tmp(0.0f, -1.0f, -1.0f);tmp.normalize();
	float Position[]		= {tmp.x, tmp.y, tmp.z, 1.0f};
	glLightfv(GL_LIGHT0, GL_POSITION, Position);

	glDisable(GL_LIGHTING);
	glDisable(GL_LIGHT0);

	if(!this->common_handler.tmpl_init(gut, *this))
	{
		return this->find_error();
	}

	GL_ERROR_ASSERT;

	::usage();

	return true;
}

void gl_handler::draw(gpuppurut& gut)
{
//	static Timer timer(80);
//	timer.start();

	if(!gut.begin_scene())
	{
		this->find_error();
		return;
	}

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	GL_ERROR_ASSERT;

	this->common_handler.tmpl_draw(gut, *this);

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glEnable(GL_DEPTH_TEST);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	this->common_handler.fix_shader.bind_shader();
	this->texture.activate(boost::mpl::void_());

//	glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

	{
		vector3 zaxis = this->camera.get_front()*-1.0f;
		vector3 yaxis = this->camera.get_up();
		vector3 xaxis = yaxis.outerProduct(zaxis);
		vector3 pos = this->camera.get_pos()*-1.0f;
//*
		float rot_mat[16] =
		{
			xaxis[0],	yaxis[0],	zaxis[0],	0.0f,
			xaxis[1],	yaxis[1],	zaxis[1],	0.0f,
			xaxis[2],	yaxis[2],	zaxis[2],	0.0f,
			0.0f,		0.0f,	0.0f,		1.0f,
		};
		glMultMatrixf(rot_mat);
		float trans_mat[16] = 
		{
			1.0f,		0.0f,	0.0f,		0.0f,
			0.0f,		1.0f,	0.0f,		0.0f,
			0.0f,		0.0f,	1.0f,		0.0f,
			pos[0],		pos[1],	pos[2],		1.0f
		};
		glMultMatrixf(trans_mat);
//*/
/*
		gluLookAt(pos[0], pos[1], pos[2],
			front[0]+pos[0],
			front[1]+pos[1],
			front[2]+pos[2],
			up[0],
			up[1],
			up[2]
		);
	/**/
	}

	if(this->sphere_rendering_mode == Displaylist)
	{
		glCallList(this->displaylist);
	}else if(this->sphere_rendering_mode == SmallMemory)
	{
		glPushMatrix();
		for(std::size_t i=0; i<sphere_depth; ++i)
		{
			glPushMatrix();
			for(std::size_t j=0; j<i*2+sphere_width; ++j)
			{
				glTranslatef(1.0f, 0.0f, 0.0f);
				glCallList(this->displaylist);
			}
			glPopMatrix();
			glTranslatef(-1.0f, 0.0f, -1.0f);
		}
		glPopMatrix();
	}else if
	(
		this->sphere_rendering_mode == VertexBuffer
		||
		this->sphere_rendering_mode == BigVertexBuffer
	)
	{
		glEnableClientState(GL_VERTEX_ARRAY);
		glEnableClientState(GL_NORMAL_ARRAY);

		this->mesh.bind_mesh
		(
			boost::mpl::void_(),
			gpuppur::mesh_opengl::sub_context_type()
		);

		matrix4x4 mat;
		mat.load_unit_matrix();

		if(this->sphere_rendering_mode == VertexBuffer)
		{
			for(std::size_t i=0; i<sphere_depth; ++i)
			{
				matrix4x4 mat2(mat);
				for(std::size_t j=0; j<i*2+sphere_width; ++j)
				{
					mat2.column(3).get_sub<0, 3>() += vector3(1.0f, 0.0f, 0.0f);
					this->mesh.draw
					(
						mat2,
						vector3(1.0f, 0.5f, 0.4f),
						boost::mpl::void_(),
						gpuppur::mesh_opengl::sub_context_type()
					);
				}
				mat.column(3).get_sub<0, 3>() += vector3(-1.0f, 0.0f, -1.0f);
			}
		}else
		{
			this->mesh.draw
			(
				mat,
				vector3(1.0f, 0.5f, 0.4f),
				boost::mpl::void_(),
				gpuppur::mesh_opengl::sub_context_type()
			);
		}

		glDisableClientState(GL_VERTEX_ARRAY);
		glDisableClientState(GL_NORMAL_ARRAY);
	}

	if(this->is_render_cube)
	{
		glEnable(GL_TEXTURE_2D);
	//	glEnable(GL_TEXTURE_RECTANGLE_ARB);

		this->phys.process();

	//	glDisable(GL_TEXTURE_RECTANGLE_ARB);
		glDisable(GL_TEXTURE_2D);
	}

	glDisable(GL_LIGHT0);
	glDisable(GL_LIGHTING);
	glDisable(GL_DEPTH_TEST);

	GL_ERROR_ASSERT;

	gut.end_scene();

//	timer.report();
}

void gl_handler::resize(gpuppurut& /*gut*/, int width, int height)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
//	gluPerspective(45.0, (double)width/height, 1.0, 1024.0);
	glFrustum(-1.0, 1.0, -1.0, 1.0, near_plane, 255.0);
	glViewport(0, 0, width, height);

	glMatrixMode(GL_MODELVIEW);
}

void gl_handler::keyboard_up(gpuppur::gpuppurut& gut, int key, int x, int y)
{
	base::keyboard_up(gut, key, x, y);

	switch(key)
	{
	case '1':
		this->uninit_render_sphere_resource();
		this->sphere_rendering_mode = Raytracing;
		::usage();
		std::cout << "\n	Rendering GPU Raytracing spheres" << std::endl;
		break;
	case '2':
		this->uninit_render_sphere_resource();
		this->sphere_rendering_mode = SmallMemory;
		this->create_small_displaylist();
		::usage();
		std::cout << "\n	Rendering spheres using display list." << std::endl;
		std::cout << "Calling displaylist many times per frame. 1 displaylist render 1 sphere." << std::endl;
		break;
	case '3':
		this->uninit_render_sphere_resource();
		this->sphere_rendering_mode = Displaylist;
		this->create_displaylist();
		::usage();
		std::cout << "\n	Rendering spheres using display list." << std::endl;
		std::cout << "Calling 1 displaylist per frame. 1 displaylist render all sphere." << std::endl;
		break;
	case '4':
		this->uninit_render_sphere_resource();
		this->sphere_rendering_mode = VertexBuffer;
		this->create_mesh(false);
		std::cout << "\n	Rendering spheres using small Vertex Buffer Object" << std::endl;
		break;
	case '5':
		this->uninit_render_sphere_resource();
		this->sphere_rendering_mode = BigVertexBuffer;
		this->create_mesh(true);
		std::cout << "\n	Rendering spheres using big Vertex Buffer Object" << std::endl;
		break;

	case 'C':
		this->is_render_cube = !this->is_render_cube;
		::usage();
		break;

	default:
		break;
	}
}

void gl_handler::uninit_render_sphere_resource()
{
	if(this->displaylist)
	{
		glDeleteLists(this->displaylist, 1);
		this->displaylist = 0;
	}

	this->mesh.release();
}

void gl_handler::create_displaylist()
{
	std::vector<vector3>	vertices;
	std::vector<vector3>	normals;

	::gen_sphere_vertex(vertices, normals, num_div_theta, num_div_phi);

	this->displaylist = glGenLists(1);
	glNewList(this->displaylist, GL_COMPILE);
		glPushMatrix();
		for(std::size_t i=0; i<sphere_depth; ++i)
		{
			glPushMatrix();
			for(std::size_t j=0; j<i*2+sphere_width; ++j)
			{
				glTranslatef(1.0f, 0.0f, 0.0f);
				::draw_sphere(vertices, normals, num_div_theta, num_div_phi);
			}
			glPopMatrix();
			glTranslatef(-1.0f, 0.0f, -1.0f);
		}
		glPopMatrix();
	glEndList();

	GL_ERROR_ASSERT;
}

void gl_handler::create_small_displaylist()
{
	std::vector<vector3>	vertices;
	std::vector<vector3>	normals;

	::gen_sphere_vertex(vertices, normals, num_div_theta, num_div_phi);

	this->displaylist = glGenLists(1);
	glNewList(this->displaylist, GL_COMPILE);
		::draw_sphere(vertices, normals, num_div_theta, num_div_phi);
	glEndList();

	GL_ERROR_ASSERT;
}

void gl_handler::create_mesh(bool is_big)
{
	struct inr
	{
		static void add_1_sphere
		(
			std::vector<vector3>&			vertices,
			std::vector<vector3>&			normals,
			std::vector<unsigned int>&		indices,
			const vector3&					position
		)
		{
			std::size_t base=vertices.size();

			::gen_sphere_vertex(vertices, normals, num_div_theta, num_div_phi, position);

			for(std::size_t i=0; i<num_div_theta; ++i)
			{
				indices.push_back(static_cast<unsigned int>(base));
				indices.push_back
				(
					static_cast<unsigned int>
					(
						base+1+(i+1)%num_div_theta
					)
				);
				indices.push_back
				(
					static_cast<unsigned int>(base+1+i)
				);
			}

			base += 1;
			for(int i=1; i<num_div_phi-1; ++i, base+=num_div_theta)
			for(int j=0; j<num_div_theta; ++j)
			{
				unsigned int
				index1	= static_cast<unsigned int>(base+j),
				index2	= static_cast<unsigned int>(base+(j+1)%num_div_theta),
				index3	= static_cast<unsigned int>(index2+num_div_theta),
				index4	= index1+num_div_theta;

				indices.push_back(index1);
				indices.push_back(index3);
				indices.push_back(index4);

				indices.push_back(index1);
				indices.push_back(index2);
				indices.push_back(index3);
			}

			for(std::size_t i=0; i<num_div_theta; ++i)
			{
				indices.push_back(static_cast<unsigned int>(base+num_div_theta));
				indices.push_back(static_cast<unsigned int>(base+i));
				indices.push_back(static_cast<unsigned int>(base+(i+1)%num_div_theta));
			}
		}
	};

	struct mesh_feeder
	{
		void get_mesh_data
		(
			std::vector<vector3>&			vertices,
			std::vector<vector3>&			normals,
			std::vector<vector2>&			,
			gpuppur::index_buffer&			index_buf
		) const
		{
			vertices.clear();
			normals.clear();
			std::vector<unsigned int>	indices;

			if(!this->is_big)
			{
				inr::add_1_sphere(vertices, normals, indices, vector3(0.0f, 0.0f, 0.0f));
			}else
			{
				vector3 pos_depth(0.0f, 0.0f, 0.0f);
				for(std::size_t i=0; i<sphere_depth; ++i)
				{
					vector3 pos(pos_depth);
					for(std::size_t j=0; j<i*2+sphere_width; ++j)
					{
						pos += vector3(1.0f, 0.0f, 0.0f);
						inr::add_1_sphere
						(
							vertices, normals, indices,
							pos
						);
					}
					pos_depth += vector3(-1.0f, 0.0f, -1.0f);
				}
			}

			index_buf.load_data
			(
				vertices.size(),
				&indices[0],
				boost::type<unsigned int>(),
				sizeof(unsigned int)*3,
				indices.size()
			);
		}

		operator bool () const
		{
			return true;
		}

		mesh_feeder(bool is_big):
			is_big(is_big)
		{
		}

		bool	is_big;
	};

	this->mesh.mesh_opengl::mesh_opengl
	(
		mesh_feeder(is_big),
		boost::mpl::void_(),
		gpuppur::mesh_opengl::gl_shader()
	);
}
