#include "stdafx.h"
#include "nxphysics.hpp"
#include <gpuppur/utility/frame_per_second.hpp>

using namespace std;
using namespace boost;
using namespace boost::posix_time;

namespace
{
	const float cube_size = 1.0f;
}

physics::physics()
:physics_sdk(NULL), scene(NULL)
{
}

physics::~physics()
{
	this->uninitialize();
}

void physics::uninitialize()
{
	for(std::vector<NxActor*>::iterator i = this->actors.begin(); i!=this->actors.end(); i++)
	{
		this->scene->releaseActor(**i);
	}

	this->actors.clear();

	if(this->scene != NULL)
	{
		this->physics_sdk->releaseScene(*this->scene);
		this->scene = NULL;
	}
}

bool physics::initialize()
{
	NxPhysicsSDKDesc sdkDesc;
//	sdkDesc.flags |= NX_SDKF_NO_HARDWARE;	//Disable Hardware acceleration
	this->physics_sdk = NxCreatePhysicsSDK(NX_PHYSICS_SDK_VERSION, 0, 0, sdkDesc);
	if(!this->physics_sdk)
		return false;

	bool has_hard;
	if(this->physics_sdk->getHWVersion() == NX_HW_VERSION_NONE)
	{
		std::cout << "No PhysX hardware" << std::endl;
		has_hard = false;
	}else
	{
		std::cout << "PhysX hardware available" << std::endl;
		has_hard = true;
	}

	NxSceneDesc scene_desc;

	scene_desc.simType = has_hard ? NX_SIMULATION_HW : NX_SIMULATION_SW;
	scene_desc.flags |= has_hard ? NX_SF_RESTRICTED_SCENE : 0;
	scene_desc.gravity.set(0.0f, -9.8f, 0.0f);
//	scene_desc.collisionDetection = true;

	this->scene = this->physics_sdk->createScene(scene_desc);
	if(!this->scene)
		return false;

	NxMaterial * defaultMaterial = this->scene->getMaterialFromIndex(0); 
	defaultMaterial->setRestitution(0.3f);
	defaultMaterial->setStaticFriction(0.8f);
	defaultMaterial->setDynamicFriction(0.9f);

	NxPlaneShapeDesc PlaneDesc;
	NxActorDesc ActorDesc;
	PlaneDesc.normal = NxVec3(0.0f, 1.0f, 0.0f);
	PlaneDesc.d = 0.0f;
	ActorDesc.shapes.pushBack(&PlaneDesc);
	this->scene->createActor(ActorDesc);
//	this->actors.push_back(this->scene->createActor(ActorDesc));

	const size_t h = 9;
	const float offset=cube_size/100000.0f;
	const float hoffset=cube_size/100000.0f;
	const vector3 pos(0.0f, 1.0f, -20.0f);
	for(int k=0; k<h; ++k)
	for(int j=0; j<k+1; ++j)
	for(int i=0; i<k+1; ++i)
	{
		float x = (i-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float z = (j-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float y = k*(cube_size*2+hoffset);
		this->create_cube(vector3(x, y, z) + pos, vector3(0.0f, 0.0f, 0.0f));
	}

	for(int k=h-2; k>=0; --k)
	for(int j=0; j<k+1; ++j)
	for(int i=0; i<k+1; ++i)
	{
		float x = (i-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float z = (j-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float y = (h*2-k-2)*(cube_size*2+hoffset);
		this->create_cube(vector3(x, y, z) + pos, vector3(0.0f, 0.0f, 0.0f));
	}

	for(int k=0; k<h*3; ++k)
	{
		float y = (h*2+k-1)*(cube_size*2+hoffset);
		this->create_cube(vector3(0.0f, y, 0.0f) + pos, vector3(0.0f, 0.0f, 0.0f));
	}

	return true;
}

//GL_TEXTURE_RECTANGLE_NVを使う場合、この変数に画像の実際のサイズを入れる。 
const GLfloat tex_width = 1.0f;//32;
const GLfloat tex_height = 1.0f;//32;

static void draw_cude()
{
	const float half_size = cube_size;

	glBegin(GL_TRIANGLE_STRIP);
//	glBegin(GL_LINE_STRIP);
		// front face
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f((float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, tex_height);
		glNormal3f((float)M_SQRT1_3, -(float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(half_size, -half_size, half_size);

		//bottom face
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f((float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(half_size, -half_size, -half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, -half_size);

		//left face
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, tex_height);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, -half_size);

		//top face
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f((float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, 0.0f);
		glNormal3f((float)M_SQRT1_3, (float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(half_size, half_size, -half_size);

		//right face
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f((float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(half_size, -half_size, -half_size);

		//back face
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, -half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, tex_height);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, -half_size);
	glEnd();
}

void physics::process()
{
	assert(this->physics_sdk != NULL);
	assert(this->scene != NULL);

	float mat[16];

	std::vector<NxActor*>::iterator i = this->actors.begin();
	for(;i!=this->actors.end(); ++i)
	{
		(*i)->getGlobalPose().getColumnMajor44(mat);

		glPushMatrix();
		glMultMatrixf(mat);
		draw_cude();
		glPopMatrix();
	}

	this->scene->simulate(gpuppur::frame_per_second::get_spf());
	this->scene->flushStream();
	this->scene->fetchResults(NX_RIGID_BODY_FINISHED, true);
}

void physics::create_cube(const vector3& pos, const vector3& initial_velocity)
{
	assert(this->physics_sdk != NULL);
	assert(this->scene != NULL);

	// Create body
	NxBodyDesc BodyDesc;
	BodyDesc.angularDamping		= 0.5f;
//	BodyDesc.maxAngularVelocity	= 10.0f;
	vector3 vel					= initial_velocity;
	BodyDesc.linearVelocity		= *reinterpret_cast<NxVec3*>(&vel);

	NxBoxShapeDesc BoxDesc;
	BoxDesc.dimensions			= NxVec3(cube_size, cube_size, cube_size);

	NxActorDesc ActorDesc;
	ActorDesc.userData			= NULL;
	ActorDesc.shapes.pushBack(&BoxDesc);
	ActorDesc.body				= &BodyDesc;
	ActorDesc.density			= 10.0f;
	vector3 tmp_pos				= pos;
	ActorDesc.globalPose.t		= *reinterpret_cast<NxVec3*>(&tmp_pos);

	NxActor* tmp = this->scene->createActor(ActorDesc);
	tmp->userData = NULL;
	this->actors.push_back(tmp);
}