#ifndef HANDLER_HPP
#define HANDLER_HPP

#include <cmath>

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <iostream>
#include <boost/bind.hpp>
#include <boost/mpl/list.hpp>
#include <boost/mpl/for_each.hpp>
#include <boost/utility.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/gpuppurut/camera.hpp>
#include <gpuppur/utility/script.hpp>
#include <gpuppur/utility/image_io.hpp>

class handler_dynamic_base
{
public:

	virtual void uninit(gpuppur::gpuppurut& gut) = 0;
	virtual bool init(gpuppur::gpuppurut& gut) = 0;
	virtual void draw(gpuppur::gpuppurut& gut) = 0;
	virtual gpuppur::gpuppur_virtual& get_gpuppur() = 0;
	virtual gpuppur::gpuppurut& get_gpuppurut() = 0;		
};

template<class typeList>
class handler_dynamic
{
public:

	typedef handler_dynamic<typeList>	this_type;
	typedef	gpuppur::gpuppurut			gpuppurut_type;
	typedef gpuppur::gpuppur_virtual	gpuppur_type;

	class pimpl_setter
	{
	public:

		template<typename T>
		void operator()(T&)
		{
			hd.pimpl_vector.push_back
			(
				boost::shared_ptr<handler_dynamic_base>
				(
					new T
				)
			);
		}

		pimpl_setter(handler_dynamic& hd):
			hd(hd)
		{
		}

		pimpl_setter(const pimpl_setter& ps):
			hd(ps.hd)
		{
		}

	private:
		handler_dynamic& hd;

		pimpl_setter& operator = (const pimpl_setter& rhs);
	};

	handler_dynamic()
	{
		boost::mpl::for_each<typeList>
		(
			pimpl_setter(*this)
		);

		this->pimpl = this->pimpl_vector[0];
	}

	~handler_dynamic()
	{
	}

	void set_impl(handler_dynamic_base* pimpl)
	{
		this->pimpl = pimpl;
	}

protected:

	void uninit(gpuppur::gpuppurut& gut)
	{
		return this->pimpl.lock()->uninit(gut);
	}

	bool init(gpuppur::gpuppurut& gut)
	{
		return this->pimpl.lock()->init(gut);
	}

	void draw(gpuppur::gpuppurut& gut)
	{
		return this->pimpl.lock()->draw(gut);
	}

	gpuppur::gpuppur_virtual& get_gpuppur()
	{
		return this->pimpl.lock()->get_gpuppur();
	}

	gpuppur::gpuppurut& get_gpuppurut()
	{
		return this->pimpl.lock()->get_gpuppurut();
	}

private:

	std::vector<boost::shared_ptr<handler_dynamic_base> >
												pimpl_vector;
	boost::weak_ptr<handler_dynamic_base>		pimpl;
};

template<bool>
class dep_init;

template<>
class dep_init<true>
{
public:
	static bool init()
	{
		GLenum err = glewInit();
		if (GLEW_OK != err)
		{
			std::cerr << "Failed to glewInit()" << std::endl;
			return false;
		}

		if (!GLEW_VERSION_2_0)
		{
			std::cerr << "OpenGL2.0 or higher is needed." << std::endl;
			return false;
		}

		if(!GLEW_ARB_texture_rectangle)
		{
			std::cerr << "GL_EXT_texture_rectangle is needed." << std::endl;
			return false;
		}

		if(!GLEW_ARB_pixel_buffer_object)
		{
			std::cerr << "GL_ARB_pixel_buffer_object is needed." << std::endl;
			return false;
		}

		return true;
	}
};

template<>
class dep_init<false>
{
public:
	static bool init()
	{
		return true;
	}
};

template<bool>
class init_specific_gpuppur;

template<>
class init_specific_gpuppur<true>
{
public:

	template<class GPUPPUR>
	static bool init(GPUPPUR& gpuppur)
	{
		gpuppur.set_sampling_step
		(
			gpuppur::get_global_script().get_value("sampling_step", 1)
		);

		return true;
	}
};

template<>
class init_specific_gpuppur<false>
{
public:

	template<class GPUPPUR>
	static bool init(GPUPPUR&)
	{
		return true;
	}
};

template<bool Renderer, bool Implement>
class handler_static
{
public:
	typedef handler_static<Renderer, Implement> this_type;

	typedef typename boost::mpl::if_c
	<
		Implement,
		gpuppur::gglut,
		gpuppur::gdxut
	>::type gpuppurut_type;

	typedef typename boost::mpl::if_c
	<
		Renderer,
		typename boost::mpl::if_c
		<
			Implement,
			gpuppur::gl_raytracer_static,
			gpuppur::dx_raytracer_static
		>::type,
		typename boost::mpl::if_c
		<
			Implement,
			gpuppur::gl_rasterizer_static,
			gpuppur::dx_rasterizer_static
		>::type
	>::type gpuppur_type;


protected:

	void uninit(gpuppur::gpuppurut&)
	{
	}

	bool init(gpuppur::gpuppurut&)
	{
		if(!dep_init<Implement>::init())
		{
			return false;
		}

		if(!init_specific_gpuppur<Renderer>::init(this->my_gpuppur))
		{
			return false;
		}

		return true;
	}

	void draw(gpuppur::gpuppurut&)
	{
	}

	gpuppur_type& get_gpuppur()
	{
		return this->my_gpuppur;
	}

	gpuppur::gpuppurut& get_gpuppurut()
	{
		return this->my_gpuppurut;
	}

private:

	gpuppur_type		my_gpuppur;
	gpuppurut_type		my_gpuppurut;
};

template<typename Base>
class gpuppurut_handler : public Base, boost::noncopyable
{
public:

	typedef Base											base;
	typedef gpuppurut_handler<base>							this_type;
	typedef typename Base::gpuppurut_type					gpuppurut_type;
	typedef typename Base::gpuppur_type						gpuppur_type;
	typedef typename gpuppur_type::instance3d				instance3d_type;
	typedef typename gpuppur_type::mesh						mesh_type;
//	typedef typename gpuppur_type::texture<vector4>::type	texture_type;

	void set_gpuppurut(gpuppur::gpuppurut& gut)
	{
		gut.setUninit3DContextFunc(boost::bind(&this_type::uninit, this, _1));
	//	gut.setUninit3DContextFunc(boost::bind(&this_type::~gpuppurut_handler, this, _1));
		gut.setInit3DContextFunc(boost::bind(&this_type::init, this, _1));
		gut.setRedrawFunc(boost::bind(&this_type::draw, this, _1));
		gut.setResizeFunc(boost::bind(&this_type::resize, this, _1, _2, _3));
		gut.setKeyPressingFunc(boost::bind(&this_type::keyboard, this, _1, _2, _3, _4));
		gut.setKeyUpFunc(boost::bind(&this_type::keyboard_up, this, _1, _2, _3, _4));
		gut.setMouseFunc(boost::bind(&this_type::motion, this, _1, _2, _3, _4, _5));
	}

	gpuppurut_handler():
		scr_width(/*128*/512), scr_height(/*128*/512),
		camera(0.0f, 0.0f, 5.0f),
		input_mode(camera_move)
	{
		this->scr_width
		=
		gpuppur::get_global_script().get_value("screen_width", 128);
		this->scr_height
		=
		gpuppur::get_global_script().get_value("screen_height", 128);

		this->set_gpuppurut(this->get_gpuppurut());
	}

private:

	void gen_mesh
	(
		std::vector<vector3>&			vertices,
		std::vector<unsigned short>&	indices
	)
	{
		//Generate Donut mesh
		const std::size_t h_div = 200;
		const std::size_t v_div = 300;
		const float h_rad = 8.0f;
		const float v_rad = 2.0f;

		//Generate vertex positions
		for(std::size_t i=0; i<h_div; ++i)
		{
			float rate = static_cast<float>(i)/h_div;
			vector3 h_point
			(
				vector3(cos(2.0f*M_PIf*rate), 0.0f, -sin(2.0f*M_PIf*rate))*h_rad
			);
			vector3 x_dir(h_point.getNormalized());

			for(std::size_t j=0; j<v_div; ++j)
			{
				float rate = static_cast<float>(j)/v_div;

				vector3 local_p
				(
					v_rad*cos(2.0f*M_PIf*rate), v_rad*sin(2.0f*M_PIf*rate), 0.0f
				);
				matrix3x3 to_global(x_dir, vector3(0.0f, 1.0f, 0.0f), vector3(0.0f));
				vertices.push_back(vector3(to_global*local_p+h_point));
			}
		}

		//Generate triangles
		unsigned short count = 0;
		unsigned short offset = static_cast<unsigned short>(v_div);
		for(std::size_t i=0; i<h_div-1; ++i)
		{
			for(std::size_t j=0; j<v_div-1; ++j)
			{
				indices.push_back(count);
				indices.push_back(count+offset);
				indices.push_back(count+1);

				indices.push_back(count+1);
				indices.push_back(count+offset);
				indices.push_back(count+offset+1);
				++count;
			}
			++count;
		}

/*
		//Generate Simplest mesh
		vertices.push_back(vector3(0.0f, 1.0f, 0.0f));
		vertices.push_back(vector3(0.0f, 0.0f, 1.0f));
		vertices.push_back(vector3(1.0f, 0.0f, 0.0f));
		vertices.push_back(vector3(-1.0f, 0.0f, 0.0f));

		indices.push_back(0);indices.push_back(1);indices.push_back(2);
		indices.push_back(1);indices.push_back(0);indices.push_back(3);
		indices.push_back(0);indices.push_back(2);indices.push_back(3);
		indices.push_back(1);indices.push_back(3);indices.push_back(2);
*/
	}

	bool init_3d_model()
	{
		std::vector<vector3>			vertices;
		std::vector<unsigned short>		indices;

	//	this->gen_mesh(vertices, indices);
	//
		this->my_instances.clear();
		this->my_meshes.clear();

		this->my_meshes.push_back
		(
			this->get_gpuppur().create_mesh
			(
				"../data/bunny.obj"
			//	"../data/model.obj"
			//	"../data/Armadillo.obj"
			//	"../data/sphere.obj"
			//	"../data/leg6.obj"
			)
		);
	//	this->my_meshes.push_back(this->get_gpuppur().create_mesh(vertices, indices));
	/*	this->my_meshes.push_back
		(
			this->get_gpuppur().create_mesh("../data/obj.obj")
		);
	*/
	//	this->my_meshes.push_back(this->get_gpuppur().create_mesh("../data/test.obj"));
	//	mesh_type tmp();

	//	tmp.release();

		for(std::size_t i=0; i<this->my_meshes.size(); ++i)
		{
			if(!this->my_meshes[i])
			{
				std::cerr << "Failed to load mesh!" << std::endl;
				return false;
			}
		}

		gpuppur::material green(vector3(0.0f, 1.0f, 0.0f));
		gpuppur::material red(vector3(1.0f, 0.0f, 0.0f));

		this->my_instances.push_back
		(
			this->get_gpuppur().create_instance
			(
				this->my_meshes[0],
				vector3(0.0f, 0.0f, -4.0f),	//initial pos
			//	vector3(-2.0f, 2.0f, 0.0),
				vector3(0.7f, 0.6f, 0.5f)	//color
			)
		);

		// create hoge instances
/*		for(std::size_t i=0; i<20; ++i)
		{
			this->my_instances.push_back
			(
				this->get_gpuppur().create_instance
				(
					this->my_meshes[1],
					vector3(20.0f, 5.0f, -15.0f*static_cast<float>(i)),
					red
				)
			);
		}

		for(std::size_t i=0; i<20; ++i)
		{
			this->my_instances.push_back
			(
				this->get_gpuppur().create_instance
				(
					this->my_meshes[1],
					vector3(-20.0f, 5.0f, -10.0f-15.0f*static_cast<float>(i)),
					green
				)
			);
		}

*/
		for(std::size_t i=0; i<this->my_instances.size(); ++i)
		{
			if(!this->my_instances[i])
			{
				std::cerr << "Failed to create instance!!" << std::endl;
				return false;
			}
		}
/*
		this->my_instances.push_back
		(
			this->get_gpuppur().create_instance
			(
				this->my_meshes[2],
				vector3(0.0f, 8.0f, -12.0f)
			)
		);
*/
/*		this->my_sphere
		=
		this->get_gpuppur().create_sphere
		(
			8.0f,
			vector3(0.0f, 0.0f, -12.0f)
		);
*/
/*
		matrix3x3 mat;
		mat.load_unit_matrix();

		this->my_instances.set_orientation(mat);
*/
/*
		gpuppur::image img;
		if(!img.load_image("../data/env_map.png"))
		{
			std::cerr << "Failed to load env map image!" << std::endl;
			return false;
		}

		const std::size_t w=img.get_width(), h=img.get_height();
		this->my_texture
		=
		this->get_gpuppur().create_texture<vector4>(w, h);
		if(!this->my_texture)
		{
			std::cerr << "Failed to create texture!:p" << std::endl;
			return false;
		}

		this->my_texture.lock_for_write();
		for(std::size_t i=0; i<h; ++i)
		for(std::size_t j=0; j<w; ++j)
		{
			this->my_texture.set
			(
			//	vector4
			//	(
			//		static_cast<float>(j)/w, static_cast<float>(i)/h, 0.0f, 0.0f
			//	)

				img.get_pixel()
			);
		}
		this->my_texture.unlock();
*/
		return true;
	}

protected:

	void uninit(gpuppur::gpuppurut& gut)
	{
	//	this->my_texture.release();
		this->my_instances.clear();
		this->my_meshes.clear();
		this->my_sphere.release();
		this->get_gpuppur().uninitialize();
		base::uninit(gut);
	}

	bool init(gpuppur::gpuppurut& g)
	{
		this->uninit(g);

		if(!base::init(g))
		{
			return false;
		}

		gpuppurut_type& gut = static_cast<gpuppurut_type&>(g);
		if
		(
			!this->get_gpuppur().initialize
			(
				this->scr_width, this->scr_height,
				gut.get_context()
			)
		)
		{
			std::cerr << "Failed to initialize gpuppur." << std::endl;
			return false;
		}

		if(!this->init_3d_model())
		{
			return false;
		}

		return true;
	}

	void draw(gpuppur::gpuppurut& gut)
	{
		gpuppur::matrixRxC<float, 3, 3> mat;
		mat.load_unit_matrix();

		static float angle = 0.0f;
		angle += 0.1f;
		angle = angle > 2*M_PIf ? angle-2*M_PIf : angle;

		rotate_2vectors(mat.row(2), mat.row(1), angle);
		this->my_instances[0].set_orientation(mat);

		this->my_instances[0].set_position
		(
			vector3(0.0f, 4.0f*sin(angle*2.0f), -10.0f)
		);

		std::size_t num_ins = this->my_instances.size();
		for(std::size_t i=1; i<num_ins; ++i)
		{
		#if 0
			buggy!
			this->my_instances[i].set_position
			(
				this->my_instances[i].get_position()
				+
				vector3
				(
					0.0f,
					1.0f*sin(angle*1.0f/*+static_cast<float>(i)/num_ins*3.0f*/),
					0.0f
				)
			);
		#endif
			gpuppur::matrixRxC<float, 3, 3> mat;
			mat.load_unit_matrix();
			rotate_2vectors
			(
				mat.row(1), mat.row(0),
				angle+static_cast<float>(i)/num_ins*6.0f
			);
			this->my_instances[i].set_orientation(mat);
		}

		this->camera.process();
		this->get_gpuppur().look_at
		(
			this->camera.get_pos(),
			this->camera.get_front(),
			this->camera.get_up()
		);

		if(!gut.begin_scene(true))
		{
			return;
		}

		this->get_gpuppur().process();

		base::draw(gut);

		{
			/*
			int s = this->get_gpuppur().get_ppu().get_count_sampling();

			std::cout
				<< "count sampling:"<<s<<"/"<<this->scr_width*this->scr_height
				<< "(" << 100.0f * float(s)/(this->scr_width*this->scr_height) << "%)"
				<< std::endl;
			*/
		}

		gut.end_scene();
	}

	void resize(gpuppur::gpuppurut& /*gut*/, int width, int height)
	{
		this->get_gpuppur().set_viewport(0, 0, width, height);
	//	this->get_gpuppur().set_viewport(0, 0, 64, 64);
	}

	void lens_keyboard(int key)
	{
		static float l=-1.0f, r=1.0f, b=-1.0f, t=1.0f, n=1.0f, f=256.0f;
		const float speed = 0.1f;
		static bool is_perspective = true;

		switch(key)
		{
		case 'l':
			l += speed;
			break;
		case 'L':
			l -= speed;
			break;

		case 'r':
			r += speed;
			break;
		case 'R':
			r -= speed;
			break;

		case 'b':
			b += speed;
			break;
		case 'B':
			b -= speed;
			break;

		case 't':
			t += speed;
			break;
		case 'T':
			t -= speed;
			break;

		case 'n':
			n += speed;
			break;
		case 'N':
			n -= speed;
			break;

		case 'f':
			f += speed*100.0f;
			break;
		case 'F':
			f -= speed*100.0f;
			break;

		case 'z':
			l += speed;
			r -= speed;
			b += speed;
			t -= speed;
			break;

		case 'Z':
			l -= speed;
			r += speed;
			b -= speed;
			t += speed;
			break;

		case 'p':
			is_perspective = true;
			break;

		case 'o':
			is_perspective = false;
			break;

		default:
			break;
		}


		std::cout << "l="<<l<<", r="<<r<<", b="<<b<<", t="<<t<<", n="<<n<<", f="<<f << std::endl;
		std::cout << (is_perspective ? "perspective" : "orthographic") << std::endl;

		this->get_gpuppur().set_projection(l, r, b, t, n, f, is_perspective);
	}

	void mode_keyboard(int key)
	{
		switch(this->input_mode)
		{
		case camera_move:
			this->camera.keyboard(key);
			break;
			
		case lens_setting:
			this->lens_keyboard(key);
			break;
		}
	}

	void mode_keyboard_up(int key)
	{
		//Set Mode by Function key
		switch(key)
		{
		case gpuppur::GPUPPURUTBase::F1:
			this->input_mode = camera_move;
			std::cout << "camera move mode" << std::endl;
			break;

		case gpuppur::GPUPPURUTBase::F2:
			this->input_mode = lens_setting;
			std::cout << "lens setting mode" << std::endl;
			break;
			
		default:
			break;
		}

		switch(this->input_mode)
		{
		case camera_move:
			this->camera.keyboard_up(key);
			break;
			
		case lens_setting:
			break;
		}
	}

	void keyboard(gpuppur::gpuppurut& gut, int key, int /*x*/, int /*y*/)
	{
		this->mode_keyboard(key);

		switch(key)
		{

		case '\033':	//esc key
			gut.uninitialize();
			break;

		case '1':
			this->scr_width = this->scr_height = 256;
			this->get_gpuppur().initialize
			(
				this->scr_width, this->scr_height,
				gut.get_context()
			);
			this->init_3d_model();
			break;

		case '2':
			this->scr_width = this->scr_height = 512;
			this->get_gpuppur().initialize
			(
				this->scr_width, this->scr_height,
				gut.get_context()
			);
			this->init_3d_model();
			break;

		case '3':
			this->scr_width = this->scr_height = 1024;
			this->get_gpuppur().initialize
			(
				this->scr_width, this->scr_height,
				gut.get_context()
			);
			this->init_3d_model();
			break;

	/*	case 'l':
			this->my_instance.release();
			this->my_mesh
			=
			this->get_gpuppur().create_mesh_from_cooked
			("../data/test_low.obj.cooked");

			this->my_instance
			=
			this->get_gpuppur().create_instance
			(
				this->my_mesh, vector3(1.0f, 2.0f, -18.0f)
			);
	*/		break;
		}
	}

	void keyboard_up(gpuppur::gpuppurut& /*gut*/, int key, int /*x*/, int /*y*/)
	{
		this->mode_keyboard_up(key);
	}

	void motion(gpuppur::gpuppurut& /*gut*/, int button, int state, int x, int y)
	{
		if
		(
			button == gpuppur::gpuppurut::MOUSE_LEFT
			&&
			state == gpuppur::gpuppurut::KEY_DOWN
		)
		{
			this->camera.begin_moving(x, y);
		}else if
		(
			button == gpuppur::gpuppurut::MOUSE_LEFT
			&&
			state == gpuppur::gpuppurut::KEY_UP
		)
		{
			this->camera.end_moving();
		}

		this->camera.move_mouse(x, y);
	}

public:

	gpuppur_type& get_gpuppur()
	{
		return base::get_gpuppur();
	}

	gpuppur::gpuppurut& get_gpuppurut()
	{
		return base::get_gpuppurut();
	}

	gpuppur::camera					camera;
	size_t							scr_width, scr_height;

	instance3d_type					my_sphere;
	std::vector<instance3d_type>	my_instances;
	std::vector<mesh_type>			my_meshes;
//	texture_type					my_texture;

	enum {camera_move, lens_setting}
	input_mode;
};

#endif
