#ifndef LIBRARY

#include "dx_tester.hpp"
#include <gpuppur/error_manage/error_report_dx.hpp>
#include <gpuppur/gpuppurut/gdxut.hpp>
#include <boost/random.hpp>
#include <boost/bind.hpp>
#include <cmath>

using namespace boost;
using namespace gpuppur;

namespace
{
	boost::mt19937			gen;
	boost::uniform_int<>	binary_dst(0, 1);
	boost::variate_generator<boost::mt19937, boost::uniform_int<> >
							binary_rand(gen, binary_dst);
}

dx_viewer::dx_viewer(add_func add, quit_func quit):
	tester(add, quit),
	pressed_left(false),	pressed_right(false),
	current_pos_x(0),		current_pos_y(0),
	current_rot_x(0),		current_rot_y(0)
{
}

dx_viewer::~dx_viewer()
{
}

void dx_viewer::resize(gpuppurut& gut, int width, int height)
{
	gdxut& d3d = static_cast<gdxut&>(gut);
	D3DXMATRIX mat;

	D3DXMatrixPerspectiveFovLH(&mat, 3.14f/4.0f, (float)width/height, 1.0f, 100.0f);
	d3d.get_device()->SetTransform(D3DTS_PROJECTION, &mat);

	D3DVIEWPORT9 vp = {0, 0, width, height, 0.0f, 1.0f};
	d3d.get_device()->SetViewport(&vp);
}

void dx_viewer::draw(gpuppurut& gut)
{
	gdxut& d3d = static_cast<gdxut&>(gut);

	if(gut.get_is_visible() != gpuppurut::ReturnCode::Visible)
	{
		return;
	}

	D3DXMATRIX rot_mat, trans_mat, tmp_mat;
	D3DXMatrixRotationYawPitchRoll(&rot_mat, this->current_rot_x/256.0f, (float)this->current_rot_y/256.0f, 0.0f);
	D3DXMatrixTranslation(&trans_mat, this->current_pos_x/100.0f, -this->current_pos_y/100.0f, 4.0f);

	d3d.get_device()->SetTransform(D3DTS_WORLDMATRIX(0), D3DXMatrixMultiply(&tmp_mat, &rot_mat, &trans_mat));

	if(d3d.begin_scene())
	{
		this->teapot->DrawSubset(0);
		d3d.end_scene();
	}
}

bool dx_viewer::init_device(gpuppurut& gut)
{
	gdxut& d3d = static_cast<gdxut&>(gut);

	d3d.get_device()->LightEnable(0, TRUE);
	d3d.get_device()->SetFVF(D3DFVF_XYZ | D3DFVF_NORMAL);

	D3DLIGHT9 light;
	ZeroMemory(&light, sizeof(light));
	light.Type = D3DLIGHT_DIRECTIONAL;
	light.Diffuse.r = 0.0f;
	light.Diffuse.g = 1.0f;
	light.Diffuse.b = 0.0f;
	D3DCOLORVALUE ambient = {0.2f, 0.2f, 0.2f, 1.0f};
	light.Ambient = ambient;
	light.Direction = D3DXVECTOR3(0.2f, -0.4f, 0.6f);

	d3d.get_device()->SetLight(0, &light);

	D3DMATERIAL9 material = {
		{1.0f, 1.0f, 1.0f, 1.0f},	//diffuse
		{0.2f, 0.2f, 0.2f, 1.0f},	//ambient
		{0.0f, 0.0f, 0.0f, 1.0f},	//specular
		{0.2f, 0.2f, 0.2f, 1.0f}	//emissive
	};
	d3d.get_device()->SetMaterial(&material);

	LPD3DXMESH tmp;
	LPD3DXBUFFER padj;
	B_RET(D3DXCreateTeapot(d3d.get_device(), &tmp, &padj));
	B_RET(tmp->Optimize(D3DXMESHOPT_ATTRSORT, (const DWORD*)padj->GetBufferPointer(), NULL, NULL, NULL, &this->teapot));
	SAFE_RELEASE(tmp);
	SAFE_RELEASE(padj);

	d3d.set_clear_color(50, 90, 60);

	V(("test error reporter", D3DERR_OUTOFVIDEOMEMORY));

	return true;
}

void dx_viewer::uninit_device(gpuppur::gpuppurut&)
{
	SAFE_RELEASE(this->teapot);
}

void dx_viewer::key_pressing(gpuppurut& gut, int key, int /*mouse_x*/, int /*mouse_y*/)
{
	const int rot_speed = 32;
	const int mov_speed = 32;

	if(key == gpuppurut::LEFT)
	{
		this->current_rot_x -= rot_speed;
	}else if(key == gpuppurut::RIGHT)
	{
		this->current_rot_x += rot_speed;
	}else if(key == gpuppurut::UP)
	{
		this->current_rot_y += rot_speed;
	}else if(key == gpuppurut::DOWN)
	{
		this->current_rot_y -= rot_speed;
	}else if(key == 'h')
	{
		this->current_pos_x -= mov_speed;
	}else if(key == 'l')
	{
		this->current_pos_x += mov_speed;
	}else if(key == 'j')
	{
		this->current_pos_y += mov_speed;
	}else if(key == 'k')
	{
		this->current_pos_y -= mov_speed;
	}

	gut.redraw();
}

void dx_viewer::key_up(gpuppurut& gut, int key, int, int)
{
	gdxut& d3d = static_cast<gdxut&>(gut);

	if(key == gpuppurut::F1)
	{
		gut.set_fullscreen(!d3d.get_is_fullscreen());
	}else if(key == '\x1b')
	{
		gut.uninitialize();
	}else if(key == 'a')
	{
		this->add();
	}else if(key == 'Q')
	{
		this->quit();
	}
}

void dx_viewer::mouse(gpuppurut& gut, int button, int state, int x, int y)
{
	if(state == gpuppurut::NONE)
	{
		if(this->pressed_left)
		{
			this->current_pos_x = x - this->begin_mouse_x + this->begin_x;
			this->current_pos_y = y - this->begin_mouse_y + this->begin_y;
		}else if(this->pressed_right)
		{
			this->current_rot_x = x - this->begin_mouse_x + this->begin_x;
			this->current_rot_y = y - this->begin_mouse_y + this->begin_y;
		}else
		{
			return;
		}

		gut.redraw();
		return;
	}

	if(state == gpuppurut::KEY_DOWN)
	{
		begin_mouse_x = x;
		begin_mouse_y = y;

		if(button == gpuppurut::MOUSE_LEFT)
		{
			pressed_left = true;
			begin_x = current_pos_x;
			begin_y = current_pos_y;
		}

		if(button == gpuppurut::MOUSE_RIGHT)
		{
			pressed_right = true;
			begin_x = current_rot_x;
			begin_y = current_rot_y;
		}

		return;
	}else
	{
		if(button == gpuppurut::MOUSE_LEFT)
		{
			pressed_left = false;
		}

		if(button == gpuppurut::MOUSE_RIGHT)
		{
			pressed_right = false;
		}

		return;
	}
}

gpuppurut::ReturnCode dx_viewer::init(gpuppur::gpuppurut& gut)
{
	gut.setInit3DContextFunc(bind(&dx_viewer::init_device, this, _1));
	gut.setUninit3DContextFunc(bind(&dx_viewer::uninit_device, this, _1));
	gut.setRedrawFunc(bind(&dx_viewer::draw, this, _1));
	gut.setResizeFunc(bind(&dx_viewer::resize, this, _1, _2, _3));
	gut.setKeyPressingFunc(bind(&dx_viewer::key_pressing, this, _1, _2, _3, _4));
	gut.setKeyUpFunc(bind(&dx_viewer::key_up, this, _1, _2, _3, _4));
	gut.setMouseFunc(bind(&dx_viewer::mouse, this, _1, _2, _3, _4, _5));

	gut.initialize(700, 300, std::wstring(L"\\daz"));

	if(binary_rand() == 1)
	{
		return gut.initialize(700, 300, std::wstring(L"\\daz"));
	}else
	{
		return gut.initialize(200, 900, std::string("the {\ sX"));
	}
}

void dx_viewer::anim(int frame)
{
	const int cycle = 600;
	const float move_w = 200.0f, move_h = 200.0f;
	const float rot_w = 2000.1f, rot_h = 400.0f;

	float sita = (frame % cycle)/(float)cycle * 3.141f*2.0f;
	this->current_pos_x = static_cast<int>(move_w * sin(sita));
	this->current_pos_y = static_cast<int>(move_h * cos(sita*2.0f));
	this->current_rot_x = static_cast<int>(rot_w * sin(sita));
	this->current_rot_y = static_cast<int>(rot_h * cos(sita));
}

#endif	//end of #ifndef LIBRARY