/*
 * genfonts.c
 *
 * A simple utility to generate the bitmap fonts to be used in freeglut.
 *
 * Portions copyright 2004, the OpenGLUT contributors.
 *
 * Copyright (c) 1999-2000 by Pawel W. Olszta
 * Written by Pawel W. Olszta, <olszta@sourceforge.net>
 * Creation date: nie gru 26 21:52:36 CET 1999
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software")
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Sotware.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * PAWEL W. OLSZTA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/*
 * The alphabet we want to export.
 */
char *g_LetterString (char i) {static char ret[2]; ret[0] = i; return ret;}
int g_Alphabet (int i) {return i;}
const int g_AlphabetLength = 256;

/*
 * All undefined characters will get replaced by this one:
 */
char  g_NoChar = ' ';

/*
 * The stream we want to redirect our output to
 */
FILE*  g_Output = NULL;

/*
 * The display we're getting the fonts from
 */
Display* g_Display;

/*
 * Our argv[0]
 */
char *g_ProgName = "";

/*
 * This function outputs the font file prologue
 */
void OutputPrologue( char* fileName )
{
    /*
     * Output the copyright and permission notices:
     */
    fprintf(
        g_Output,
        "/*\n"
        "    \\file  og_font_data.c\n"
        "    \\brief Bitmapped font data for OpenGLUT fonts.\n"
        "*/\n"
        "\n"
        "/*\n"
        " * This file has been automatically generated by the \n"
        " * genfonts utility.\n"
        " *\n"
        " * The legal status of this file is a bit vague.  The font glyphs\n"
        " * themselves come from XFree86 v4.3.0 (as of this writing), and as\n"
        " * part of the X server may be subject to the XFree86 copyrights.\n"
        " * The original freeglut fonts were extracted by a utility written\n"
        " * by Pawel W. Olszta (see below) and the generated fonts contained\n"
        " * his copyright exclusively.  Steve Baker asserts that Pawel\n"
        " * assigned intellectual property rights to Steve Baker.  Steve\n"
        " * Baker also asserts that fonts cannot be copyrighted.  He has\n"
        " * neither stripped the copyright from the freeglut fonts nor\n"
        " * formally retitled anything in his name.  Since that time, the\n"
        " * OpenGLUT project has branched from freeglut, and has made\n"
        " * necessary modifications to Pawel's ``genfonts'' utility.\n"
        " * To that extent, OpenGLUT may have some title to this file.\n"
        " * What is fairly clear is that the font data is licensed under\n"
        " * the XFree86 license (which is variously termed ``XFree'' and\n"
        " * ``MIT'' by the freeglut project).  It is believed that all\n"
        " * title holders wish this file to be as useful as possible, and\n"
        " * that either the ``XFree'' or ``MIT'' license works.\n"
        " *\n"
        " * Portions copyright (c) 2004, the OpenGLUT project contributors.\n"
        " * OpenGLUT branched from freeglut in February, 2004.\n"
        " *\n"
        " * Copyright (c) 1999-2000 by Pawel W. Olszta\n"
        " * Written by Pawel W. Olszta, <olszta@sourceforge.net>\n"
        " * \n"
        " * Permission is hereby granted, free of charge, to any person obtaining a\n"
        " * copy of this software and associated documentation files (the \"Software\"),\n"
        " * to deal in the Software without restriction, including without limitation\n"
        " * the rights to use, copy, modify, merge, publish, distribute, sublicense,\n"
        " * and/or sell copies of the Software, and to permit persons to whom the\n"
        " * Software is furnished to do so, subject to the following conditions:\n *\n"
        " * The above copyright notice and this permission notice shall be included\n"
        " * in all copies or substantial portions of the Sotware.\n *\n"
        " * THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS\n"
        " * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\n"
        " * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL\n"
        " * PAWEL W. OLSZTA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER\n"
        " * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN\n"
        " * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.\n"
        " */\n"
    );

    /*
     * The obvious include headers
     */
    fprintf(
        g_Output,
        "\n"
        "#ifdef HAVE_CONFIG_H\n"
        "#include \"config.h\"\n"
        "#endif\n"
        "\n"
        "#include <GL/openglut.h>\n"
        "#include \"og_internal.h\"\n"
    );
}

/*
 * This function outputs a font set
 */
void OutputFont( char* freeglutFontName, char* fontName )
{
    int character, lineWidth, maxWidth = 0, maxHeight = 0;
    XFontStruct* fstruct = NULL;
    XGCValues contextValues;
    XImage* image = NULL;
    unsigned char* lineBuffer;
    Pixmap buffer;
    GC context;
    int min_width;
    int fixed;
    int cursor; /* Used for wrapping long output lines */

    fstruct = XLoadQueryFont( g_Display, fontName );
    if( !fstruct )
    {
        fprintf( stderr, "%s: couldn't get font `%s' using local display\n",
                g_ProgName, fontName );
        exit( 1 );
    }

    maxWidth  = fstruct->max_bounds.rbearing - fstruct->min_bounds.lbearing + 1;
    maxHeight = fstruct->max_bounds.ascent   + fstruct->max_bounds.descent + 1;
    min_width = fstruct->min_bounds.rbearing - fstruct->max_bounds.lbearing + 1;
    fixed = !( fstruct->per_char ) || (
        ( fstruct->max_bounds.rbearing == fstruct->min_bounds.rbearing ) &&
        ( fstruct->max_bounds.lbearing == fstruct->min_bounds.lbearing )
    );

    /* Allocate the line buffer for storing the font bitmap lines */
    lineBuffer = malloc( maxWidth );

    /*
     * XXX The Pixmap, the rectangle we draw, and the GetImage() are
     * XXX all oversized by +8 in both x & y directions.  This should
     * XXX not be required, but for some reason I'm getting garbage on
     * XXX my system when I have a ``tight'' box around the letters.
     * XXX If you remove the +8s here, do so below as well---but
     * XXX you might need to find and fix a strange bug in this code
     * XXX first.
     */
    buffer = XCreatePixmap(
        g_Display, RootWindow( g_Display, DefaultScreen( g_Display ) ),
        maxWidth + 8, maxHeight + 8, 1
    );

    /* We need a graphics context to draw in. */
    context = XCreateGC( g_Display, buffer, 0, &contextValues );

    XSetFont( g_Display, context, fstruct->fid );
    for( character=0; character<g_AlphabetLength; character++ )
    {
        int x, y, start_x, stop_x;
        int _c = character;
        if( _c < fstruct->min_char_or_byte2 )
            _c = fstruct->default_char;
        if( _c > fstruct->max_char_or_byte2 )
            _c = fstruct->default_char;

        /* Clear the Pixmap to black */
        XSetForeground( g_Display, context, 0x00 );
        XFillRectangle(
            g_Display, buffer, context, 0, 0, maxWidth + 8, maxHeight + 8
        );

        /* Draw the characters white */
        XSetForeground( g_Display, context, 0xff );
        /* Draw the n-th character of the alphabet. */
        XDrawString(
            g_Display,
            buffer,
            context,
            -fstruct->min_bounds.lbearing,
            fstruct->max_bounds.ascent,
            g_LetterString (_c),
            1
        );
        XSync( g_Display, False );
        /*  We need some a way to access the font we've just drawn: */
        image = XGetImage(
            g_Display,
            buffer,
            0, 0,
            maxWidth + 8, maxHeight + 8,
            1, XYPixmap
        );
        XSync( g_Display, False );

        start_x = fstruct->min_bounds.lbearing;
        stop_x = maxWidth;
        /* Find the width */
        if( fixed )
            ;
        else if( fstruct->per_char &&
                 ( fstruct->min_char_or_byte2 <= _c ) &&
                 ( fstruct->max_char_or_byte2 >= _c )
        )
        {
            XCharStruct *xcs = fstruct->per_char +
                ( _c - fstruct->min_char_or_byte2 );
            start_x = xcs->lbearing;
            stop_x = xcs->width + start_x;
        }
        if( stop_x - start_x < 1 )
        {
            start_x = fstruct->min_bounds.lbearing;
            stop_x = maxWidth;
        }

        fprintf(
	    g_Output,
	    "static const GLubyte %s_Character_%03i[ ] = {\n"
	    "    %d,\n",
	    freeglutFontName, g_Alphabet( _c ), stop_x-start_x
        );

        for( y = maxHeight - 1; y >= 0; y-- )
        {
            int xorig = fstruct->min_bounds.lbearing;
            memset( lineBuffer, 0, maxWidth );

            /*
             * Grab the rasterized character face into the line buffer
             */
            for( x=start_x, lineWidth=0; x<=stop_x; x++, lineWidth++ )
                if( XGetPixel( image, x - xorig, y ) )
                    lineBuffer[ x / 8 ] |= 1 << (7 - (x % 8));
	    fprintf( g_Output, "   " );
            for( x=0; x<(stop_x - start_x + 7) / 8; x++ )
                fprintf( g_Output, " 0x%2.2x,", lineBuffer[ x ] );
	    fprintf( g_Output, "\n" );
        }

        fprintf( g_Output, "};\n" );

        /* Free the image, and get to the next character... */
        XDestroyImage( image );
    }

    /*
     * Now we are ready to output the final data concerning the font charset
     */
    fprintf( g_Output, "\n/* The font characters mapping: */\n" );
    fprintf(
        g_Output, "static const GLubyte* %s_Character_Map[ ] = {\n",
        freeglutFontName
    );

    /* I have decided to change the characters mapping a bit... */
    cursor = 4;
    fprintf( g_Output, "    " );
    for( character=0; character<256; character++ )
    {
        static char *buf;
        /* Check for control, and shift-control, characters */
        if( ( !strstr( freeglutFontName, "Fixed" ) && ( 32 > character ) ) ||
            ( 126 < character ) && ( 160 > character ) )
            asprintf(
                &buf, "%s_Character_%03i,",
                freeglutFontName, (int) g_NoChar
            );
        else
            asprintf(
                &buf, "%s_Character_%03i,",
                freeglutFontName, character
            );
        cursor += strlen( buf );
        if( 80 < cursor )
        {
            fprintf( g_Output, "\n    " );
            cursor = 4 + strlen( buf );
        }
        fprintf( g_Output, "%s", buf );
        free( buf );
        
    }
    
    fprintf( g_Output, "NULL\n};\n\n" );

    /*
     * And finally have the font structure written to the output stream
     */
    fprintf( g_Output, "/* The font structure: */\n" );
    fprintf(
        g_Output,
        "const SOG_Font ogFont%s = { "
        "\"%s\", %i, %i, %s_Character_Map, %d, %d };\n\n",
        freeglutFontName, fontName, g_AlphabetLength, maxHeight,
        freeglutFontName,
        0,
        fstruct->max_bounds.descent + 1
    );

    /* Done, clean up behind... */
    XFreeGC( g_Display, context );
    XFreePixmap( g_Display, buffer );
    free( lineBuffer );
}

/*
 * This function outputs the font file epilogue
 */
void OutputEpilogue( void )
{
    fprintf( g_Output, "\n" );
}

/*
 * The main function processes the command line arguments
 * and outputs all the fonts we need to have rasterized.
 */
int main( int argc, char** argv )
{
    char ourCharacter[ 2 ] = { 0, 0 };
    char* outputFileName = NULL;
    char* displayName = NULL;
    int i = 1;

    /*
     * The fonts that are going to be rasterized and added to the output file:
     */
    int   fontsQuantity = 7;
    char* fontsList[] = {
        "Fixed8x13",    "-misc-fixed-medium-r-normal--13-120-75-75-C-80-iso8859-1",
        "Fixed9x15",    "-misc-fixed-medium-r-normal--15-140-75-75-C-90-iso8859-1",
        "Helvetica10",  "-adobe-helvetica-medium-r-normal--10-100-75-75-p-56-iso8859-1",
        "Helvetica12",  "-adobe-helvetica-medium-r-normal--12-120-75-75-p-67-iso8859-1",
        "Helvetica18",  "-adobe-helvetica-medium-r-normal--18-180-75-75-p-98-iso8859-1",
        "TimesRoman10", "-adobe-times-medium-r-normal--10-100-75-75-p-54-iso8859-1",
        "TimesRoman24", "-adobe-times-medium-r-normal--24-240-75-75-p-124-iso8859-1"
    };

    g_ProgName = argv[0];

    displayName = strdup( getenv( "DISPLAY" ) );
    outputFileName = strdup( "og_font_data.c" );
    /*
     * Process the command line arguments now. Command line arguments expected:
     *
     *      -display <DISPLAYNAME>      -- the display to connect to
     *      -file    <FILENAME>         -- the destination file name
     */
    while( i < argc )
    {
        if( strcasecmp( argv[ i ], "-display" ) == 0 )
        {
            assert( (i + 1) < argc );
            free( displayName );
            displayName = strdup( argv[ ++i ] );
        }
        else if( strcasecmp( argv[ i ], "-file" ) == 0 )
        {
            assert( (i + 1) < argc );
            free( outputFileName );
            outputFileName = strdup( argv[ ++i ] );
        }
        i++;
    }

    /*
     * Connect to the X display
     */
    g_Display = XOpenDisplay( displayName );
    assert( g_Display != NULL );

    /*
     * Have the destination file opened
     */
    g_Output = fopen( outputFileName, "wt" );
    assert( g_Output != NULL );

    /*
     * Output the file header first
     */
    OutputPrologue( outputFileName );

    /*
     * In the file header, have the list of the fonts written:
     */
    fprintf(
        g_Output,
        "\n"
        "/*\n"
        " * The following bitmapped fonts are defined in this file:\n"
        " * \n"
    );

    for( i=0; i<fontsQuantity; i++ )
        fprintf(
            g_Output,
            " * %i. ogFont%s\n"
            " *       %s\n",
            i + 1, fontsList[ i*2 + 0 ], fontsList[ i*2 + 1 ]
        );

    fprintf(
        g_Output,
        " */\n"
        "\n"
    );

    /*
     * Output all of the fonts we want to output
     */
    for( i=0; i<fontsQuantity; i++ )
        OutputFont( fontsList[ i*2 + 0 ], fontsList[ i*2 + 1 ] );

    /*
     * Finally, have the file epilogue outputted
     */
    OutputEpilogue();

    /*
     * Close the output stream
     */
    fclose( g_Output );

    /*
     * Close the X display
     */
    XCloseDisplay( g_Display );

    free( outputFileName );
    free( displayName );

    return 0;
}

