/**
 * Classes for reading wave front format 3d model file(*.obj)
*/

#ifndef GPUPPUR_WF_FORMATS_HPP
#define GPUPPUR_WF_FORMATS_HPP

#include <cstdio>
#include <string>
#include <fstream>
#include <iostream>
#include <vector>
#include <string>

#include <gpuppur/3dmath/vectorNd.hpp>

typedef gpuppur::VectorNd<double, 3> Vector3d;
typedef gpuppur::VectorNd<double, 2> texcoord2d;
typedef gpuppur::VectorNd<int, 3> Indices3;

/*------------------obj------------------------*/
struct Face
{
	Indices3 vertIndices;
	Indices3 coordIndices;
	Indices3 normalIndices;

	Face(int  v0, int  v1, int  v2,
	     int vt0, int vt1, int vt2,
	     int vn0, int vn1, int vn2)
	: vertIndices(v0-1, v1-1, v2-1),
	  coordIndices(vt0-1, vt1-1, vt2-1),
	  normalIndices(vn0-1, vn1-1, vn2-1)
	{
	}
};

#if 0
struct Material
{
	char mtlName[255];
	char texFile[255];
	unsigned char color[3];
	int texureId;
	double uTile;		/* u tiling of texture  (Currently not used)*/
	double vTile;		/* v tiling of texture	(Currently not used)*/
	double uOffset;		/* u offset of texture	(Currently not used)*/
	double vOffset;		/* v offset of texture	(Currently not used)*/
	Vector4d Diffuse;
	Vector4d Ambient;
	Vector4d Specular;
};
#endif

struct Object
{
	typedef std::vector<Vector3d> Verts;
	typedef std::vector<texcoord2d> TexVerts;
	typedef std::vector<Vector3d> Normals;
	typedef std::vector<Face> Faces;

	int materialID;
	int vertPerFace;
	char		trName[255];
	Verts		verts;
	TexVerts	texVerts;
	Normals		normals;
	Faces		faces;
};

struct Model
{
	typedef std::vector<Object> Objects;
//	typedef std::vector<Material> Materials;

//	Materials materials;
	Objects objects;

	bool load(const std::string& filename)
	{
		std::fstream in(filename.c_str(), std::fstream::in);

		if(!in.good())
		{
			return false;
		}

		char line[1024];

		this->objects.resize(1);
		Object& object = this->objects[0];
		while(!in.eof())
		{
			in.getline(line, 1024);
			if(strlen(line) < 2)
				continue;
			if(line[0] == '#')
			{
				continue;
			}

			switch(line[0])
			{
			case 'v':
				double x, y, z;

				switch(line[1])
				{
				case 'n':
				case 't':
					break;
				default:
					sscanf(line, "v %lf %lf %lf", &x, &y, &z);
					object.verts.push_back(Vector3d(x, y, z));
					break;
				}
				break;
			case 'f':
				int i0, i1, i2, i3;
				int it0, it1, it2, it3;
				int in0, in1, in2, in3;

				object.vertPerFace = 4;
				if(sscanf(line, "f %d/%d/%d", &i0, &it0, &in0) == 3)
				{
					if(sscanf(line, "f %d/%d/%d %d/%d/%d "
								"%d/%d/%d %d/%d/%d",
							   &i0, &it0, &in0,
							   &i1, &it1, &in1,
							   &i2, &it2, &in2,
							   &i3, &it3, &in3) != 12)
					{
						if(sscanf(line,
							"f %d/%d/%d %d/%d/%d %d/%d/%d",
							&i0, &it0, &in0,
								&i1, &it1, &in1,
								&i2, &it2, &in2) != 9)
						{
							break;
						}

						object.vertPerFace = 3;
					}
				}else if(sscanf(line, "f %d/%d", &i0, &it0) == 2)
				{
					if(sscanf(line, "f %d/%d %d/%d %d/%d %d/%d",
							   &i0, &it0,
							   &i1, &it1,
							   &i2, &it2,
						   &i3, &it3) != 8)
					{
						if(sscanf(line,
							"f %d/%d %d/%d %d/%d",
							&i0, &it0,
								&i1, &it1,
								&i2, &it2) != 6)
						{
							break;
						}

						object.vertPerFace = 3;
					}
				}else
				{
					if(sscanf(line, "f %d %d %d %d",
							&i0, &i1, &i2, &i3) != 4)
					{
						if(sscanf(line, "f %d %d %d",
							&i0, &i1, &i2) != 3)
						{
							break;
						}

						object.vertPerFace = 3;
					}
				}

				if(object.vertPerFace == 4)
				{
					object.faces.push_back(Face(
								i0,  i1,  i2,
								it0, it1, it2,
								in0, in1, in2));
					object.faces.push_back(Face(
								i0,  i2,  i3,
								it0, it2, it3,
								in0, in2, in3));
				}else
				{
					object.faces.push_back(Face(
								i0,  i1,  i2,
								it0, it1, it2,
								in0, in1, in2));
				}

				break;
			}
		}

		return true;
	}
};

#endif

