#ifndef MATH3D_AABB_HPP
#define MATH3D_AABB_HPP

/**
 *	@file	
 *	@brief	Axis aligned bounding box class
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <limits>
#include <iterator>
#include <boost/tuple/tuple.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

namespace gpuppur
{

template<class Vector>
class aabb
{
public:

	typedef Vector						vector_type;
	typedef typename Vector::value_type	value_type;

	aabb()
	{
	}

	aabb(const aabb& src):
		min(src.min), max(src.max)
	{
	}

	aabb(const Vector min, const Vector& max):
		min(min), max(max)
	{
	}

	Vector get_min()
	{
		return this->min;
	}

	const Vector get_min() const
	{
		return this->min;
	}

	Vector get_max()
	{
		return this->max;
	}

	const Vector get_max() const
	{
		return this->max;
	}

private:

	Vector	min;
	Vector	max;
};

template<class InputIterator>
aabb<typename std::iterator_traits<InputIterator>::value_type>
get_axis_aligned_bounding_box
(
	InputIterator first,
	InputIterator end
)
{
	typedef typename std::iterator_traits<InputIterator>::value_type vector;

	InputIterator i = first;
	vector	min=*i, max=*i;

	++i;

	for(; i!=end; ++i)
	{
		const vector elem = *i;

		for(int j=0; j<vector::dimention; ++j)
		{
			min[j] = min[j] <= elem[j] ? min[j] : elem[j];
			max[j] = max[j] >= elem[j] ? max[j] : elem[j];
		}
	}

	return aabb<vector>(min, max);
}

template<typename Vector>
bool is_in
(
	const aabb<Vector>&	box,
	const Vector&		point
)
{
	for(std::size_t i=0; i<Vector::dimention; ++i)
	{
		if(point[i] <= box.get_min()[i] || point[i] >= box.get_max()[i])
		{
			return false;
		}
	}

	return true;
}

/**
 *	@brief Cast ray to axis aligned bounding box
 *
 *	plane index -1 == no plane.(specified ray don't touch aabb at all.)
 *	plane index 0, 1, ..., dimention-1 == 0(most of people call x) axis min plane, 1(most of people call y) axis min plane, ..., dimention-1 axis min plane.
 *	plane index dimention, dimention+1, ..., dimention+dimention-1 == 0 axis max plane, 1 axis max plane, ..., dimention-1 axis max plane.
 *
 *	@param box	The axis aligned bounding box.
 *	@param ray	The ray you want to cast.
 *	@param t_min	min value of t of specified ray.
 *	@param t_max	max value of t of specified ray.
 */
template<typename Line>
boost::tuple
<
	typename Line::vector_type::value_type,	//<	t min
	typename Line::vector_type::value_type,	//< t max
	int,									//< in plane index
	int										//< out plane index
>
cast_ray_to_aabb
(
	const aabb<typename Line::Vector>&	box,
	const Line							ray,
	typename Line::vector_type::value_type_param	t_min
	=
		// This code means zero XD
			std::numeric_limits<typename Line::vector_type::value_type>::max()
		-
			std::numeric_limits<typename Line::vector_type::value_type>::max()
	,
	typename Line::vector_type::value_type_param	t_max
	=
	std::numeric_limits<typename Line::vector_type::value_type>::max()
)
{
	typedef typename Line::vector_type::value_type scaler;
	typedef boost::tuple
	<
		scaler,		//<	t min
		scaler,		//< t max
		int,		//< in plane index
		int			//< out plane index
	>	return_type;
	const static std::size_t	dimention = Line::Vector::dimention;

	int		in_plane_index = -1, out_plane_index = -1;

	scaler tmn = t_min, tmx = t_max;

	for(int i=0; i<static_cast<int>(dimention); ++i)
	{
		if(ray.getDir()[i] == 0)
		{
			continue;
		}

		scaler t0 = (box.get_min()[i] - ray.getPos()[i])/ray.getDir()[i];
		scaler t1 = (box.get_max()[i] - ray.getPos()[i])/ray.getDir()[i];

		if(t0 < t1)
		{
			if(t1 < tmn || t0 > tmx)
			{
				//no cross
				return return_type(0, 0, -1, -1);
			}

			if(t0 > tmn)
			{
				in_plane_index = i;
				tmn = t0;
			}

			if(t1 < tmx)
			{
				out_plane_index = i+dimention;
				tmx = t1;
			}

		}else
		{
			if(t0 < tmn || t1 > tmx)
			{
				//no cross
				return return_type(0, 0, -1, -1);
			}

			if(t1 > tmn)
			{
				in_plane_index = i+dimention;
				tmn = t1;
			}

			if(t0 < tmx)
			{
				out_plane_index = i;
				tmx = t0;
			}

		}
	}

	//This ray cross this aabb.
	return return_type(tmn, tmx, in_plane_index, out_plane_index);
}

namespace aabb_implement
{
struct division
{
	template<typename Vector, typename OutputIterator, int dimention>
	static void div
	(
		const Vector&	pos,
		const Vector&	size,
		const Vector&	num_divideds,
		OutputIterator&	out,
		boost::mpl::int_<dimention>,
		typename boost::enable_if_c<Vector::dimention!=dimention>::type* =0
	)
	{
		Vector current_pos(pos);
		for(std::size_t i=0; i<num_divideds[dimention]; ++i)
		{
			div
			(
				current_pos, size, num_divideds,
				out, boost::mpl::int_<dimention+1>()
			);

			current_pos[dimention] += size[dimention];
		}
	}

	template<typename Vector, typename OutputIterator, int dimention>
	static void div
	(
		const Vector&	pos,
		const Vector&	size,
		const Vector&	/*num_divideds*/,	//Why doesn't Visual C++ 2005 warn unused argument parameter?
		OutputIterator&	out,
		boost::mpl::int_<dimention>,
		typename boost::disable_if_c<Vector::dimention!=dimention>::type* =0
	)
	{
		*out = aabb<Vector>(pos, pos+size);
		out++;
	}
} Hey__why_cant_I_define_this_class_in__get_uniform_divided_aabbs______________________________________________________________________________________________________________________________________________________Fu_king_V_s_al_Cpp;;
}

template<typename Vector, typename OutputIterator>
void get_uniform_divided_aabbs
(
	const aabb<Vector>&	src_aabb,
	const Vector&		num_divideds,
	OutputIterator		first
)
{
	const Vector size=(src_aabb.get_max()-src_aabb.get_min())/num_divideds;

	OutputIterator	itr = first;
	aabb_implement::division::div
	(
		src_aabb.get_min(),
		size,
		num_divideds,
		itr,
		boost::mpl::int_<0>()
	);
}

}	// end of namespace gpuppur

#endif

