#ifndef MATH3D_MATRIXRXC_HPP
#define MATH3D_MATRIXRXC_HPP

/**
 *	@file	
 *	@brief	N rows M column Matrix template class and functions for it.
 *	@author	Tomohiro Matsumoto
 */

#include <cassert>
#include <iostream>
#include <boost/array.hpp>
#include <boost/call_traits.hpp>
#include <boost/preprocessor.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/static_assert.hpp>
#include "vectorNd.hpp"

namespace gpuppur
{

// Macros for generate constructor code of matrixRxC
// In Visual C++ 2005, i cant write comment in macro.
// In g++ 3.4.4, i can write comment in macro.
#define MAX_CONSTRUCTOR_ARG	5
#define MAX_CONSTRUCTOR_ARG_2 BOOST_PP_MUL(MAX_CONSTRUCTOR_ARG, MAX_CONSTRUCTOR_ARG)

#define GET_ASSIGN_GEN(assign_gen, constructor_name)			\
		assign_gen
#define GET_CONSTRUCTOR_NAME(assign_gen, constructor_name)		\
		constructor_name

/**	This macro generates constructor which construct object from scalers.
 *
 *	constructor like:
 *	constructor(T arg0, T arg1, T arg2, ...)
 *	{
 *		this->elems[0]=arg0;
 *		this->elems[1]=arg1;
 *		...
 *	}
 *
 */
#define ASSIGN_GEN(z, n, data)									\
	this->elems[n]=arg##n;

#define ASSIGN_num_column_MAJOR_GEN(z, n, data)					\
	this->elems[n%num_column*num_row+n/num_column]=arg##n;

#define CONSTRUCTOR_GEN(z, n, context)							\
	explicit													\
	GET_CONSTRUCTOR_NAME context								\
	(BOOST_PP_ENUM_PARAMS(n, T arg))							\
	{															\
		BOOST_STATIC_ASSERT(num_row*num_column>=n);				\
		BOOST_PP_REPEAT											\
		(														\
			n,													\
			GET_ASSIGN_GEN context,								\
			_													\
		)														\
	}

#define CONSTRUCTOR(constructor_name)							\
		BOOST_PP_REPEAT_FROM_TO									\
		(														\
			1,													\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG_2),				\
			CONSTRUCTOR_GEN,									\
			(ASSIGN_GEN, constructor_name)						\
		)

#define CONSTRUCTOR_COLUMN_MAJOR(constructor_name)				\
		BOOST_PP_REPEAT_FROM_TO									\
		(														\
			1,													\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG_2),				\
			CONSTRUCTOR_GEN,									\
			(ASSIGN_num_column_MAJOR_GEN, constructor_name)		\
		)


/**
 * construct from 2 smaller matrices.
 * For example, construct 4x4 matrix from 4x3 matrix and 4x1 matrix.
 * constructor like:
 *
 * constructor(matrixRxC<T, Row, n> mat1, matrixRxC<T, Row, Column-n> mat2)
 * {
 *		std::size_t j=0, k=0;
 *		for(std::size_t i=0; i<num_row*num_column; ++i)
 *		{
 *			this->elems[i] = (i%num_column<n) ? mat1[j++] : mat2[k++];
 *		}
 *	}
 *
 */
#define CONSTRUCTOR_FROM_2_SMALLER_MATRICES_VSPLIT_GEN(z, n, context)	\
		BOOST_PP_SEQ_ELEM(0, context)							\
		(														\
			matrixRxC											\
			<													\
				T,												\
				Row,											\
				n,												\
				BOOST_PP_SEQ_ELEM(1, context)					\
			>		mat_left,									\
			matrixRxC											\
			<													\
				T,												\
				Row,											\
				Column-n,										\
				BOOST_PP_SEQ_ELEM(1, context)					\
			>	mat_right										\
		)														\
		{														\
			std::size_t j=0, k=0;								\
			for(std::size_t i=0; i<num_row*num_column; ++i)		\
			{													\
				this->elems[i]									\
				=												\
				(												\
					BOOST_PP_IF									\
					(											\
						BOOST_PP_SEQ_ELEM(1, context),			\
						(i%num_column<n),						\
						(i<n*num_row)							\
					)											\
					?											\
					mat_left[j++]								\
					:											\
					mat_right[k++]								\
				);												\
			}													\
		}

#define CONSTRUCTOR_FROM_2_SMALLER_MATRICES_HSPLIT_GEN(z, n, context)	\
		BOOST_PP_SEQ_ELEM(0, context)							\
		(														\
			matrixRxC											\
			<													\
				T,												\
				n,												\
				Column,											\
				BOOST_PP_SEQ_ELEM(1, context)					\
			>	mat_top,										\
			matrixRxC											\
			<													\
				T,												\
				Row-n,											\
				Column,											\
				BOOST_PP_SEQ_ELEM(1, context)					\
			>	mat_bottom										\
		)														\
		{														\
			BOOST_STATIC_ASSERT(n<Row);							\
			std::size_t j=0, k=0;								\
			for(std::size_t i=0; i<num_row*num_column; ++i)		\
			{													\
				this->elems[i]									\
				=												\
				(												\
					BOOST_PP_IF									\
					(											\
						BOOST_PP_SEQ_ELEM(1, context),			\
						(i<n*num_column),						\
						(i%num_row<n)							\
					)											\
					?											\
					mat_top[j++]								\
					:											\
					mat_bottom[k++]								\
				);												\
			}													\
		}

#define CONSTRUCTOR_FROM_2_SMALLER_MATRICES(constructor_name, is_row_major, is_vsplit)	\
		BOOST_PP_REPEAT_FROM_TO									\
		(														\
			1,													\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG),					\
			BOOST_PP_IF											\
			(													\
				is_vsplit,										\
				CONSTRUCTOR_FROM_2_SMALLER_MATRICES_VSPLIT_GEN,	\
				CONSTRUCTOR_FROM_2_SMALLER_MATRICES_HSPLIT_GEN	\
			),													\
			(constructor_name)(is_row_major)(is_vsplit)			\
		)


/**
 * construct matrixRxC from major vectors(if row major matrix, major vector is row_type).
 * For example, construct 3x4 matrix from 3 vector4d.
 * constructor like:
 *
 * constructor(const row_type& vec0, const row_type& vec1, ...)
 * {
 *		for(std::size_t i=0; i<num_column; ++i)
 *		{
 *			this->elems[0*num_column+i] = vec0[i];
 *		}
 *
 *		for(std::size_t i=0; i<num_column; ++i)
 *		{
 *			this->elems[1*num_column+i] = vec1[i];
 *		}
 *
 *		...
 *	}
 *
 */
#define ASSIGN_VECTOR_GEN(z, n, context)						\
		for														\
		(														\
			std::size_t i=0;									\
			i<BOOST_PP_IF										\
			(													\
				BOOST_PP_SEQ_ELEM(1, context),					\
				num_column,										\
				num_row											\
			);													\
			++i													\
		)														\
		{														\
			this->elems											\
			[													\
				n												\
				*												\
				BOOST_PP_IF										\
				(												\
					BOOST_PP_SEQ_ELEM(1, context),				\
					num_column,									\
					num_row										\
				)												\
				+i												\
			] = vec##n[i];										\
		}

#define ASSIGN_NON_MAJOR_VECTOR_GEN(z, n, context)				\
		for														\
		(														\
			std::size_t i=0;									\
			i<BOOST_PP_IF										\
			(													\
				BOOST_PP_SEQ_ELEM(1, context),					\
				num_row,										\
				num_column										\
			);													\
			++i													\
		)														\
		{														\
			this->elems											\
			[													\
				i												\
				*												\
				BOOST_PP_IF										\
				(												\
					BOOST_PP_SEQ_ELEM(1, context),				\
					num_column,									\
					num_row										\
				)												\
				+n												\
			] = vec##n[i];										\
		}

#define CONSTRUCTOR_FROM_VECTORS_TO_NON_SQUARE_MATRIX_GEN(z, n, context)	\
	template<class Vector>										\
	explicit BOOST_PP_SEQ_ELEM(0, context)						\
	(															\
		BOOST_PP_ENUM_PARAMS									\
		(														\
			n,													\
			const Vector& vec									\
		),														\
		typename												\
		boost::enable_if_c										\
		<														\
			num_column!=num_row									\
			&&													\
			boost::is_same										\
			<													\
				Vector,											\
				BOOST_PP_IF										\
				(												\
					BOOST_PP_XOR								\
					(											\
						BOOST_PP_SEQ_ELEM(1, context),			\
						BOOST_PP_SEQ_ELEM(2, context)			\
					),											\
					column_type,								\
					row_type									\
				)												\
			>::value											\
		>::type* = 0,											\
		BOOST_PP_IF												\
		(														\
			BOOST_PP_SEQ_ELEM(2, context),						\
			int,												\
			double												\
		)=0														\
	)															\
	{															\
		BOOST_STATIC_ASSERT										\
		(														\
			n<=BOOST_PP_IF										\
			(													\
				BOOST_PP_XOR									\
				(												\
					BOOST_PP_SEQ_ELEM(1, context),				\
					BOOST_PP_SEQ_ELEM(2, context)				\
				),												\
				num_column,										\
				num_row											\
			)													\
		);														\
																\
		BOOST_PP_REPEAT											\
		(														\
			n,													\
			BOOST_PP_IF											\
			(													\
				BOOST_PP_SEQ_ELEM(2, context),					\
				ASSIGN_VECTOR_GEN,								\
				ASSIGN_NON_MAJOR_VECTOR_GEN						\
			),													\
			context												\
		)														\
	}

#define CONSTRUCTOR_FROM_VECTORS_TO_SQUARE_MATRIX_GEN(z, n, context)	\
	template<class Vector>										\
	explicit BOOST_PP_SEQ_ELEM(0, context)						\
	(															\
		BOOST_PP_ENUM_PARAMS									\
		(														\
			n,													\
			const Vector& vec									\
		),														\
		bool is_row_vector = true,								\
		typename												\
		boost::enable_if_c										\
		<														\
			num_column==num_row									\
			&&													\
			boost::is_same<Vector, row_type>::value				\
		>::type* = 0											\
	)															\
	{															\
		BOOST_STATIC_ASSERT										\
		(														\
			n<=num_column										\
		);														\
																\
		bool is_row_major =										\
		BOOST_PP_IF												\
		(														\
			BOOST_PP_SEQ_ELEM(1, context),						\
			true,												\
			false												\
		);														\
		if														\
		(														\
			(is_row_major && is_row_vector)						\
			||													\
			(!is_row_major && !is_row_vector)					\
		)														\
		{														\
			BOOST_PP_REPEAT										\
			(													\
				n,												\
				ASSIGN_VECTOR_GEN,								\
				context											\
			)													\
		}else													\
		{														\
			BOOST_PP_REPEAT										\
			(													\
				n,												\
				ASSIGN_NON_MAJOR_VECTOR_GEN,					\
				context											\
			)													\
		}														\
	}

#define CONSTRUCTOR_FROM_VECTORS(constructor_name, is_row_major)\
		BOOST_PP_REPEAT_FROM_TO									\
		(														\
			1,													\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG),					\
			CONSTRUCTOR_FROM_VECTORS_TO_NON_SQUARE_MATRIX_GEN,	\
			(constructor_name)(is_row_major)(1)					\
		)														\
		BOOST_PP_REPEAT_FROM_TO									\
		(														\
			1,													\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG),					\
			CONSTRUCTOR_FROM_VECTORS_TO_NON_SQUARE_MATRIX_GEN,	\
			(constructor_name)(is_row_major)(0)					\
		)														\
		BOOST_PP_REPEAT_FROM_TO									\
		(														\
			1,													\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG),					\
			CONSTRUCTOR_FROM_VECTORS_TO_SQUARE_MATRIX_GEN,		\
			(constructor_name)(is_row_major)(_)					\
		)

#if 0
#define CONSTRUCTOR_FROM_NON_MAJOR_TYPE_VECTORS(constructor_name, is_row_major)	\
		BOOST_PP_REPEAT_FROM_TO								\
		(													\
			1,												\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG),				\
			CONSTRUCTOR_FROM_MAJOR_TYPE_VECTORS_GEN,		\
			(ASSIGN_NON_MAJOR_VECTOR_GEN)(constructor_name)(is_row_major)	\
		)
#endif

/**
 *	@brief	Basic matrix class
 *
 */
template<typename T, std::size_t Row, std::size_t Column>
class matrixRxC_base
{
public:

	typedef T										value_type;
	typedef matrixRxC_base<T, Row, Column>			this_type;
	const static std::size_t						num_row = Row;
	const static std::size_t						num_column = Column;
	typedef VectorNd<T, this_type::num_column>		row_type;
	typedef VectorNd<T, this_type::num_row>			column_type;

	T& operator[](std::size_t i)
	{
		assert(i<num_row*num_column);

		return this->elems[i];
	}

	const T& operator[](std::size_t i) const
	{
		assert(i<num_row*num_column);

		return this->elems[i];
	}

	const T* raw_data() const
	{
		return this->elems.data();
	}

protected:
	typedef	boost::array<T, num_row*num_column>	element_type;
	element_type elems;
};


/**
 *	@brief	Row major matrix class.
 */
template<typename T, std::size_t Row, std::size_t Column, bool IsRowMajor=true>
class matrixRxC : public matrixRxC_base<T, Row, Column>
{
public:

	typedef matrixRxC_base<T, Row, Column>		base;
	typedef matrixRxC<T, Row, Column, true>		this_type;
	const static std::size_t					num_row = base::num_row;
	const static std::size_t					num_column = base::num_column;
	typedef typename base::row_type				row_type;
	typedef typename base::column_type			column_type;
	typedef typename base::value_type			value_type;
	typedef matrixRxC<T, Row, Column, false>	column_major_type;

	matrixRxC()
	{
	}

	matrixRxC(const column_major_type& rhs)
	{
		for(std::size_t i=0; i<num_row*num_column; ++i)
		{
			this->elems[i] = rhs.elems[i/num_column+(i%num_column)*num_row];
		}
	}

	CONSTRUCTOR(matrixRxC)
	CONSTRUCTOR_FROM_2_SMALLER_MATRICES(matrixRxC, 1, 1)
	CONSTRUCTOR_FROM_2_SMALLER_MATRICES(matrixRxC, 1, 0)
	CONSTRUCTOR_FROM_VECTORS(matrixRxC, 1)

	/**	Return specified column as vector.
	 *
	 *	@param[in]	n	specify nth column which is returned as vector. If n==0, left column will be returned. range [0,number of column)
	 *	@return	Vector of nth column.
	 */
	const column_type column(std::size_t n) const
	{
		assert(n<num_column);

		column_type ret;

		for(std::size_t i=0; i<num_row; ++i)
		{
			ret[i] = this->elems[i*num_column+n];
		}

		return ret;
	}

	/**	Return reference to specified row as vector.
	 *
	 *	@param[in]	n	specify nth row which is returned as vector. If n==0, top row will be returned. range [0,number of row)
	 *	@return	Vector of nth row.
	 */
	row_type& row(std::size_t n)
	{
		assert(n<num_row);

		return *reinterpret_cast<row_type*>(&this->elems[n*num_column]);
	}

	const row_type& row(std::size_t n) const
	{
		assert(n<num_row);

		return *reinterpret_cast<const row_type*>(&this->elems[n*num_column]);
	}

	this_type& load_unit_matrix()
	{
		for(std::size_t i=0; i<num_row*num_column; ++i)
		{
			this->elems[i]
			=
			(
				i%(num_column+1) == 0
				&&
				(
					num_row < num_column
					?
					i%num_column < num_row
					:
					i/num_column < num_column
				)
			)
			?
			static_cast<T>(1)
			:
			0
			;
		}

		return *this;
	}

	const value_type& operator()(std::size_t row, std::size_t column) const
	{
		assert(row < num_row);
		assert(column < num_column);

		return (*this)[row*num_column+column];
	}

	value_type& operator()(std::size_t row, std::size_t column)
	{
		assert(row < num_row);
		assert(column < num_column);

		return (*this)[row*num_column+column];
	}
/*
	const column_type operator * (const row_type& rhs) const
	{
		column_type ret;

		for(std::size_t i=0; i<num_row; ++i)
		{
			ret[i] = this->row(i).innerProduct(rhs);
		}

		return ret;
	}
*/
	friend	class	matrixRxC<T, Row, Column, false>;
};

/**
 *	Column major matrix class.
 */
template<typename T, std::size_t Row, std::size_t Column>
class matrixRxC<T, Row, Column, false> : public matrixRxC_base<T, Row, Column>
{
public:

	typedef matrixRxC_base<T, Row, Column>		base;
	typedef matrixRxC<T, Row, Column, false>	this_type;
	const static std::size_t					num_row = base::num_row;
	const static std::size_t					num_column = base::num_column;
	typedef typename base::row_type				row_type;
	typedef typename base::column_type			column_type;
	typedef typename base::value_type			value_type;
	typedef matrixRxC<T, Row, Column, true>		row_major_type;
	friend	class	matrixRxC<T, Row, Column, true>;

	matrixRxC()
	{
	}

	matrixRxC(const row_major_type& rhs)
	{
		for(std::size_t i=0; i<num_row*num_column; ++i)
		{
			this->elems[i] = rhs.elems[i/num_row+(i%num_row)*num_column];
		}
	}

	CONSTRUCTOR_COLUMN_MAJOR(matrixRxC)
	CONSTRUCTOR_FROM_2_SMALLER_MATRICES(matrixRxC, 0, 1)
	CONSTRUCTOR_FROM_2_SMALLER_MATRICES(matrixRxC, 0, 0)
	CONSTRUCTOR_FROM_VECTORS(matrixRxC, 0)

	column_type& column(std::size_t n)
	{
		assert(n<num_column);

		return *reinterpret_cast<column_type*>(&this->elems[n*num_row]);
	}

	const column_type& column(std::size_t n) const
	{
		assert(n<num_column);

		return *reinterpret_cast<const column_type*>(&this->elems[n*num_row]);
	}

	const row_type row(std::size_t n) const
	{
		assert(n<num_row);

		row_type ret;

		for(std::size_t i=0; i<num_column; ++i)
		{
			ret[i] = this->elems[i*num_row+n];
		}

		return ret;
	}

	this_type& load_unit_matrix()
	{
		for(std::size_t i=0; i<num_row*num_column; ++i)
		{
			this->elems[i]
			=
			(
				i%(num_row+1) == 0
				&&
				(
					num_row < num_column
					?
					i/num_row < num_row
					:
					i%num_row < num_column
				)
			)
			?
			static_cast<T>(1)
			:
			0
			;
		}

		return *this;
	}

	value_type& operator()(std::size_t row, std::size_t column)
	{
		assert(row < num_row);
		assert(column < num_column);

		return (*this)[column*num_row+row];
	}

	const value_type& operator()(std::size_t row, std::size_t column) const
	{
		assert(row < num_row);
		assert(column < num_column);

		return (*this)[column*num_row+row];
	}

/*
	const column_type operator * (const row_type& rhs) const
	{
		column_type ret;

		for(std::size_t i=0; i<num_row; ++i)
		{
			ret[i] = this->row(i).innerProduct(rhs);
		}

		return ret;
	}
*/
};

template<typename T, std::size_t Row, std::size_t Column, bool IsRowMajor>
std::ostream& operator<<
(
	std::ostream&									os,
	const matrixRxC<T, Row, Column, IsRowMajor>&	mat
)
{
	os << "----------------------------\n";
	for(std::size_t i=0; i<Row; ++i)
	{
		for(std::size_t j=0; j<Column; ++j)
		{
			os << mat(i, j) << "\t" ;
		}
		os << "\n";
	}

	return os;
}

template
<
	typename T,
	std::size_t CommonSize,
	std::size_t RowLHS,
	std::size_t ColumnRHS,
	bool IsRowMajorLHS,
	bool IsRowMajorRHS
>
const matrixRxC<T, RowLHS, ColumnRHS, IsRowMajorLHS> operator *
(
	const matrixRxC<T, RowLHS, CommonSize, IsRowMajorLHS>& lhs,
	const matrixRxC<T, CommonSize, ColumnRHS, IsRowMajorRHS>& rhs
)
{
	matrixRxC<T, RowLHS, ColumnRHS, IsRowMajorLHS>	ret;

	for(std::size_t i=0; i<RowLHS; ++i)
	for(std::size_t j=0; j<ColumnRHS; ++j)
	{
		ret(i, j) = lhs.row(i).innerProduct(rhs.column(j));
	}

	return ret;
}

template
<
	typename T,
	std::size_t Row,
	std::size_t Column,
	bool IsRowMajorLHS,
	bool IsRowMajorRHS
>
bool operator ==
(
	const matrixRxC<T, Row, Column, IsRowMajorLHS>& lhs,
	const matrixRxC<T, Row, Column, IsRowMajorRHS>& rhs
)
{
	for(std::size_t i=0; i<Row; ++i)
	for(std::size_t j=0; j<Column; ++j)
	{
		if(lhs(i, j) != rhs(i, j))
		{
			return false;
		}
	}

	return true;
}

template
<
	std::size_t BeginRow,
	std::size_t BeginColumn,
	std::size_t EndRow,
	std::size_t EndColumn,
	bool IsRowMajorRet,
	typename T,
	std::size_t Row,
	std::size_t Column,
	bool IsRowMajorArg
>
const matrixRxC<T, EndRow-BeginRow, EndColumn-BeginColumn, IsRowMajorRet>
get_sub_major(const matrixRxC<T, Row, Column, IsRowMajorArg>& src)
{
	matrixRxC<T, EndRow-BeginRow, EndColumn-BeginColumn, IsRowMajorRet>
	ret;

	for(std::size_t i=BeginRow; i<EndRow; ++i)
	for(std::size_t j=BeginColumn; j<EndColumn; ++j)
	{
		ret(i-BeginRow, j-BeginColumn) = src(i, j);
	}

	return ret;
}

template
<
	std::size_t BeginRow,
	std::size_t BeginColumn,
	std::size_t EndRow,
	std::size_t EndColumn,
	typename	T,
	std::size_t Row,
	std::size_t Column,
	bool		IsRowMajor
>
const matrixRxC<T, EndRow-BeginRow, EndColumn-BeginColumn, IsRowMajor>
get_sub(const matrixRxC<T, Row, Column, IsRowMajor>& src)
{
	return get_sub_major<BeginRow, BeginColumn, EndRow, EndColumn, IsRowMajor>(src);
}

template
<
	typename	TRHS,
	std::size_t	RowRHS,
	std::size_t	ColumnRHS,
	bool		IsRowMajorRHS,
	typename	TLHS,
	std::size_t RowLHS,
	std::size_t ColumnLHS,
	bool		IsRowMajorLHS
>
void set_sub
(
	matrixRxC<TRHS, RowRHS, ColumnRHS, IsRowMajorRHS>& dst,
	const
	matrixRxC<TLHS, RowLHS, ColumnLHS, IsRowMajorLHS>& src,
	const std::size_t	dst_row,
	const std::size_t	dst_column,
	const std::size_t	src_row=0,
	const std::size_t	src_column=0,
	const std::size_t	height=RowLHS,
	const std::size_t	width=ColumnLHS
)
{
	assert(dst_row < RowRHS);
	assert(dst_column < ColumnRHS);
	assert(src_row < RowLHS);
	assert(src_column < ColumnLHS);
	assert(height <= RowRHS-dst_row);
	assert(width <= ColumnRHS-dst_column);
	assert(height <= RowLHS-src_row);
	assert(width <= ColumnLHS-src_column);

	for
	(
		std::size_t	i=dst_row, l=src_row;
		i<dst_row+height;
		++i, ++l
	)
	for
	(
		std::size_t j=dst_column, m=src_column;
		j<dst_column+width;
		++j, ++m
	)
	{
		dst(i, j) = src(l, m);
	}
}

template
<
	typename	T,
	std::size_t	Row,
	std::size_t	Column,
	bool		IsRowMajor
>
const matrixRxC<T, Row-1, Column-1, IsRowMajor>
get_cofactor
(
	const matrixRxC<T, Row, Column, IsRowMajor>& mat,
	std::size_t i, std::size_t j
)
{
	matrixRxC<T, Row-1, Column-1, IsRowMajor> ret;

	if(i>0)
	{
		if(j>0)
		{
			set_sub(ret, mat, 0, 0, 0, 0, i, j);
		}

		if(j<Column-1)
		{
			set_sub(ret, mat, 0, j, 0, j+1, i, Column-j-1);
		}
	}

	if(i<Row-1)
	{
		if(j>0)
		{
			set_sub(ret, mat, i, 0, i+1, 0, Row-i-1, j);
		}

		if(j<Column-1)
		{
			set_sub(ret, mat, i, j, i+1, j+1, Row-i-1, Column-j-1);
		}
	}

	return ret;
}

template
<
	typename	T,
	std::size_t	Row,
	std::size_t	Column,
	bool		IsRowMajor
>
const T get_determinant(const matrixRxC<T, Row, Column, IsRowMajor>& mat)
{
	T ret = static_cast<T>(0.0);

	for(std::size_t i=0; i<Row; i++)
	{
		ret += mat(i, 0)*get_determinant(get_cofactor(mat, i, 0))
			*
			static_cast<T>(i%2==0 ? 1.0 : -1.0);
	}

	return ret;
}

template
<
	typename T,
	bool IsRowMajor
>
const T get_determinant(const matrixRxC<T, 1, 1, IsRowMajor>& mat)
{
	return mat(0, 0);
}

template
<
	typename	T,
	std::size_t	Row,
	std::size_t Column,
	bool		IsRowMajor
>
const matrixRxC<T, Column, Row, IsRowMajor>
get_transposed
(
	const matrixRxC<T, Row, Column, IsRowMajor>& mat
)
{
	matrixRxC<T, Column, Row, IsRowMajor> ret;

	for(std::size_t i=0; i<Row; ++i)
	for(std::size_t j=0; j<Column; ++j)
	{
		ret(j, i) = mat(i, j);
	}

	return ret;
}

template
<
	typename	T,
	std::size_t	Size,
	bool		IsRowMajor
>
const matrixRxC<T, Size, Size, IsRowMajor>
get_inversed
(
	const matrixRxC<T, Size, Size, IsRowMajor>& mat
)
{
	matrixRxC<T, Size, Size, IsRowMajor> ret;

	for(std::size_t i=0; i<Size; ++i)
	for(std::size_t j=0; j<Size; ++j)
	{
		ret(i, j) = get_determinant(get_cofactor(mat, j, i))
				  * static_cast<T>((i+j)%2==0 ? 1.0 : -1.0);
	}

	return ret/get_determinant(mat);
}

template
<
	typename	T,
	bool		IsRowMajor
>
const matrixRxC<T, 3, 3, IsRowMajor>
get_inversed
(
	const matrixRxC<T, 3, 3, IsRowMajor>& m
)
{
	T det
		= m(0, 0)*(m(1, 1)*m(2, 2) - m(1, 2)*m(2, 1))
		- m(0, 1)*(m(1, 0)*m(2, 2) - m(1, 2)*m(2, 0))
		- m(0, 2)*(m(1, 0)*m(2, 1) - m(1, 1)*m(2, 0));

	return matrixRxC<T, 3, 3, IsRowMajor>
	(
		m(1, 1)*m(2,2)-m(1,2)*m(2, 1), m(0, 2)*m(2,1)-m(0,1)*m(2,2), m(0,1)*m(1,2)-m(0,2)*m(1,1),
		m(1, 2)*m(2,0)-m(1,0)*m(2, 2), m(0, 0)*m(2,2)-m(0,2)*m(2,0), m(0,2)*m(1,0)-m(0,0)*m(1,2),
		m(1, 0)*m(2,1)-m(1,1)*m(2, 0), m(0, 1)*m(2,0)-m(0,0)*m(2,1), m(0,0)*m(1,1)-m(0,1)*m(1,0)
	)/det;
}

/*--------------------------------------------------*/
/*	gpuppur::get_normal_matrix						*/
/*--------------------------------------------------*/
/** This function Returns normal matrix from model view matrix
 *	@param	m	: [in] model (or/and) view matrix.
 *	@return	Normal matrix which used to
 *	translate normal vector from model space to camera space. 
 *	It is similar to gl_NormalMatrix in GLSL ver1.20.
 **/
template
<
	typename	T,
	bool		IsRowMajor
>
matrixRxC<T, 3, 3, IsRowMajor>
get_normal_matrix
(
	const matrixRxC<T, 4, 4, IsRowMajor>& m
)
{
	return get_transposed(get_sub<0, 0, 3, 3>(get_inversed(m)));
}

template
<
	typename	T,
	std::size_t	Row,
	std::size_t	Column,
	bool		IsRowMajor
>
const matrixRxC<T, Row, Column, IsRowMajor> operator *
(
	const matrixRxC<T, Row, Column, IsRowMajor>&	lhs,
	typename boost::call_traits<T>::param_type		rhs
)
{
	matrixRxC<T, Row, Column, IsRowMajor> ret;

	for(std::size_t i=0; i<Row*Column; ++i)
	{
		ret[i] = lhs[i] * rhs;
	}

	return ret;
}

template
<
	typename T,
	std::size_t Row,
	std::size_t Column,
	bool IsRowMajor
>
const matrixRxC<T, Row, Column, IsRowMajor> operator *
(
	typename boost::call_traits<T>::param_type		lhs,
	const matrixRxC<T, Row, Column, IsRowMajor>&	rhs
)
{
	matrixRxC<T, Row, Column, IsRowMajor> ret;

	for(std::size_t i=0; i<Row*Column; ++i)
	{
		ret[i] = lhs * rhs[i];
	}

	return ret;
}

template
<
	typename	T,
	std::size_t	Row,
	std::size_t	Column,
	bool		IsRowMajor
>
const matrixRxC<T, Row, Column, IsRowMajor> operator /
(
	const matrixRxC<T, Row, Column, IsRowMajor>&	lhs,
	typename boost::call_traits<T>::param_type		rhs
)
{
	matrixRxC<T, Row, Column, IsRowMajor> ret;

	for(std::size_t i=0; i<Row*Column; ++i)
	{
		ret[i] = lhs[i] / rhs;
	}

	return ret;
}

template
<
	typename	T,
	std::size_t	CommonSize,
	std::size_t	ColumnRHS,
	bool		IsRowMajorRHS
>
VectorNd<T, ColumnRHS> operator *
(
	typename
	gpuppur::matrixRxC<T, CommonSize, ColumnRHS, IsRowMajorRHS>::column_type const &
	lhs,
	const gpuppur::matrixRxC<T, CommonSize, ColumnRHS, IsRowMajorRHS>& rhs
)
{
	VectorNd<T, ColumnRHS>	ret;

	for(std::size_t i=0; i<ColumnRHS; ++i)
	{
		ret[i] = lhs.innerProduct(rhs.column(i));
	}

	return ret;
}

template
<
	typename T,
	std::size_t RowLHS,
	std::size_t CommonSize,
	bool IsRowMajorLHS
>
VectorNd<T, RowLHS> operator *
(
	const gpuppur::matrixRxC<T, RowLHS, CommonSize, IsRowMajorLHS>& lhs,
	typename
	gpuppur::matrixRxC<T, RowLHS, CommonSize, IsRowMajorLHS>::row_type const &
	rhs
)
{
	VectorNd<T, RowLHS>	ret;

	for(std::size_t i=0; i<RowLHS; ++i)
	{
		ret[i] = lhs.row(i).innerProduct(rhs);
	}

	return ret;
}

template
<
	typename	T,
	bool		IsRowMajor
>
matrixRxC<T, 4, 4, IsRowMajor> get_view_matrix
(
	const VectorNd<T, 3>& position,
	const VectorNd<T, 3>& front_dir,
	const VectorNd<T, 3>& up_dir
)
{
	matrixRxC<T, 4, 3, IsRowMajor> direction_mat
	(
		front_dir.outerProduct(up_dir),
		up_dir,
		-front_dir,
		VectorNd<T, 3>(static_cast<T>(0.0), static_cast<T>(0.0), static_cast<T>(0.0))
	);

	matrixRxC<T, 4, 4, IsRowMajor> ret
	(
		direction_mat,
		matrixRxC<T, 4, 1, IsRowMajor>
		(
			direction_mat*-position
		)
	);

	ret[15] = static_cast<T>(1.0);

	return ret;
}

}	//end of namespace gpuppur

#undef MAX_CONSTRUCTOR_ARG
#undef GET_ASSIGN_GEN
#undef GET_CONSTRUCTOR_NAME
#undef ASSIGN_GEN
#undef ASSIGN_num_column_MAJOR_GEN
#undef CONSTRUCTOR_GEN
#undef CONSTRUCTOR
#undef CONSTRUCTOR_COLUMN_MAJOR

#undef ASSIGN_VECTOR_GEN
#undef ASSIGN_NON_MAJOR_VECTOR_GEN
#undef CONSTRUCTOR_FROM_VECTORS

#endif
