#ifndef MATH3D_VECTOR_HPP
#define MATH3D_VECTOR_HPP

/**
 *	@file	
 *	@brief	Vector class
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <cassert>
#include <iostream>
#include <boost/array.hpp>
#include <boost/call_traits.hpp>
#include <boost/mpl/integral_c.hpp>
#include <boost/preprocessor.hpp>
#include <boost/static_assert.hpp>
#include <boost/tuple/tuple.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

namespace gpuppur
{
#define MAX_CONSTRUCTOR_ARG	8

#define ASSIGN_GEN(z, n, data)								\
		this->_vec[(n)+(data)] = arg##n;

#define CONSTRUCTOR_FROM_ELEMENT_GEN(z, n, _)				\
		VectorNd											\
		(													\
			BOOST_PP_ENUM_PARAMS							\
			(												\
				n,											\
				typename									\
				boost::call_traits<real>::param_type arg	\
			)												\
		)													\
		{													\
			BOOST_STATIC_ASSERT(N>=n);						\
			BOOST_PP_REPEAT									\
			(												\
				n,											\
				ASSIGN_GEN,									\
				0											\
			)												\
		}

#define CONSTRUCTOR_FRON_ELEMENT							\
		BOOST_PP_REPEAT_FROM_TO								\
		(													\
			2,												\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG),				\
			CONSTRUCTOR_FROM_ELEMENT_GEN,					\
			_												\
		)


#define CONSTRUCTOR_FROM_1VEC_AND_ELEMENT_GEN(z, n, _)		\
		VectorNd											\
		(													\
			const VectorNd<real, N-n>& vec,					\
			BOOST_PP_ENUM_PARAMS							\
			(												\
				n,											\
				typename									\
				boost::call_traits<real>::param_type arg	\
			)												\
		)													\
		{													\
			BOOST_STATIC_ASSERT(N>n);						\
			this->get_sub<0, N-n>() = vec;					\
			BOOST_PP_REPEAT									\
			(												\
				n,											\
				ASSIGN_GEN,									\
				N-n											\
			)												\
		}													\
															\
		VectorNd											\
		(													\
			BOOST_PP_ENUM_PARAMS							\
			(												\
				n,											\
				typename									\
				boost::call_traits<real>::param_type arg	\
			),												\
			const VectorNd<real, N-n>& vec					\
		)													\
		{													\
			BOOST_STATIC_ASSERT(N>n);						\
			BOOST_PP_REPEAT									\
			(												\
				n,											\
				ASSIGN_GEN,									\
				0											\
			)												\
			this->get_sub<n, N>() = vec;					\
		}

#define CONSTRUCTOR_FRON_1VEC_AND_ELEMENT					\
		BOOST_PP_REPEAT_FROM_TO								\
		(													\
			1,												\
			BOOST_PP_INC(MAX_CONSTRUCTOR_ARG),				\
			CONSTRUCTOR_FROM_1VEC_AND_ELEMENT_GEN,			\
			_												\
		)

/**
 *	@brief N dimention vector template class.
 *
 *	Number of elements of this vector class is decided in compile time.
 *	Memory image of array of this class(e.g. VectorNd<float, 3> a[10];) must be identical to array of "real"(e.g. float a[30];).
 *	So you can use this class to manage data which passed to 3D API (like OpenGL, DirectX or PhysX).
 *
 *	@todo	Implement with boost library to simplify implementation.
 *
 *	@param real	Type of element of this vector class.
 *	@param N	Number of element of this vector class.
 */
template <typename real, std::size_t N>
struct VectorNd
{
public:
	typedef VectorNd<real, N>	this_type;
	typedef VectorNd<real, N-1>	down_dimention_type;
	typedef real				element_type;
	typedef element_type		value_type;
	typedef typename boost::call_traits<value_type>::param_type value_type_param;

	enum
	{
		dimention = N
	};

	template<std::size_t Scaler>
	struct scaled_type
	{
		typedef VectorNd<real, N*Scaler>	type;
	};

	template<int Adder>
	struct increased_type
	{
		typedef VectorNd<real, N+Adder>		type;
	};

protected:

//	element_type _vec[N];
	boost::array<real, N>	_vec;

public:
	VectorNd()
	{
	}
 
	/**
	 *	@brief Construct from single scaler.
	 *
	 *	Specified val is copied to all elements of this vector.
	 */
	explicit VectorNd(value_type_param val)
	{
		for(std::size_t i=0; i< N; ++i)
			this->_vec[i] = val;
	}

	/**
	 *	@brief Copy constructor
	 */
	template<typename T>
	VectorNd(const VectorNd<T, N>& vec)
	{
		this->_vec = vec._vec;
	}

	CONSTRUCTOR_FRON_ELEMENT
	CONSTRUCTOR_FRON_1VEC_AND_ELEMENT

	template<typename T>
	VectorNd(const T vec[])
	{
		for(std::size_t i=0; i< N; ++i)
			this->_vec[i] = static_cast<real>(vec[i]);
	}

	template<typename T0, std::size_t N0, typename T1, std::size_t N1>
	VectorNd(const VectorNd<T0, N0>& v0, const VectorNd<T1, N1>& v1)
	{
		this->get_sub<0, N0>() = v0;
		this->get_sub<N0, N0+N1>() = v1;
	/*
		std::size_t i=0;

		for(; i<N0; ++i)
			this->_vec[i] = v0[i];
		for(; i<N0+N1; ++i)
			this->_vec[i] = v1[i-N0];
	*/
	}

	const element_type& operator[] (std::size_t i) const
	{
		return this->_vec[i];
	}

	element_type& operator[] (std::size_t i)
	{
		return this->_vec[i];
	}

	this_type& operator+= (const this_type& vec)
	{
		for(std::size_t i=0; i<N; ++i)
			(*this)[i] = static_cast<element_type>((*this)[i] + vec[i]);

		return *this;
	}

	const element_type* raw_data() const
	{
		return this->_vec.data();
	}

	this_type& operator-= (const this_type& vec)
	{
		for(std::size_t i=0; i<N; ++i)
			(*this)[i] = static_cast<element_type>((*this)[i] - vec[i]);

		return *this;
	}

	this_type& operator*= (value_type_param val)
	{
		for(std::size_t i=0; i<N; ++i)
			(*this)[i] = static_cast<element_type>((*this)[i] * val);

		return *this;
	}

	this_type& operator*= (const this_type& val)
	{
		for(std::size_t i=0; i<N; ++i)
			(*this)[i] = static_cast<element_type>((*this)[i]*val[i]);

		return *this;
	}

	this_type& operator /= (value_type_param val)
	{
		for(std::size_t i=0; i<N; ++i)
			(*this)[i] = static_cast<element_type>((*this)[i] / val);

		return *this;
	}

	this_type& operator /= (const this_type& val)
	{
		for(std::size_t i=0; i<N; ++i)
			(*this)[i] = static_cast<element_type>((*this)[i] / val[i]);

		return *this;
	}

	const this_type operator + (const this_type& vec) const
	{
		this_type r(*this);

		r += vec;
		return r;
	}

	const this_type operator - (const this_type& vec) const
	{
		this_type r(*this);

		r -= vec;
		return r;
	}

	const this_type operator * (value_type_param val) const
	{
		this_type r(*this);

		r *= val;
		return r;
	}

	const this_type operator * (const this_type& val) const
	{
		this_type r(*this);

		r *= val;
		return r;
	}

	const this_type operator / (value_type_param val) const
	{
		this_type r(*this);

		r /= val;
		return r;
	}

	const this_type operator / (const this_type& val) const
	{
		this_type r(*this);

		r /= val;
		return r;
	}

	const this_type operator - () const
	{
		this_type r(*this);

		r *= static_cast<element_type>(-1.0);

		return r;
	}

	bool operator == (const this_type& val) const
	{
		for(std::size_t i=0; i<N; ++i)
		{
			if((*this)[i] != val[i])
			{
				return false;
			}
		}

		return true;
	}

	bool operator != (const this_type& val) const
	{
		return !(*this == val);
	}

	const element_type innerProduct(const this_type& vec) const
	{
		element_type r = static_cast<element_type>(0.0);

		for(std::size_t i=0; i<N; ++i)
			r += (*this)[i] * vec[i];

		return r;
	}

	const typename gpuppur::VectorNd<real, 3>
		outerProduct(const gpuppur::VectorNd<real, 3>& vec) const
	{
		BOOST_STATIC_ASSERT(N==3);

		return VectorNd<real, 3>
		(
			(*this)[1] * vec[2] - (*this)[2] * vec[1],
			(*this)[2] * vec[0] - (*this)[0] * vec[2],
			(*this)[0] * vec[1] - (*this)[1] * vec[0]
		);
	}

	/**	Return square of length of vector
	 *
	 *	This function is probably faster than length().
	 *
	 *	@return	Square of length of this vector.
	 */
	element_type length2() const
	{
		element_type r = (*this)[0] * (*this)[0];

		for(std::size_t i=1; i<N; ++i)
			r += (*this)[i] * (*this)[i];

		return r;
	}

	/**	Return square of length of vector
	 *
	 *	When calculationg length of vector, temporary variable might exceed value which the type "real" can represent,
	 *	and wrong value is returned.
	 *	To avoid such situation, you can specify type of temporary variable and return value.
	 *
	 *	@return	Square of length of this vector.
	 */
	template<class Ret_Type>
	Ret_Type length2() const
	{
		Ret_Type r = static_cast<Ret_Type>((*this)[0])
					 *
					 (*this)[0];

		for(std::size_t i=1; i<N; ++i)
			r += static_cast<Ret_Type>((*this)[i])
				 *
				 (*this)[i];

		return r;
	}

	/**
	 *	Return length of this vector
	 *
	 */
	element_type length() const
	{
		return sqrt(this->length2());
	}

	void normalize()
	{
		element_type l = this->length();

		for(std::size_t i=0; i<N; ++i)
			(*this)[i] /= l;
	}

	const this_type getNormalized() const
	{
		element_type l = this->length();
		this_type ret;

		for(std::size_t i=0; i<N; ++i)
			ret[i] = (*this)[i] / l;

		return ret;
	}

	const this_type reflect(const this_type& normal) const
	{
		return (*this - normal*normal.innerProduct(*this))*2.0 - *this;
	}

	template<std::size_t Begin, std::size_t End>
	VectorNd<real, End-Begin>& get_sub()
	{
		BOOST_STATIC_ASSERT(Begin >= 0 && End > 0);
		BOOST_STATIC_ASSERT(Begin < N && End <= N);

		return *reinterpret_cast<VectorNd<real, End-Begin>*>(&this->_vec[Begin]);
	}

	template<std::size_t Begin, std::size_t End>
	const VectorNd<real, End-Begin>& get_sub() const
	{
		BOOST_STATIC_ASSERT(Begin >= 0 && End > 0);
		BOOST_STATIC_ASSERT(Begin < N && End <= N);

		return *reinterpret_cast<const VectorNd<real, End-Begin>*>(&this->_vec[Begin]);
	}

	std::size_t	getMaxAbsComp() const
	{
		element_type max = (*this)[0]*(*this)[0];
		std::size_t ret = 0;

		for(std::size_t i=1; i<N; ++i)
		{
			element_type v = (*this)[i]*(*this)[i];

			if(v > max)
			{
				max = v;
				ret = i;
			}
		}

		return ret;
	}

	bool	isMaxAbsComp(std::size_t comp) const
	{
		element_type v = (*this)[comp]*(*this)[comp];
		bool ret = true;

		for(std::size_t i=0; i<N; ++i)
		{
			ret = (ret && v >= (*this)[i]*(*this)[i]);
		}

		return ret;
	}

	// 下のテンプレートクラスは
	// template<typename ret, class function>
	// ret	forMaxAbsComp(function f) const
	// 関数でのみ使う。本当はローカルクラスにしたいんだけど。 

	//下の複雑なテンプレートクラスは標準C++への準拠度が低いVisual Studio 2005には扱えない。 
#ifndef _MSC_VER
	template<std::size_t v0, std::size_t v1>
	class maxCond
	{
	private:
		BOOST_STATIC_ASSERT(v0 < this_type::dimention && v1 < this_type::dimention);
	public:
		template<typename array>
		static bool comp(const array& value)
		{
			return
				value[v0]*value[v0] >
				value[v1]*value[v1] &&
				maxCond<v0, v1+1>::comp(value);
		}
	};

	template<std::size_t v0>
	class maxCond<v0, this_type::dimention>
	{
	public:
		template<typename array>
		static bool comp(const array&)
		{
			return true;
		}
	};

	template<typename ret, std::size_t v0>
	class call_if
	{
	private:
		BOOST_STATIC_ASSERT(v0 < this_type::dimention-1);
	public:
		template<typename array, class function>
		static ret call(const array& value, function f)
		{
			if(maxCond<v0, v0+1>::comp(value))
			{
				//本当はboost::mpl::integral<>の
				//型さえ渡せればいいのだが、
				//f<boost::mpl::integral<> >()とかって
				//書くことができないようだ。
				return f(
						boost::mpl::integral_c
						<std::size_t, v0>()
					);
			}

			return call_if<ret, v0+1>::call(value, f);
		}
	};

	template<typename ret>
	class call_if<ret, this_type::dimention-1>
	{
	public:
		template<typename array, class function>
		static ret call(const array&, function f)
		{
			return f(
					boost::mpl::integral_c
					<std::size_t, dimention-1>()
				);
		}
	};

	template<class function, typename ret>
	ret	forMaxAbsComp(function f, ret) const
	{
		return call_if<ret, 0>::call(*this, f);
	}
#endif	//end of #ifndef _MSC_VER

	template<typename, std::size_t> friend struct VectorNd;
};

/**
 *	Get rotated vector by argument2 angle_rad radian 
 *	from argument0 base to argument1 vertical.
 *	Argument vertical should be vertical to base,
 *	if you want to rotate vector by desired angle.
 */
template<typename real, std::size_t N>
const VectorNd<real, N> rotate_vector
(
	const VectorNd<real, N>&	base,
	const VectorNd<real, N>&	vertical,
	real						angle_rad
)
{
	return VectorNd<real, N>
	(
		base*cos(angle_rad) + vertical*sin(angle_rad)
	);
}

template<typename real, std::size_t N>
void rotate_2vectors
(
	VectorNd<real, N>&	base,
	VectorNd<real, N>&	vertical,
	real				angle_rad
)
{
	VectorNd<real, N> old_base(base);

	real cos_angle = cos(angle_rad);
	real sin_angle = sin(angle_rad);

	base		= old_base*cos_angle+vertical*sin_angle;
	vertical	= vertical*cos_angle+old_base*(-sin_angle);
}

template <typename real, std::size_t N>
std::ostream& operator<< (std::ostream& os, const VectorNd<real, N>& vec)
{
	std::size_t i;

	for(i=0; i<N-1; ++i)
		os << vec[i] << "\t" ;
	os << vec[i] << "\n";

	return os;
}

template <typename real, std::size_t N>
std::istream& operator>> (std::istream& is, VectorNd<real, N>& vec)
{

	for(std::size_t i=0; i<N; ++i)
		is >> vec[i];

	return is;
}

template<typename real, std::size_t N>
const VectorNd<real, N> operator * (real scaler, const VectorNd<real, N>& vec)
{
	return vec*scaler;
}

template<typename real, std::size_t N>
const VectorNd<real, N> operator / (real scaler, const VectorNd<real, N>& vec)
{
	return vec/scaler;
}

}	//end of namespace gpuppur

#undef MAX_CONSTRUCTOR_ARG	
#undef ASSIGN_GEN
#undef CONSTRUCTOR_FROM_ELEMENT_GEN
#undef CONSTRUCTOR_FRON_ELEMENT
#undef CONSTRUCTOR_FROM_1VEC_AND_ELEMENT_GEN
#undef CONSTRUCTOR_FRON_1VEC_AND_ELEMENT

#endif
