#ifndef GPUPPUR_ERROR_HANDLER
#define GPUPPUR_ERROR_HANDLER

/**
 *	@file	
 *	@brief	Class and macros for handling error.
 *	@author	Tomohiro Matsumoto
 */

#include <iostream>
#include <string>

/**
 *	Call this macro when file is not found.
 */
#define ERRHNDLR_FILE_NOT_FOUND(filename)					\
		(gpuppur::error_handler::get_error_handler()		\
		 ->file_not_found(__FILE__, __LINE__, (filename)))
/**
 *	Call this macro when shader file is not found.
 */
#define ERRHNDLR_FAILED_TO_LOAD_SHADER(filename)			\
		(gpuppur::error_handler::get_error_handler()		\
		 ->failed_to_load_shader(__FILE__, __LINE__, (filename)))

/**
 *	Call this macro when there is no enough memory to store vertex data.
 */
#define ERRHNDLR_NOT_ENOUGH_MEMORY_TO_STORE_VERTICES(vert_size)		\
		(gpuppur::error_handler::get_error_handler()				\
		 ->not_enough_memory_to_store_vertices(						\
			 __FILE__, __LINE__, (vert_size)))
/**
 *	Call this macro when there is no enough memory to store index data.
 */
#define ERRHNDLR_NOT_ENOUGH_MEMORY_TO_STORE_TRIANGLE_INDICES(index_size)	\
		(gpuppur::error_handler::get_error_handler()						\
		 ->not_enough_memory_to_store_triangle_indices(						\
			 __FILE__, __LINE__, (index_size)))
#define ERRHNDLR_INVALID_TRIANGLE_MESH()									\
		(gpuppur::error_handler::get_error_handler()						\
		 ->invalid_triangle_mesh(__FILE__, __LINE__))

namespace gpuppur
{
	class error_handler_interface
	{
	public:
		virtual void file_not_found(const char* file, int line, const std::string& filename) = 0;
		virtual void file_not_found(const char* file, int line, const std::wstring& filename) = 0;
		virtual void failed_to_load_shader(const char* file, int line, const std::string& filename) = 0;
		virtual void failed_to_load_shader(const char* file, int line, const std::wstring& filename) = 0;

		virtual void not_enough_memory_to_store_vertices
		(
			const char*	file,
			int			line,
			std::size_t	total_vertices_size
		) = 0;
		virtual void not_enough_memory_to_store_triangle_indices
		(
			const char*	file,
			int			line,
			std::size_t	total_indices_size
		) = 0;
		virtual void invalid_triangle_mesh
		(
			const char* file,
			int			line
		) = 0;
	};

	class default_error_handler : public error_handler_interface
	{
	public:
		default_error_handler()
		{
			std::wcerr.imbue(std::locale(""));
		}

		void file_not_found(const char*, int, const std::string& filename)
		{
			std::cerr << "File - " << filename << " - is not found." << std::endl;
		}

		void file_not_found(const char*, int, const std::wstring& filename)
		{
			std::wcerr << L"File - " << filename << L" - is not found." << std::endl;
		}

		void failed_to_load_shader(const char*, int, const std::string& filename)
		{
			std::cerr << "Failed to load shader " << filename << std::endl;
		}

		void failed_to_load_shader(const char*, int, const std::wstring& filename)
		{
			std::wcerr << L"Failed to load shader " << filename << std::endl;
		}
		
		void not_enough_memory_to_store_vertices
		(
			const char*,
			int,
			std::size_t	total_vertices_size
		)
		{
			std::cerr << "Not enough memory to store vertex data."
						"(tried to allocate " << total_vertices_size << " bytes)"
						<< std::endl;
		}

		void not_enough_memory_to_store_triangle_indices
		(
			const char*,
			int,
			std::size_t	total_indices_size
		)
		{
			std::cerr << "Not enough memory to store triangle index data."
						"(tried to allocate " << total_indices_size << " bytes)"
						<< std::endl;
		}

		void invalid_triangle_mesh
		(
			const char*,
			int
		)
		{
			std::cerr << "Invalid triangle mesh data" << std::endl;
		}
	};

	class error_handler
	{
	public:
		static error_handler_interface* get_error_handler()
		{
			static default_error_handler handler;

			return &handler;
		}
	};

}	//end of namespace gpuppur

#endif
