#ifndef ERROR_REPORT_GL_HPP
#define ERROR_REPORT_GL_HPP

/**
 *	@file	
 *	@brief	
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/error_manage/error_report_common.hpp>


#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <cstdlib>
#include <GL/gl.h>
#include <GL/glu.h>
#include <cassert>
#include <boost/array.hpp>
#include <boost/preprocessor.hpp>
#include <boost/mpl/void.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#define MAX_NUM_EXPECTED_ERRORS 8

namespace gpuppur
{
	class error_report_gl
	{
	public:

	#ifdef BUILD_DEBUG_PRINTER
		#define CHECK_EXPECTED_ERROR_GEN(z, n, _)	\
			if(err##n== GL_NO_ERROR)				\
			{										\
				goto handle_unexpected_err;			\
			}										\
			if(error_code == err##n)				\
			{										\
				ret = false;						\
				continue;							\
			}

		static bool gl_assert
		(
			const _TCHAR* const file,
			unsigned int line,
			BOOST_PP_ENUM_BINARY_PARAMS
			(
				MAX_NUM_EXPECTED_ERRORS,
				GLenum err,
				= GL_NO_ERROR BOOST_PP_INTERCEPT
			)
		)
		{
			GLenum error_code;
			bool	ret = true;
			GLenum last_error_code = GL_NO_ERROR;

			while( (error_code = glGetError()) != GL_NO_ERROR)
			{
				//If glGetError() called when there are no GL context,
				//glGetError() always returns GL_INVALID_OPERATION.
				if
				(
					last_error_code == GL_INVALID_OPERATION
					&&
					error_code == GL_INVALID_OPERATION
				)
				{
					gpuppur::error_report_common::debug_print
					(
						file, line, _T("Operation without GL context.\n")
					);

					return ret;
				}
				last_error_code = error_code;


				BOOST_PP_REPEAT
				(
					MAX_NUM_EXPECTED_ERRORS,
					CHECK_EXPECTED_ERROR_GEN,
					_
				)

			handle_unexpected_err:

				const GLubyte * error_msg = gluErrorString(error_code);

		#ifdef UNICODE
				const std::size_t len
				=
				mbstowcs(NULL, reinterpret_cast<const char*>(error_msg), 0);

				std::vector<wchar_t> wcs(len+1);

				std::size_t ret_val
				=
				mbstowcs(&wcs[0], reinterpret_cast<const char*>(error_msg), len);

				if(ret_val == static_cast<std::size_t>(-1))
				{
					assert(("Wrong Err message from gluErrorString()", false));
					return ret;
				}
				gpuppur::error_report_common::debug_print
				(
					file, line, _T("OpenGL Assertion failed %ls\n"), &wcs[0]
				);
		#else
				gpuppur::error_report_common::debug_print
				(
					file, line, _T("OpenGL Assertion failed %s\n"), error_msg
				);
		#endif
			}

			return ret;
		}
	#else
		#define CHECK_EXPECTED_ERROR_GEN(z, n, _)	\
			if(err##n== GL_NO_ERROR)				\
			{										\
				continue;							\
			}										\
			if(error_code == err##n)				\
			{										\
				return false;						\
			}

		static bool gl_assert
		(
			boost::mpl::void_,
			BOOST_PP_ENUM_BINARY_PARAMS
			(
				MAX_NUM_EXPECTED_ERRORS,
				GLenum err,
				= GL_NO_ERROR BOOST_PP_INTERCEPT
			)
		)
		{
			GLenum error_code;// = glGetError();

			//if (error_code != GL_NO_ERROR)
			while( (error_code = glGetError()) != GL_NO_ERROR)
			{
				BOOST_PP_REPEAT
				(
					MAX_NUM_EXPECTED_ERRORS,
					CHECK_EXPECTED_ERROR_GEN,
					_
				)
			}

			return true;
		}

	#endif

	};
}	// end of namespace gpuppur

#if 0
GL_ERROR_CHECK return false if glGetError() returned specified Error.
If glGetError() returned unspecified Error, GL_ERROR_CHECK return true.
If there is no error, GL_ERROR_CHECK return true.
Argument of GL_ERROR_CHECK must be sequences.(refer boost.preprocessor)
Use GL_ERROR_CHECK like
GL_ERROR_CHECK((GL_INVALID_OPERATION))
GL_ERROR_CHECK((GL_INVALID_OPERATION)(GL_OUT_OF_MEMORY))

#endif

#define EXPAND_SEQ(r, data, elem)	\
		, elem

#ifdef BUILD_DEBUG_PRINTER

	#define GL_ERROR_ASSERT gpuppur::error_report_gl::gl_assert(_T(__FILE__), __LINE__)
	#define GL_ERROR_CHECK(expected_errors)			\
			gpuppur::error_report_gl::gl_assert		\
			(										\
				_T(__FILE__), __LINE__				\
				BOOST_PP_SEQ_FOR_EACH				\
				(									\
					EXPAND_SEQ,						\
					_,								\
					expected_errors					\
				)									\
			)

#else
	#define GL_ERROR_ASSERT
		#define GL_ERROR_CHECK(expected_errors)		\
			gpuppur::error_report_gl::gl_assert		\
			(										\
				boost::mpl::void_()					\
				BOOST_PP_SEQ_FOR_EACH				\
				(									\
					EXPAND_SEQ,						\
					_,								\
					expected_errors					\
				)									\
			)
#endif

#endif
