#ifndef GPUPPUR_CPU_RAYTRACER_HPP
#define GPUPPUR_CPU_RAYTRACER_HPP

/**
 *	@file	
 *	@brief	CPU Raytracer class
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <algorithm>
#include <list>
#include <map>
#include <set>
#include <boost/tuple/tuple.hpp>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/call_traits.hpp>
#include <boost/type.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

namespace gpuppur
{
namespace cpu_raytracers
{
namespace detail
{
	class mesh;
	void release_mesh(mesh*);
//	template<class> class instance3d;

//	template<class UserDataType>
//	void release_instance3d(instance3d<UserDataType>*);

	class cpu_raytracer_implement;

}}
	template<typename UserDataType> class cpu_raytracer;
}

#include "../handle_generic.hpp"
#include "../handler.hpp"
#include "mesh.hpp"
#include "instance3d.hpp"
#include <gpuppur/3dfileloader/3dfileLoader.hpp>

namespace gpuppur
{
namespace cpu_raytracers
{
namespace detail
{

template<class Handled, class Lister, class Iterator>
void release_handled
(
	Handled*		handled,
	const Lister&	lister,
	boost::type<Iterator>
)
{
	if(!handled)
	{
		return;
	}

	if(lister().empty())
		return;

	Iterator i;
	for(i=lister().begin();
		i!=lister().end();
		++i)
	{
		if(i->second.find(handled) != i->second.end())
		{
			i->second.erase(handled);
			i->first->erase(handled);
			return;
		}
	}
}

/*
 *	List of pointer to alive cpu_raytracer,
 *	and set of pointer to alive instace3d.
 */
typedef std::map
<
	cpu_raytracer_implement*,
	std::set<cpu_raytracers::detail::mesh*>
> mesh_set_per_raytracer;

inline mesh_set_per_raytracer& get_valid_mesh_set()
{
	static mesh_set_per_raytracer mspr;

	return mspr;
}

inline void release_mesh
(
	mesh*	ins
)
{
	release_handled
	(
		ins, &get_valid_mesh_set,
		boost::type<mesh_set_per_raytracer::iterator>()
	);
}

/**
 *	@brief	CPU Raytracer class
 *
 *	You can create/destory 3D object in this class.
 *	And this class can raytracing at them.
 */
class cpu_raytracer_implement
{
public:

	typedef gpuppur::handler_tmpl<handler_to_mesh>::c_static		mesh_static;
	typedef gpuppur::handler_tmpl<handler_to_mesh>::c_tmp			mesh_tmp;
	typedef gpuppur::handler_tmpl<handler_to_mesh>::c_alone			mesh_alone;
/*	typedef boost::function
	<
		void
		(
			const instance3d_tmp,
			const vector3&	pos,
			const vector3&	normal,
			const int		id
		)
	>	casted_ray_function;
*/
protected:

	cpu_raytracer_implement():
		is_initialized(false)
	{
	}

	~cpu_raytracer_implement()
	{
	}

	void uninitialize()
	{
		this->meshes.clear();
		get_valid_mesh_set().erase(this);
		this->is_initialized = false;
	}

	bool initialize()
	{
		this->uninitialize();

		this->self_itr =
		get_valid_mesh_set().insert
		(
			mesh_set_per_raytracer::value_type
			(
				this,
				mesh_set_per_raytracer::value_type::second_type()
			)
		).first;

		this->is_initialized = true;

		return true;
	}

public:

	void process()
	{
	}

/*
	void cast_ray_batch
	(
		const	ray& casted_ray,
		float	max_t,
		int		id
	) const
	{
		assert
		(
			this->scene_query
			&&
			"call set_casted_ray_func() before calling this function!"
		);

		NxRaycastHit hit;

		this->scene_query->raycastClosestShape
		(
			*reinterpret_cast<NxRay*>(&const_cast<ray&>(casted_ray)),
			NX_ALL_SHAPES,
			hit,
			0xffffffff,
			max_t,
			NX_RAYCAST_SHAPE | NX_RAYCAST_FACE_NORMAL,//| NX_RAYCAST_NORMAL,
			NULL,
			NULL,	//cache,
			reinterpret_cast<void*>(id)		//userData
		);
	}

	void cast_ray_execute()
	{
		assert(this->physics_sdk);
		assert(this->scene);
		assert(this->scene_query);
		assert(this->query_reported.has_data());

		this->scene_query->execute();

		this->scene_query->finish(true);
	}

	void set_casted_ray_func(const casted_ray_function& func)
	{
		assert(this->physics_sdk);
		assert(this->scene);

		this->query_reported.func = func;

		if(this->scene_query)
		{
			bool ret = this->scene->releaseSceneQuery(*this->scene_query);
			assert(ret && "I thought this->scene->releaseSceneQuery never fail...");
		}

		NxSceneQueryDesc	desc;
		desc.report			= &this->query_reported;
		desc.executeMode	= NX_SQE_ASYNCHRONOUS;
		assert(desc.isValid() && "barairo no zinsei");
		this->scene_query = this->scene->createSceneQuery(desc);
		assert
		(
			this->scene_query
			&&
			"I thought this->scene->createSceneQuery() never fail..."
		);
	}

	const casted_ray_function get_casted_ray_func() const
	{
		return this->query_reported.func;
	}
*/
	mesh_tmp load_mesh
	(
		const std::vector<vector3>& vertices,
		const std::vector<unsigned short>& triangles
	)
	{
		this->assert_initialized();

		this->meshes.push_front(cpu_raytracers::detail::mesh(vertices, triangles));
		this->meshes.front().set_self_iterator(this->meshes.begin());

		this->self_itr->second.insert(&this->meshes.front());

		return mesh_tmp(this->meshes.front());
	}

	mesh_tmp load_mesh
	(
		const std::vector<vector3>& /*vertices*/,
		const std::vector<unsigned short>& /*triangles*/,
		const std::string& /*filename*/
	)
	{
		this->assert_initialized();
		assert(false && "This function will be implemented in the near future.");

		return mesh_tmp();
	}

	mesh_tmp load_mesh_from_wavefront(const std::string& filename)
	{
		this->assert_initialized();

		Model model;

		if(!model.load(filename))
		{
			return mesh_tmp();
		}

		// only use model.objects[0]
		std::vector<vector3> vertices(model.objects[0].verts.size());
		std::vector<unsigned short> triangles(model.objects[0].faces.size()*3);

		int n=0;
		for
		(
			std::vector<Vector3d>::iterator i=model.objects[0].verts.begin();
			i!=model.objects[0].verts.end();
			++i
		)
		{
			vertices[n++]
			=
			vector3
			(
				static_cast<float>((*i)[0]),
				float((*i)[1]),
				float((*i)[2])
			);
		}

		n=0;
		for
		(
			Object::Faces::iterator i=model.objects[0].faces.begin();
			i!=model.objects[0].faces.end();
			++i
		)
		for(size_t j=0; j<3; ++j)
		{
			triangles[n++] = static_cast<unsigned short>(i->vertIndices[j]);
		}

		return this->load_mesh(vertices, triangles/*, cooked_filename*/);
	}

	mesh_tmp load_mesh_from_cooked(const std::string& /*filename*/)
	{
		this->assert_initialized();
		assert(false && "This function is not yet implemented");

		return mesh_tmp();
	}

protected:

	void assert_initialized() const
	{
		assert(this->is_initialized);
	}

private:

	void erase(cpu_raytracers::detail::mesh*const p_mesh)
	{
		this->assert_initialized();

		this->meshes.erase(p_mesh->get_self_iterator());
	}

	mutable int					count_sampling;
	std::list<mesh>				meshes;
	mesh_set_per_raytracer::iterator
								self_itr;
	bool						is_initialized;

//	friend	void release_mesh(mesh*);

	template<class Handled, class Lister, class Iterator>
	friend void release_handled
	(
		Handled*,
		const Lister&,
		boost::type<Iterator>
	);

};	// end of class cpu_raytracer_implement

}	// end of namespace detail
}	// end of namespace cpu_raytracers

template<typename UserDataType>
class cpu_raytracer : public gpuppur::cpu_raytracers::detail::cpu_raytracer_implement
{
public:

	typedef gpuppur::cpu_raytracers::detail::instance3d<UserDataType> instance3d_state;
	typedef typename gpuppur::handler_tmpl
	<
		gpuppur::cpu_raytracers::detail::handler_to_instance3d<UserDataType>::template c
	>::c_static		instance3d_static;

	typedef typename gpuppur::handler_tmpl
	<
		gpuppur::cpu_raytracers::detail::handler_to_instance3d<UserDataType>::template c
	>::c_tmp		instance3d_tmp;

	typedef cpu_raytracer_implement		base;
	typedef cpu_raytracer<UserDataType>	this_type;
/*
	friend
	bool operator < 
	(
		const typename
	//	std::list<gpuppur::cpu_raytracers::detail::instance3d<UserDataType> >
	//	::iterator lhs,
		instance3d_state::iterator& lhs,
		const typename
	//	std::list<gpuppur::cpu_raytracers::detail::instance3d<UserDataType> >
	//	::iterator rhs
		instance3d_state::iterator& rhs
	)
	{
		return &(*lhs) < &(*rhs);
	}
*/
	cpu_raytracer()
	{
	//	bool ins = std::operator < /*<UserDataType>*/(this->instances.begin(), this->instances.end());
	}

	~cpu_raytracer()
	{
		this->uninitialize();
	}

private:

	typedef std::map
	<
		cpu_raytracer<UserDataType>*,
		std::set<instance3d_state*>
	> instance3d_set_per_raytracer;

	static instance3d_set_per_raytracer&	get_valid_instance3d_set()
	{
		static instance3d_set_per_raytracer ispr;

		return ispr;
	}

public:

	void uninitialize()
	{
		this->instances.clear();
		get_valid_instance3d_set().erase(this);

		base::uninitialize();
	}

	bool initialize()
	{
		this->uninitialize();

		this->self_instance3d_set_itr =
		get_valid_instance3d_set().insert
		(
			typename instance3d_set_per_raytracer::value_type
			(
				this,
				typename instance3d_set_per_raytracer::value_type::second_type()
			)
		).first;

		return base::initialize();
	}

	instance3d_tmp create_mesh
	(
		const mesh_tmp		handle,
		const vector3&		pos,
		const UserDataType& user_data
	)
	{
		this->assert_initialized();

		this->instances.push_front
		(
			cpu_raytracers::detail::instance3d<UserDataType>
			(
				handle.get_handled().get_self_iterator(),
				user_data
			)
		);
		this->instances.front().set_self_iterator(this->instances.begin());
		this->instances.front().set_position(pos);

		this->self_instance3d_set_itr->second.insert
		(
			&this->instances.front()
		);

		return instance3d_tmp(this->instances.front());
	}

	instance3d_tmp create_sphere
	(
		const float			radius,
		const vector3&		pos,
		const UserDataType&	user_data
	)
	{
		assert(false && "I hate SPHERE!!");

		return instance3d_tmp();
	}

private:

	typedef 
	std::list
	<
		instance3d_state
	> instance3d_list;

public:

	instance3d_tmp cast_ray
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal
	) const
	{
		this->assert_initialized();

		typename instance3d_list::const_iterator i = this->instances.begin();

		for(;i!=this->instances.end(); ++i)
		{
			if(i->cast_ray(casted_ray, max_t, pos, normal))
			{
				return instance3d_tmp(*i);
			}
		}

		return instance3d_tmp();
	}

	instance3d_tmp 
	cast_ray
	(
		const ray& casted_ray,
		float max_t
	) const;

private:

	void erase(instance3d_state* p_instance3d)
	{
		this->assert_initialized();

		this->instances.erase(p_instance3d->get_self_iterator());
	}

	instance3d_list			instances;
	typename instance3d_set_per_raytracer::iterator
							self_instance3d_set_itr;

	friend
	#ifdef _MSC_VER
		typename
	#else
		class
	#endif
	cpu_raytracers::detail::cpu_raytracer_implement;

	friend void release_instance3d
	(
		instance3d_state*			ins
	)
	{
	/*	typedef gpuppur::cpu_raytracer<UserDataType> raytracer;

		boost::type
		<
			typename raytracer::instance3d_set_per_raytracer::iterator
		> tmp;
	*/
		release_handled
		(
			ins,
		//	&raytracer::get_valid_instance3d_set,
			&get_valid_instance3d_set,
		//	tmp
			boost::type
			<
				typename instance3d_set_per_raytracer::iterator
			>()
		);
	}

//	typedef std::def_ins_op<UserDataType> nothing;
//	std::def_ins_op<UserDataType> z;

	template<class Handled, class Lister, class Iterator>
	friend void gpuppur::cpu_raytracers::detail::release_handled
	(
		Handled*,
		const Lister&,
		boost::type<Iterator>
	);
};	// end of class cpu_raytracer

}	// end of namespace gpuppur

#endif

