#ifndef GPUPPUR_CPU_INSTANCE3D_HPP
#define GPUPPUR_CPU_INSTANCE3D_HPP

/**
 *	@file
 *	@brief	Instance class of CPU Raytracer 
 *	@auther Tomohiro Matsumoto
 *
 *	@note	This file is only included from gpuppur/ppu/nxphysics.hpp.
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/3dmath/matrixRxC.hpp>
#include "mesh.hpp"

namespace gpuppur
{
namespace cpu_raytracers
{
namespace detail
{

template<typename UserDataType>
class instance3d
{
public:

	typedef typename std::list<instance3d<UserDataType> >::iterator iterator;

	instance3d
	(
		const mesh::iterator& mesh_ref,
		const UserDataType& user_data
	):
		mesh_ref(mesh_ref), user_data(user_data)
	{
		this->trans_mat.load_unit_matrix();
	}

	void set_self_iterator(iterator itr)
	{
		this->self = itr;
	}

	const iterator& get_self_iterator() const
	{
		return this->self;
	}

	/**	Set position of the instance
	 *
	 *	@param	new_pos	: [in]	new position of instance.
	 */
	void set_position(const vector3& new_pos)
	{
		this->trans_mat.column(3).get_sub<0, 3>() = new_pos;
	}

	/** Get position of the instance
	 *
	 *	@return	Current position of the instace.
	 */
	vector3 get_position() const
	{
		return this->trans_mat.column(3).get_sub<0, 3>();
	}

	void set_orientation(const matrix3x3& mat)
	{
		gpuppur::set_sub(this->trans_mat, mat, 0, 0);
	}

	const matrix4x4 get_trans_matrix() const
	{
		return this->trans_mat;
	}

	UserDataType& get_user_data()
	{
		return this->user_data;
	}

	const UserDataType& get_user_data() const
	{
		return this->user_data;
	}

	bool cast_ray
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal
	) const
	{
		if(
			this->mesh_ref->cast_ray
			(
				gpuppur::get_inversed(this->trans_mat)*casted_ray,
				max_t,
				pos, normal
			)
		)
		{
			ray ret = this->trans_mat*ray(pos, normal);

			pos = ret.getPos();
			normal = ret.getDir();

			return true;
		}

		return false;
	}

private:

	matrix4x4		trans_mat;
	mesh::iterator	mesh_ref;
	iterator		self;		//! Iterator to std::list<this class> for fast erase object from pointer to this class.
	UserDataType	user_data;	//! User can associate any data to this object.
};

/**
 *	@brief	CPU Raytracer's Instance3D class implement
 *	You can create instance of 3D object by calling some member function of class gpuppur::cpu_raytracer,
 *	and the instance of 3D object is rendered in scene.
 *	Such member function returns handle to instance, and you can control instance by calling member function of that.
 */
template<class UserDataType>
class handler_to_instance3d
{
public:
template<class Parent, class Base>
class c : public Base
{
public:

	typedef Base						base;
	typedef gpuppur::cpu_raytracers::detail::instance3d<UserDataType>
										handled_type;
	typedef gpuppur::instance3d_virtual	virtual_type;
	typedef gpuppur::instance3d_generic generic_type;
	typedef gpuppur::cpu_raytracer<UserDataType>
										friend_type;

private:

	handled_type& get_handled()
	{
		return reinterpret_cast<Parent*>(this)->get_handle();
	}

	const handled_type& get_handled() const
	{
		return reinterpret_cast<const Parent*>(this)->get_handle();
	}

public:

	/**	Set position of the instance
	 *
	 *	@param	new_pos	: [in]	new position of instance.
	 */
	void set_position(const vector3& new_pos)
	{
		return this->get_handled().set_position(new_pos);
	}

	/** Get position of the instance
	 *
	 *	@return	Current position of the instace.
	 */
	vector3 get_position() const
	{
		return this->get_handled().get_position();
	}

	void set_orientation(const matrix3x3& mat)
	{
		return this->get_handled().set_orientation(mat);
	}

	const matrix4x4 get_trans_matrix() const
	{
		return this->get_handled().get_trans_matrix();
	}

	UserDataType& get_user_data()
	{
		return this->get_handled().get_user_data();
	}

	const UserDataType& get_user_data() const
	{
		return this->get_handled().get_user_data();
	}

protected:

	c()
	{
	}

	static void release(handled_type* ins)
	{
		release_instance3d(ins);
	}

	friend class gpuppur::cpu_raytracer<UserDataType>;
};
};

}	// end of namespace detail
}	// end of namespace cpu_raytracers
}	// end of namespace gpuppur

#endif

