#ifndef GPUPPUR_CPU_MESH_HPP
#define GPUPPUR_CPU_MESH_HPP

/**
 *	@file	
 *	@brief	Mesh class of CPU Raytracer class
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <vector>
#include <limits>
#include <list>
#include <boost/array.hpp>
#include <boost/mpl/if.hpp>
#include <boost/mpl/int.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/3dmath/triangle.hpp>
#include <gpuppur/3dmath/axis_aligned_bounding_box.hpp>

namespace gpuppur
{
namespace cpu_raytracers
{
namespace detail
{

/**
 *	@brief	Mesh of CPU Raytracer class
 *
 */
template<class Parent>
class base_mesh
{
public:

	typedef	base_mesh<Parent>						this_type;

private:

	void common_init()
	{
		this->mesh_aabb = get_axis_aligned_bounding_box
		(
			vertices.begin(),
			vertices.end()
		);

	//	std::cout << "max=" << max << std::endl;
	//	std::cout << "min=" << min << std::endl;
	}

	Parent& get_this()
	{
		return *static_cast<Parent*>(this);
	}

	const Parent& get_this() const
	{
		return *static_cast<const Parent*>(this);
	}

protected:

	const vector3& get_vertex_from_index_of_index
	(
		boost::mpl::int_<2>,
		std::size_t i
	) const
	{
		return this->vertices[this->triangles_16bits[i]];
	}

	const vector3& get_vertex_from_index_of_index
	(
		boost::mpl::int_<4>,
		std::size_t i
	) const
	{
		return this->vertices[this->triangles_32bits[i]];
	}

	const std::size_t get_num_indices(boost::mpl::int_<2>) const
	{
		return this->triangles_16bits.size();
	}

	const std::size_t get_num_indices(boost::mpl::int_<4>) const
	{
		return this->triangles_32bits.size();
	}

	const std::size_t get_num_indices() const
	{
			return this->bytes_per_index==4
		?
			this->get_num_indices(boost::mpl::int_<4>())
		:
			this->get_num_indices(boost::mpl::int_<2>());
	}

	const gpuppur::aabb<vector3>&	get_aabb() const
	{
		return this->mesh_aabb;
	}

	const int get_bytes_per_index() const
	{
		return this->bytes_per_index;
	}

public:

	base_mesh()
	{
	}

	base_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned short>&	triangles
	):
		vertices(vertices),
		triangles_16bits(triangles),
		bytes_per_index(2)
	{
		this->common_init();
	}

	base_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned int>&	triangles
	):
		vertices(vertices),
		triangles_32bits(triangles),
		bytes_per_index(4)
	{
		this->common_init();
	}

	bool cast_ray
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal
	) const
	{
		if(this->bytes_per_index==4)
		{
			return get_this().cast_ray_impl
			(
				casted_ray, max_t, pos, normal, boost::mpl::int_<4>()
			);
		}else if(this->bytes_per_index==2)
		{
			return get_this().cast_ray_impl
			(
				casted_ray, max_t, pos, normal, boost::mpl::int_<2>()
			);
		}

		assert(false && "You are using invaild bit index");
		return false;
	}

private:

	std::vector<vector3>			vertices;
	std::vector<unsigned short>		triangles_16bits;
	std::vector<unsigned int>		triangles_32bits;
	int								bytes_per_index;

	gpuppur::aabb<vector3>			mesh_aabb;

};	// end of class base_mesh

class simple_mesh : public base_mesh<simple_mesh>
{
public:

	typedef base_mesh<simple_mesh>	base;
	typedef simple_mesh				this_type;

private:

	template<int BytesPerIndex>
	bool cast_ray_impl
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal,
		boost::mpl::int_<BytesPerIndex>
	) const
	{
	#if 1
		boost::tuple<float, float, int, int>
	//	auto	//If you are using compiler	support C++0x
		ret	= cast_ray_to_aabb
		(
			base::get_aabb(),
			casted_ray
		);

		if(boost::get<2>(ret) == -1 && boost::get<3>(ret) == -1)
		{
			return false;
		}
	#endif

		float min_t = std::numeric_limits<float>::max();
		vector3 tmp_pos, tmp_normal;
		std::size_t i=0,
					num_indices
					=
					base::get_num_indices(boost::mpl::int_<BytesPerIndex>());

		for(i=0; i<num_indices; i+=3)
		{
			float t =
			gpuppur::triangle<vector3>::get_cross
			(
				casted_ray.getPos(),
				casted_ray.getDir(),
				base::get_vertex_from_index_of_index
				(
					boost::mpl::int_<BytesPerIndex>(), i+0
				),
				base::get_vertex_from_index_of_index
				(
					boost::mpl::int_<BytesPerIndex>(), i+1
				),
				base::get_vertex_from_index_of_index
				(
					boost::mpl::int_<BytesPerIndex>(), i+2
				),
				tmp_pos, tmp_normal
			);

			if(t < 0.0f || t > max_t || t >= min_t)
			{
				continue;
			}

			min_t = t;
			pos = tmp_pos;
			normal = tmp_normal;
		}

		return min_t <= max_t ? true : false;
	}


public:

	simple_mesh()
	{
	}

	simple_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned short>&	triangles
	):
		base(vertices, triangles)
	{
	}

	simple_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned int>&	triangles
	):
		base(vertices, triangles)
	{
	}

	friend class base;
};

class octree_mesh : public base_mesh<octree_mesh>
{
private:
/*
	divide_to_8(const aabb<vector3>& box, const std::vector<unsigned int>& triangles)
	{
		std::size_t num_triangles = triangles.size();

		for(std::size_t i=0; i<num_triangles; ++i)
		{
		}
	}
*/
	template<int BytesPerIndex>
	void divide_mesh
	(
		const std::vector<unsigned int>& triangles,
		const std::size_t octree_index,
		const std::size_t level,
		const std::size_t& num_level
	)
	{
		if(level == num_level)
		{
			this->leafs.push_back(triangles);
			return;
		}

		const std::size_t child_id
		=
		this->each_level_begins[level+1]+octree_index*8;

		get_uniform_divided_aabbs
		(
			this->aabb_octree[this->each_level_begins[level]+octree_index],
			vector3(2.0f, 2.0f, 2.0f),
			this->aabb_octree.begin()+child_id
		);

		for(std::size_t i=0; i<8; ++i)
		{
			std::vector<unsigned int> inner_meshes;
			std::size_t num_triangles = triangles.size();
			for(std::size_t j=0; j<num_triangles; ++j)
			{
				for(int k=0; k<3; ++k)
				{
					if
					(
						is_in
						(
							this->aabb_octree[child_id+i],
							base::get_vertex_from_index_of_index
							(
								boost::mpl::int_<BytesPerIndex>(),
								triangles[j]+k
							)
						)
					)
					{
						inner_meshes.push_back(j);
						break;
					}
				}
			}

			this->divide_mesh<BytesPerIndex>
			(
				inner_meshes, child_id+i, level+1, num_level
			);
		}
	}

	template<int BytesPerIndex>
	void divide_root_mesh(std::size_t num_level)
	{
		const std::size_t num_indices = base::get_num_indices();

		get_uniform_divided_aabbs
		(
			base::get_aabb(),
			vector3(2.0f, 2.0f, 2.0f),
			this->aabb_octree.begin()
		);

		for(std::size_t i=0; i<8; ++i)
		{
			std::vector<unsigned int> inner_meshes;
			for(std::size_t j=0; j<num_indices; j+=3)
			{
				if
				(
					is_in
					(
						this->aabb_octree[i],
						base::get_vertex_from_index_of_index
						(
							boost::mpl::int_<BytesPerIndex>(),
							j+0
						)
					)
					||
					is_in
					(
						this->aabb_octree[i],
						base::get_vertex_from_index_of_index
						(
							boost::mpl::int_<BytesPerIndex>(),
							j+1
						)
					)
					||
					is_in
					(
						this->aabb_octree[i],
						base::get_vertex_from_index_of_index
						(
							boost::mpl::int_<BytesPerIndex>(),
							j+2
						)
					)
				)
				{
					inner_meshes.push_back(j);
				}
			}

			this->divide_mesh<BytesPerIndex>(inner_meshes, i, 1, num_level);
		}
	}

	void gen_octree()
	{
		const std::size_t num_level = 2;
		assert(num_level < 32/3);
		const std::size_t num_last_level_aabb = 1<<(num_level*3); //8^num_level==2^(num_level*3);
		this->aabb_octree.resize((num_last_level_aabb<<1-1)&0x49249248);
		this->each_level_begins.resize(num_level);

		std::size_t begin = 0;
		for(std::size_t i=0; i<num_level; ++i)
		{
			this->each_level_begins[i] = begin;
			begin += 1<<((i+1)*3);
		}

		base::get_bytes_per_index()==4
		?
		this->divide_root_mesh<4>(num_level) : this->divide_root_mesh<2>(num_level);
	}

	
	template<int BytesPerIndex>
	bool cast_ray_to_leaf
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal,
		std::size_t leaf_index,
		boost::mpl::int_<BytesPerIndex>
	)
	{
	}

	template<int BytesPerIndex>
	bool cast_ray_to_octree
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal,
		const std::size_t octree_index,
		const std::size_t level,
		boost::mpl::int_<BytesPerIndex>
	)
	{
		struct local_box
		{
			std::size_t index;
			float		t_min;

			local_box(std::size_t index, float t_min):
				index(index), t_min(t_min)
			{
			}

			bool operator < (const local_box& rhs) const
			{
				return this->t_min < rhs.t_min;
			}
		};

		const std::size_t child_id
		=
		this->each_level_begins[level+1]+octree_index*8;

		std::size_t	num_crossed = 0;
		boost::array<local_box, 8> local_boxes;
		for(std::size_t i=0; i<8; ++i)
		{
			std::size_t crnt_id = child_id+i;
			boost::tuple<float, float, int, int>
		//	auto	//If you are using compiler	support C++0x
			ret	= cast_ray_to_aabb
			(
				this->aabb_octree[crnt_id],
				casted_ray
			);

			if(boost::get<2>(ret) == -1 && boost::get<3>(ret) == -1)
			{
				if
				(
					is_in(this->aabb_octree[crnt_id], ray.getPos)
					&&
					is_in(this->aabb_octree[crnt_id], ray.getPos(max_t))
				)
				{
					return 
						level == this->each_level_begins.size()-1 ? 
						cast_ray_to_leaf
						(
							casted_ray, max_t, pos, normal, crnt_id,
							boost::mpl::int_<BytesPerIndex>>()
						)
						:
						cast_ray_to_octree
						(
							casted_ray, max_t, pos, normal, crnt_id,
							boost::mpl::int_<BytesPerIndex>>()
						);
				}

				continue;
			}else if(boost::get<3>(ret) != -1)
			{
				local_boxes[num_crossed++] = local_box(crnt_id, 0.0f);

				continue;
			}

			local_boxes[num_crossed++] = local_box(crnt_id, boost::get<0>(ret));
		}

		std::sort(local_boxes.begin(), local_boxes.begin()+num_crossed);

		if(level == this->each_level_begins.size()-1)
		{
			for(std::size_t i=0; i<num_crossed; ++i)
			{
				if
				(
					this->cast_ray_to_leaf
					(
						casted_ray, max_t, pos, normal, local_boxes[i].index,
						boost::mpl::int_<BytesPerIndex>()
					)
				)
				{
					local_boxes[i]
				}
			}
		}

		for(std::size_t i=0; i<num_crossed; ++i)
		{
			if
			(
				cast_ray_to_octree
				(
					casted_ray, max_t, pos, normal, local_boxes[i].index,
					boost::mpl::int_<BytesPerIndex>()
				)
			)
			{
				return true;
			}
		}

		return false;
	}

	template<int BytesPerIndex>
	bool cast_ray_impl
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal,
		boost::mpl::int_<BytesPerIndex>
	) const
	{
	#if 1
		boost::tuple<float, float, int, int>
	//	auto	//If you are using compiler	support C++0x
		ret	= cast_ray_to_aabb
		(
			base::get_aabb(),
			casted_ray
		);

		if(boost::get<2>(ret) == -1 && boost::get<3>(ret) == -1)
		{
			return false;
		}
	#endif

	}

public:

	typedef base_mesh<octree_mesh>	base;

	octree_mesh()
	{
	}

	octree_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned short>&	triangles
	):
		base(vertices, triangles)
	{
		this->gen_octree();
	}

	octree_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned int>&	triangles
	):
		base(vertices, triangles)
	{
		this->gen_octree();
	}

private:

	std::vector<aabb<vector3> >					aabb_octree;
	std::vector<std::size_t>					each_level_begins;
	std::vector<std::vector<unsigned int> >		leafs;
};

#if 0
	typedef simple_mesh using_mesh;
#else
	typedef octree_mesh using_mesh;
#endif

class mesh : public using_mesh
{
public:

	typedef using_mesh	base;
	typedef mesh		this_type;
	typedef std::list<this_type>::iterator	iterator;

	mesh()
	{
	}

	mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned short>&	triangles
	):
		base(vertices, triangles)
	{
	}

	mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned int>&	triangles
	):
		base(vertices, triangles)
	{
	}

	void set_self_iterator(iterator itr)
	{
		this->self = itr;
	}

	const iterator& get_self_iterator() const
	{
		return this->self;
	}

private:

	iterator		self;		//!Iterator to std::list<this class> for fast erase object from pointer to this class.
};

template<class Parent, class Base>
class handler_to_mesh : public Base
{
public:

	typedef Base						base;
	typedef gpuppur::cpu_raytracers::detail::mesh
										handled_type;
	typedef gpuppur::mesh_virtual		virtual_type;
	typedef gpuppur::mesh_generic		generic_type;
	typedef gpuppur::cpu_raytracers::detail::cpu_raytracer_implement
										friend_type;
public:

	handled_type& get_handled()
	{
		return reinterpret_cast<Parent*>(this)->get_handle();
	}

	const handled_type& get_handled() const
	{
		return reinterpret_cast<const Parent*>(this)->get_handle();
	}

protected:

	handler_to_mesh()
	{
	}

	static void release(handled_type* mesh)
	{
		gpuppur::cpu_raytracers::detail::release_mesh(mesh);
	}
};

}	// end of namespace detail
}	// end of namespace cpu_raytracers
}	// end of namespace gpuppur

#endif

