#ifndef GPUPPUR_GPUPPUR_HPP
#define GPUPPUR_GPUPPUR_HPP

/**
 *	@file	
 *	@brief	
 *	@author	Tomohiro Matsumoto
 *
 *	@mainpage
 *	GPUPPUR is realtime 3D rendering library.
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <cassert>
#include <string>
#include <vector>
#include <boost/call_traits.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/mpl/void.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include "handle_generic.hpp"
#include "math.hpp"

namespace gpuppur
{

struct material
{
	material()
	{
	}

	material(const vector3& diffuse):
		diffuse(diffuse)
	{
	}

	vector3	diffuse;
};

template<class Base, typename Context>
class gpuppur_common_base : public Base
{
public:

	typedef Base		base;
	typedef Context		context_type;
	typedef typename boost::call_traits<context_type>::param_type
		context_param_type;

protected:

	gpuppur_common_base():
		camera_pos(0.0f, 0.0f, 0.0f),
		camera_front_dir(0.0f, 0.0f, -1.0f),
		camera_up_dir(0.0f, 1.0f, 0.0f)
	{
	}

	~gpuppur_common_base()
	{
	}

	void uninitialize()
	{
		base::uninitialize();
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		context_param_type context
	)
	{
		if(!base::initialize(width, height))
		{
			return false;
		}

		assert(width > 0);
		assert(height > 0);

		this->width = width;
		this->height = height;

		this->context = context;

		return true;
	}

public:
	void look_at
	(
		const vector3& eye_pos,
		const vector3& front_dir,
		const vector3& up_dir
	)
	{
		this->camera_pos = eye_pos;
		this->camera_front_dir = front_dir;
		this->camera_up_dir = up_dir;
	}

	void set_context(context_param_type	context)
	{
		this->context = context;
	}

	context_type&	get_context()
	{
		return this->context;
	}

	const context_type& get_context() const
	{
		return this->context;
	}

protected:

	std::size_t		width;
	std::size_t		height;

	vector3			camera_pos;
	vector3			camera_front_dir;
	vector3			camera_up_dir;

	context_type	context;
};

template<template<class> class Implement/*=gpuppuray_default_implement_with_opengl::c*/>
class GPUPPUR
{
public:

	struct dummy : public boost::mpl::void_
	{
		typedef void* context_param_type;
	};

	typedef typename Implement<dummy>::context_type	context_type;

template<class Base>
class c : public Implement<gpuppur_common_base<Base, context_type> >
{
	typedef Implement<gpuppur_common_base<Base, context_type> > base;

public:

//	typedef typename base::mesh			mesh;
//	typedef typename base::instance3d	instance3d;

	c()
	{
	}

	~c()
	{
		this->uninitialize();
	}

	void uninitialize()
	{
		if(!this->is_initialized)
		{
			return;
		}

		base::uninitialize();
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		typename base::context_param_type context
	)
	{
		this->uninitialize();

		if(!base::initialize(width, height, context))
		{
			return this->failed();
		}

		return true;
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		void*		context
	)
	{
		this->uninitialize();

		if(!base::initialize(width, height, context))
		{
			return this->failed();
		}

		return true;
	}

	void process()
	{
		return base::process();
	}

	using base::create_mesh;

	typename base::mesh create_mesh(const std::string& filename)
	{
		boost::filesystem::path file_path(filename, boost::filesystem::native);

		const std::string extension(boost::filesystem::extension(file_path));

		if(extension == ".obj")
		{
			return base::create_mesh_from_wavefront(filename);
		}else if(extension == ".cooked")
		{
			return base::create_mesh_from_cooked(filename);
		}else
		{
			return typename base::mesh();
		}
	}
};
};

class gpuppur_virtual
{
public:

	typedef gpuppur::instance3d_generic	instance3d;
	typedef	gpuppur::mesh_generic		mesh;
	template<class Group>
	struct texture
	{
		typedef gpuppur::texture_generic<Group> type;
	};

	virtual ~gpuppur_virtual(){};

	virtual void uninitialize() = 0;
	virtual bool initialize(std::size_t width, std::size_t height, void* context) = 0;
	virtual void process() = 0;
	virtual void look_at
	(const vector3& eye_pos, const vector3& front_dir, const vector3& up_dir) = 0;
	virtual void set_viewport(int x, int y, int width, int height) = 0;
	virtual void set_projection
	(
		float left,
		float right,
		float bottom,
		float top,
		float neear,
		float faar,
		bool  is_perspective=true
	) = 0;

	//implemented in GPUPPUR.
	virtual	gpuppur::mesh_generic
	create_mesh(const std::string& filename) = 0;

	virtual	gpuppur::mesh_generic
	create_mesh
	(const std::vector<vector3>& vertices,
	 const std::vector<unsigned short>& triangles) = 0;

	virtual gpuppur::mesh_generic
	create_mesh_from_wavefront(const std::string& filename) = 0;

	virtual gpuppur::mesh_generic
	create_mesh_from_cooked(const std::string& filename) = 0;

	virtual gpuppur::instance3d_generic
	create_instance
	(
		gpuppur::mesh_generic handle,
		const vector3& position,
		const gpuppur::material&
	) = 0;
};



}	//end of namespace gpuppur

#endif
