#ifndef GPUPPUR_GPUPPURAS_DEFAULT_IMPLEMENT
#define GPUPPUR_GPUPPURAS_DEFAULT_IMPLEMENT

/**
 *	@file	
 *	@brief	This is default implementation of GPUPPURas.
 *	@author	Tomohiro Matsumoto
 *
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <boost/function.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/tuple/tuple.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/shader/glsl.hpp>
#include <gpuppur/utility/lens.hpp>
#include "gpuppuras.hpp"
#include "ppu/nxphysics.hpp"
#include "handler.hpp"
#include <gpuppur/mesh/mesh_opengl.hpp>
#include <gpuppur/mesh/mesh_directx.hpp>

namespace gpuppur
{

template<class MeshRenderer=gpuppur::mesh_opengl>
class gpuppuras_default_implement
{
public:
	typedef
		gpuppuras_default_implement<MeshRenderer>
		this_type;

/**
 *	@brief Default implementation of GPUPPURas.
 *
 *	@todo Refactor for more flexibility, reusability, convenience and simplicity with keeping speed.
 *
 */
template<class Base>
class c : public Base
{
public:
	typedef Base
		base;
	typedef typename this_type::template c<Base>
		this_type;
	const static bool
		is_virtual = boost::is_polymorphic<Base>::value;

	class mesh_state;
private:

	typedef boost::tuple<mesh_state*, gpuppur::material>	ins_data;
	typedef physics<ins_data>								raytracer;

public:

	/**
	 *	@brief	Rendering mesh class.
	 *
	 */
	class mesh_state
	{
	public:

		typedef MeshRenderer	mesh_renderer;

	private:
#if 0
		mesh_state
		(
			const raytracer::mesh_tmp&			data,
			const std::vector<vector3>&			vertices,
			const std::vector<unsigned short>&	triangles,
			context_type						context
		):
			mesh(data),
			renderer(vertices, triangles)
		{
		}
#endif
		mesh_state
		(
			const typename raytracer::mesh_tmp		data,
			typename c<Base>::context_type			context,
			typename
			MeshRenderer::sub_context_param_type	sub_context
		):
			mesh(data),
			renderer(mesh, context, sub_context)
		{
		}

	public:

		MeshRenderer&	get_mesh_renderer()
		{
			return this->renderer;
		}

		const MeshRenderer&	get_mesh_renderer() const
		{
			return this->renderer;
		}

		bool get_has_mesh() const
		{
			return this->renderer.get_has_mesh();
		}

	protected:

		typename raytracer::mesh_alone	mesh;
		mesh_renderer					renderer;

		friend	class c<Base>;
	};

	/**
	 *	@brief ̃NX(gpuppuras_default_implement)Ń_OmeshQƂ
	 *	handlerNXƂȂev[gNXւ̃p[^ƂȂev[gNXB
	 */
	template<class Parent, class Base2>
	class mesh_implement : public Base2
	{
	public:

		typedef Base2								base;
		typedef mesh_state							handled_type;
		typedef gpuppur::mesh_virtual				virtual_type;
		typedef gpuppur::mesh_generic				generic_type;
		typedef typename c<Base>::this_type			friend_type;

		mesh_implement()
		{
		}

	protected:

		mesh_implement(const handled_type& handler):
			base(handler)
		{
		}

		mesh_implement(const boost::shared_ptr<handled_type>& handler):
			base(handler)
		{
		}

		static void release(handled_type* mesh_handle)
		{
			delete mesh_handle;
		}
	};

	typedef typename gpuppur::handler_tmpl<mesh_implement>::c_virtual
		mesh_virtual;
	typedef typename gpuppur::handler_tmpl<mesh_implement>::c_static
		mesh_static;
	typedef typename gpuppur::handler_tmpl<mesh_implement>::c_tmp
		mesh_tmp;

	typedef
		typename boost::mpl::if_c
		<
			is_virtual,
			gpuppur::instance3d_generic,
			typename raytracer::instance3d_static
		>::type
		instance3d;

	typedef
		typename boost::mpl::if_c
		<
			is_virtual,
			gpuppur::mesh_generic,
			typename this_type::mesh_static
		>::type
		mesh;

protected:

	c()
	{
	}

	~c()
	{
	}

	void uninitialize()
	{
		this->ppu.uninitialize();
		this->sub_context.uninitialize();
		base::uninitialize();
	}

	template<typename Context>
	bool initialize(std::size_t width, std::size_t height, Context context)
	{
		if(!base::initialize(width, height, context))
		{
			return this->failed();
		}

		if
		(
			!this->sub_context.initialize
			(
				this->context,
				std::string("../data/shader_ras/vert_shader"),
				std::string("../data/shader_ras/frag_shader")
			)
		)
		{
			return this->failed();
		}

		if(!this->ppu.initialize())
		{
			fprintf(stderr, "Failed to initialize physics.\n");
			return this->failed();
		}

		mesh_state::mesh_renderer::set_projection
		(
			this->context,
			this->sub_context,
			this->lens
		);

		return true;
	}

private:
	
	template<class MeshItr>
	void draw_mesh_instance_map(MeshItr first, MeshItr last)
	{
		mesh_state::mesh_renderer::set_state
		(
			this->camera_pos,
			this->camera_front_dir,
			this->camera_up_dir,
			this->context,
			this->sub_context
		);

		typedef typename MeshItr::value_type::second_type::iterator instance_itr;

		MeshItr i = first;
		for(;i!=last; ++i)
		{
			i->first->get_mesh_renderer().bind_mesh(this->context, this->sub_context);

			instance_itr j=i->second.begin();
			for(;j!=i->second.end(); ++j)
			{
				matrix4x4 mat = j->get_trans_matrix();
				const ins_data* tmp
				= &j->get_user_data();
				i->first->get_mesh_renderer().draw
				(
					mat,
					boost::tuples::get<1>(*tmp).diffuse,
					this->context,
					this->sub_context
				);
			}
		}
	}

	void render_only_visible_instance()
	{
		this->ppu.process();

		vector3 camera_side_dir
		=
		this->camera_front_dir.outerProduct(this->camera_up_dir);
		camera_side_dir.normalize();

		typedef std::set
		<
			typename raytracer::instance3d_tmp
		>	rendered_instance_set;
		typedef std::map
		<
			mesh_state*,
			rendered_instance_set
		>	mesh_instance_map;

		mesh_instance_map	rendered_meshes;

		int half_w = static_cast<int>(this->width/2);
		int half_h = static_cast<int>(this->height/2);
		for(int i=-half_h; i<half_h; ++i)
		for(int j=-half_w; j<half_w; ++j)
		{
			float x = (float)j/(this->width/2);
			float y = (float)i/(this->height/2);

			vector3 dir
			=
			//this->camera_front_dir + camera_side_dir*x + this->camera_up_dir*y;
			this->lens.get_ray_dir
			(
				this->camera_front_dir,
				camera_side_dir,
				this->camera_up_dir,
				x,
				y
			);

			float dir_dot_front = dir.innerProduct(this->camera_front_dir);
			float min_t = this->lens.get_near()/dir_dot_front;
			float max_t = this->lens.get_far()/dir_dot_front;

			ray r
			(
				this->camera_pos
				+
				this->lens.get_ray_local_pos
				(
					this->camera_front_dir,
					camera_side_dir,
					this->camera_up_dir,
					x,
					y
				)
				+
				dir*min_t,	//Behave like near cliping plane by moving ray origin.
				dir
			);

		/*
			boost::tuple<raytracer::instance3d_tmp, void*>
			cross_mesh(this->ppu.cast_ray(r, max_t));

			if(!cross_mesh.get<1>())
			{
				continue;
			}
		*/
			typename raytracer::instance3d_tmp ins(this->ppu.cast_ray(r, max_t));

			if(!ins)
			{
				continue;
			}

			//	ins is instance of mesh hit the ray. And ms is mesh of that instance.
			//	mtrl is material assigned to instance.
		//	raytracer::instance3d_tmp& ins = cross_mesh.get<0>();

			ins_data* tmp
			= &ins.get_user_data();

			mesh_state* ms = boost::get<0>(*tmp);

			typename mesh_instance_map::iterator i
			=
			rendered_meshes.find(ms);
			if(i==rendered_meshes.end())
			{
				rendered_instance_set ins_set;
				ins_set.insert(ins);
				rendered_meshes.insert
				(
					typename mesh_instance_map::value_type(ms, ins_set)
				);

				continue;
			}

			if(i->second.find(ins) != i->second.end())
			{
				continue;
			}

			i->second.insert(ins);
		}

		this->draw_mesh_instance_map(rendered_meshes.begin(), rendered_meshes.end());
	}

	void draw_all_instance()
	{
		typedef
		std::vector<typename raytracer::instance3d_tmp>
		instance_vec;

		instance_vec ins_list = ppu.get_all_instances();

		//realign data.
		typedef
		std::map
		<
			mesh_state*,
			std::vector<typename raytracer::instance3d_tmp>
		>	mesh_instance_map;
		typedef typename mesh_instance_map::iterator mesh_instance_map_itr;

		mesh_instance_map rendered_meshes;

		for(std::size_t i = 0; i<ins_list.size(); ++i)
		{
			mesh_state* ms = boost::get<0>(ins_list[i].get_user_data());
			mesh_instance_map_itr j
			=
			rendered_meshes.find(ms);

			if(j==rendered_meshes.end())
			{
				rendered_meshes.insert
				(
					typename mesh_instance_map::value_type
					(
						ms, instance_vec(1, ins_list[i])
					)
				);
			}else
			{
				j->second.push_back(ins_list[i]);
			}
		}

		this->draw_mesh_instance_map(rendered_meshes.begin(), rendered_meshes.end());
	}

protected:

	/**
	 *	Render all 3D objects.
	 */
	void process()
	{
	//	this->render_only_visible_instance();
		this->draw_all_instance();
	}

public:

	mesh create_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned short>&	triangles
	)
	{
		std::auto_ptr<mesh_state> tmp
		(
			new mesh_state
			(
				this->ppu.load_mesh(vertices, triangles),
				this->context,
				this->sub_context
			)
		);

		if(!tmp->get_has_mesh())
		{
			return mesh();
		}

		return mesh
		(
			mesh_tmp(*tmp.release())
		);
	}

	/**
	 *	Load mesh from wave front format file.
	 *	Only vertex and face data is used.
	*/
	mesh create_mesh_from_wavefront(const std::string& filename)
	{
		std::auto_ptr<mesh_state> tmp
		(
			new mesh_state
			(
				this->ppu.load_mesh_from_wavefront(filename),
				this->context,
				this->sub_context
			)
		);

		if(!tmp->get_has_mesh())
		{
			return mesh();
		}

		return mesh
		(
			mesh_tmp(*tmp.release())
		);
	}

	/** Load cooked mesh.
	 *
	 *	Mesh is needed to be preprocessed before using with PhysX.
	 *	Cooked mesh is preprocessed mesh by Cooking API in PhysX.
	 *
	 *	@param[in]	filename	File name of cooked mesh file.
	 *	@return		Handle to created mesh.
	 */
	mesh create_mesh_from_cooked(const std::string& filename)
	{
		std::auto_ptr<mesh_state> tmp
		(
			new mesh_state
			(
				this->ppu.load_mesh_from_cooked(filename),
				this->context,
				this->sub_context
			)
		);

		if(!tmp->get_has_mesh())
		{
			return mesh();
		}

		return mesh
		(
			mesh_tmp(*tmp.release())
		);
	}

	/** Create instance of mesh.
	 *
	 *	If you want to render mesh, you have to load mesh from this->create_mesh_xxx
	 *	and pass return value to this method.
	 *	You can create some instances that share 3d mesh by passing same mesh to this method.
	*/
	instance3d create_instance
	(
		mesh						handle,
		const vector3&				position,
		const gpuppur::material&	mtrl
	)
	{
		mesh_tmp tmp(handle);
		return instance3d
		(
			this->ppu.create_mesh
			(
				tmp.handle->mesh,
				position,
				boost::tuple<mesh_state*, gpuppur::material>(tmp.handle, mtrl)
			)
		);
	}

	raytracer& get_ppu()
	{
		return this->ppu;
	}

	void set_viewport(int x, int y, int width, int height)
	{
		mesh_state::mesh_renderer::set_viewport
		(
			x,
			y,
			width,
			height,
			this->context,
			this->sub_context
		);
	}

	/**
	 *	@brief Set view frustum
	 *
	 *	Set parameter of view frustum like glFrustum or glOrtho.
	 *	You can set both perspective and parallel projection.
	 *	'near' and 'far' is reversed word in Visual C++.
	 *
	 *	@param[in] left		X coordinate for the left	side of near	clipping plane.
	 *	@param[in] right	X coordinate for the right	side of near	clipping plane.
 	 *	@param[in] bottom	Y coordinate for the bottom	side of near	clipping plane.
	 *	@param[in] top		Y coordinate for the top	side of near	clipping plane.
	 *	@param[in] neear	Distance from eye to				near	clipping plane.
	 *	@param[in] faar		Distance from eye to				far		clipping plane.
	 *	@param[in] is_perspective	Specify whether perspective or parallel projection.
	 */
	void set_projection
	(
		float left,
		float right,
		float bottom,
		float top,
		float neear,
		float faar,
		bool  is_perspective=true
	)
	{
		this->lens = gpuppur::lens<>
		(
			left,
			right,
			bottom,
			top,
			neear,
			faar,
			is_perspective
		);

		mesh_state::mesh_renderer::set_projection
		(
			this->context,
			this->sub_context,
			this->lens
		);
	}

protected:

	raytracer						ppu;
	typename 
	MeshRenderer::sub_context_type	sub_context;
	gpuppur::lens<>					lens;
};
};


class gpuppuras_default_implement_with_opengl
{
public:
	typedef gpuppuras_default_implement<> def_impl;
template<class Base>
class c : public def_impl::c<Base>
{
public:

	typedef def_impl::c<Base>	base;
	typedef boost::mpl::void_	context_type;

	using base::initialize;

	bool initialize(std::size_t width, std::size_t height, void*)
	{
		return base::initialize(width, height, boost::mpl::void_());
	}
};
};


#ifndef D3D_SDK_VERSION

class gpuppuras_default_implement_with_directx
{
public:
	typedef gpuppuras_default_implement<mesh_directx> def_impl;

	template<class Base>
	struct c : public def_impl::c<Base>
	{
		typedef void* context_type;
	};
};

#else

class gpuppuras_default_implement_with_directx
{
public:
	typedef gpuppuras_default_implement<mesh_directx> def_impl;
template<class Base>
class c : public def_impl::c<Base>
{
public:

	typedef def_impl::c<Base>	base;
	typedef IDirect3DDevice9*	context_type;

	using base::initialize;

	bool initialize(std::size_t width, std::size_t height, void* context)
	{
		return base::initialize
		(
			width,
			height,
			reinterpret_cast<IDirect3DDevice9*>(context)
		);
	}
};
};

#endif

typedef gpuppur::GPUPPUR
		<
			gpuppur::gpuppuras
			<
				gpuppuras_default_implement_with_opengl::c
			>::c
		>::c<gpuppur::tail<gpuppur::gpuppuras_virtual> >
		gl_rasterizer;
typedef gl_rasterizer gl_rasterizer_virtual;

typedef gpuppur::GPUPPUR
		<
			gpuppur::gpuppuras
			<
				gpuppuras_default_implement_with_opengl::c
			>::c
		>::c<gpuppur::tail<boost::mpl::void_> >
		gl_rasterizer_static;

typedef gpuppur::GPUPPUR
		<
			gpuppur::gpuppuras
			<
				gpuppuras_default_implement_with_directx::c
			>::c
		>::c
		<
			gpuppur::tail
			<
				gpuppur::gpuppuras_virtual
			>
		>
		dx_rasterizer;
typedef dx_rasterizer dx_rasterizer_virtual;

typedef gpuppur::GPUPPUR
		<
			gpuppur::gpuppuray
			<
				gpuppuras_default_implement_with_directx::c
			>::c
		>::c
		<
			gpuppur::tail
			<
				boost::mpl::void_
			>
		>
		dx_rasterizer_static;

}	// end of namespace gpuppur

#endif
