#ifndef GPUPPUR_HANDLE_GENERIC_HPP
#define GPUPPUR_HANDLE_GENERIC_HPP

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <boost/type_traits.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/utility/enable_if.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include "math.hpp"

namespace gpuppur
{

/**
 *	@brief Base class to implement generic handle.
 *
 */
template<class VirtualType>
class handle_generic
{
public:

	typedef VirtualType	virtual_type;

	handle_generic()
	{
	}

	handle_generic(const handle_generic& rhs):
		impl(rhs.impl)
	{
	}

	template<class HandleImpl>
    handle_generic
	(
		const HandleImpl& handle_impl,
		typename boost::enable_if_c
		<
			boost::is_base_and_derived
			<
				virtual_type,
				typename	HandleImpl::virtual_type
			>::value
			&&
			boost::is_same
			<
				HandleImpl,
				typename	HandleImpl::tmp_type
			>::value
		>::type* = 0
	)
	{
		if(!handle_impl)
		{
			return;
		}

		typename HandleImpl::virtual_type::used_handle_map_type::iterator i
		=
		HandleImpl::virtual_type::get_used_handle_map().find(handle_impl);

		if
		(
			i
			==
			HandleImpl::virtual_type::get_used_handle_map().end()
		)
		{
			this->impl = boost::shared_ptr<virtual_type>
			(
				new typename HandleImpl::virtual_type(handle_impl),
				&HandleImpl::virtual_type::release_used_by_shared_ptr
			);

			HandleImpl::virtual_type::get_used_handle_map().insert
			(
				std::pair
				<
					typename HandleImpl::tmp_type,
					boost::weak_ptr<virtual_type>
				>
				(
					handle_impl,
					this->impl
				)
			);

			return;
		}

		this->impl = boost::shared_ptr<virtual_type>(i->second);
	}

	~handle_generic()
	{
	}

	//Check whether this handle is valid.
	operator bool() const
	{
		return impl;
	}

	void release()
	{
		if(impl)
		{
			this->impl.reset();
		}
	}

	operator virtual_type*()
	{
		return this->impl.get();
	}

	operator const virtual_type*() const
	{
		return this->impl.get();
	}

protected:
	boost::shared_ptr<virtual_type> impl;
};


class mesh_virtual
{
public:

	virtual ~mesh_virtual(){};
};

class mesh_generic : public handle_generic<mesh_virtual>
{
public:

	typedef handle_generic<mesh_virtual>	base;

	template<class Mesh>
	mesh_generic
	(
		const Mesh& mesh
	):
		base(mesh)
	{
	}

	mesh_generic()
	{
	}
};

class instance3d_virtual
{
public:

	virtual ~instance3d_virtual()
	{
	}

///	virtual bool operator == (const instance3d_virtual) const = 0;
	virtual void set_position(const vector3& new_pos) = 0;
	virtual void set_orientation(const matrix3x3& mat) = 0;
	virtual const matrix4x4 get_trans_matrix() const = 0;
};

class instance3d_generic : public handle_generic<instance3d_virtual>
{
public:

	typedef handle_generic<instance3d_virtual> base;

	template<class Instance3d>
	explicit instance3d_generic
	(
		const Instance3d& instance3d
	):
		base(instance3d)
	{
	}

	instance3d_generic()
	{
	}

	void set_position(const vector3& new_pos)
	{
		assert(impl);
		return this->impl->set_position(new_pos);
	}


	void set_orientation(const matrix3x3& mat)
	{
		assert(impl);
		return this->impl->set_orientation(mat);
	}

	const matrix4x4 get_trans_matrix() const
	{
		assert(impl);
		return this->impl->get_trans_matrix();
	}
};


template<class Group>
class texture_virtual
{
public:

	virtual ~texture_virtual()
	{
	}

	virtual bool lock_for_write() = 0;
	virtual bool unlock() = 0;
	virtual void set(const Group& vec) = 0;
	virtual Group* get_buf_ptr() = 0;
};

template<class Group>
class texture_generic : public handle_generic<texture_virtual<Group> >
{
public:

	typedef handle_generic<texture_virtual<Group> > base;

	template<class Texture>
	texture_generic
	(
		const Texture& texture
	):
		base(texture)
	{
	}

	texture_generic()
	{
	}

	bool lock_for_write()
	{
		assert(this->impl);
		return this->impl->lock_for_write();
	}

	bool unlock()
	{
		assert(this->impl);
		return this->impl->unlock();
	}

	void set(const Group& vec)
	{
		assesrt(this->impl);
		return this->impl->set(vec);
	}

	Group* get_buf_ptr()
	{
		assesrt(this->impl);
		return this->impl->get_buf_ptr();
	}
};

}

#endif
