#ifndef GPUPPUR_HANDLER_HPP
#define GPUPPUR_HANDLER_HPP

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <boost/weak_ptr.hpp>
#include <boost/utility.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

namespace gpuppur
{
/**
 * 	Background:
 *	 There some objects which have resources(e.g. GPU has textures, PPU has triangle meshes).
 *	I call such objects "owner object".
 *	We get handler(reference, pointer) to these resources
 *	when we ask owner object to create resources.
 *	And many objects may have handler to resources.
 *	We use these handle to manipulate resources or to specify parameter of member function of owner object.
 *	 Resources may released(deleted, destroyed) only if there is no handle to it,
 *	or owner object is uninitialized.
 *	Releasing resources after owner object is uninitialized is fundamentally illegal.
 *
 *	This class has been born for help to implement such handler.
 *	You only need to implement functions which manipulate resource and release resource.
 *	Please refer gpuppur/ppu/instance3d.hpp or gpuppur/ppu/mesh.hpp.
 *	Resource will be released when last handler's (destructor or release function) is called.
 **/
template<template<class, class> class HandleImplement>
class handler_tmpl
{
public:
	typedef handler_tmpl<HandleImplement>
			this_type;
	typedef HandleImplement<boost::mpl::void_, boost::mpl::void_>
			independent_implement_type;

	template<class Base, template<class> class SmartPtr=gpuppur::nil>
	class handler_common_base :
	public HandleImplement
	<
		handler_common_base<Base, SmartPtr>,
		Base
	>
	{
	protected:

		typedef HandleImplement
				<
					handler_common_base<Base, SmartPtr>,
					Base
				>								base;
		typedef handler_common_base<base>		this_type;
	//	typedef HandleImplement<this_type>		implement_type;
		typedef typename base::handled_type		handled_type;
		const static bool						is_virtual
												= boost::is_polymorphic<Base>::value;

		const static bool 						is_smart_ptr
												=
		!boost::is_same<SmartPtr<void>, gpuppur::nil<void> >::value;

		typedef typename boost::mpl::if_c
		<
			!is_smart_ptr,
			handled_type*,
			SmartPtr<handled_type>
		>::type
		handler_type;

		handler_common_base():
			handle(handler_type())
		{
		}

		explicit handler_common_base(const handler_type& rhs):
			handle(rhs)
		{
		}

		handler_common_base
		(
			const handled_type& rhs
		):
			handle(&const_cast<handled_type&>(rhs))
		{
			BOOST_STATIC_ASSERT(!is_smart_ptr);
		}

		void set_raw_handle
		(
			const handler_type& src
		)
		{
			this->handle = src;
		}

		handler_type& get_raw_handle()
		{
			return this->handle;
		}

		const handler_type& get_raw_handle() const
		{
			return this->handle;
		}

	public:

		//Check whether this handle is valid.
		operator bool() const
		{
			return handle!=0;
		}

	protected:

		handled_type& get_handle()
		{
			return *this->handle;
		}

		const handled_type& get_handle() const
		{
			return *this->handle;
		}

	private:

		handler_type handle;

		friend
#ifdef _MSC_VER
		typename
#else
		class
#endif
		independent_implement_type::friend_type;

		friend
#ifndef _MSC_VER
		class	// I cant make friend if i write 'class' in visual stuio 2005(not sp1).
#endif
		HandleImplement
				<
					handler_common_base<Base, SmartPtr>,
					Base
				>;
	};

	class c_virtual;
	class c_static;
	class c_tmp;
	class c_alone;

	typedef c_virtual virtual_type;
	typedef c_static static_type;

	typedef handler_common_base
	<
		typename independent_implement_type::virtual_type
	>	virtual_base_c;

	/**
	 * Runtime Polymorphic handler.
	 * Instance of this class is pointed by gpuppur::handle_generic.
	 */
	class c_virtual : public virtual_base_c
	{
	public:

		typedef virtual_base_c			base;
		typedef c_virtual				this_type;
		typedef c_virtual				virtual_type;
		typedef c_static				static_type;
		typedef c_tmp					tmp_type;
		const static bool				is_virtual = true;

		typedef
			std::map<tmp_type, boost::weak_ptr<typename base::virtual_type> >
			used_handle_map_type;

		static used_handle_map_type& get_used_handle_map()
		{
			static used_handle_map_type uhm;

			return uhm;
		}


		c_virtual()
		{
		}

		c_virtual(const c_virtual& ins):
			base(ins.get_handle())
		{
		}

		c_virtual(const c_tmp ins):
			base(ins.get_handle())
		{
		}
#if 0
		bool operator == (const base::virtual_type* rhs) const
		{
			assert(typeid(*rhs).before(this_type));

			return this->handle
			==
			reinterpret_cast<this_type*>(rhs)->handle;
		}
#endif
	// private member function must be used by
	// only independent_implement_type::friend_type class.
	private:

		operator typename base::handled_type* ()
		{
			return base::handle;
		}

	public:

		~c_virtual()
		{
		//	this->release();
		}
#if 0
		void release()
		{
			base::release(this->handle);
		}
#endif
		static void release_used_by_shared_ptr(typename base::virtual_type* ptr)
		{
			assert(typeid(*ptr) == typeid(this_type));

			base::release(&reinterpret_cast<this_type*>(ptr)->get_handle());

			get_used_handle_map().erase(tmp_type(reinterpret_cast<this_type*>(ptr)->get_handle()));
			delete ptr;
		}

		friend class c_tmp;

		friend
#ifdef _MSC_VER
		typename
#else
		class
#endif
		independent_implement_type::friend_type;

	};


	typedef handler_common_base
	<
		boost::mpl::void_,
		boost::shared_ptr
	>	static_base_c;

	/** 
	 * static handler.
	 * You can use instance of this template class for handler type.
	 * And you can use instance of it for handling some objects.
	 * Handled object will be automatically releasead,
	 * if there are no handle pointing to that object.
	 * Because handled object is managed by boost::shared_ptr.
	 */
	class c_static : public static_base_c
	{
	public:
		typedef static_base_c		base;
		typedef c_static			this_type;
		typedef c_virtual			virtual_type;
		typedef c_static			static_type;
		typedef c_tmp				tmp_type;
		const static bool			is_virtual = false;

		typedef
			std::map<tmp_type, boost::weak_ptr<typename base::handled_type> >
			used_handle_map_type;

		static used_handle_map_type& get_used_handle_map()
		{
			static used_handle_map_type uhm;

			return uhm;
		}


		c_static():
			base()
		{
		}

		c_static(const c_static& ins):
			base(ins.get_raw_handle())
		{
		}

		c_static(tmp_type ins)
		{
			if(!ins)
			{
				return;
			}

			typename used_handle_map_type::iterator i
			=
			get_used_handle_map().find(ins);

			if(i == get_used_handle_map().end())
			{
				typename base::handler_type tmp_handler(&ins.get_handle(), &base::release);
				base::set_raw_handle(tmp_handler);

				get_used_handle_map()
				.insert
				(
					typename used_handle_map_type::value_type
					(
						ins,
						tmp_handler
					)
				);
			}else
			{
				base::set_raw_handle(typename base::handler_type(i->second));
			}
		}

		this_type& operator = (const this_type& rhs)
		{
			if(&rhs == this)
			{
				return *this;
			}

			if(this->get_raw_handle().unique())
			{
				get_used_handle_map().erase(*this->get_raw_handle().get());
			}

			this->set_raw_handle(rhs.get_raw_handle());

			return *this;
		}

	// private member function must be used by only friend class
	private:

		operator typename base::handled_type* ()
		{
			return base::handle.get();
		}

	public:

		~c_static()
		{
			this->release();
		}

		void release()
		{
			if(*this)
			{
				if(this->get_raw_handle().unique())
				{
					get_used_handle_map().erase(tmp_type(this->get_handle()));
				}

				this->get_raw_handle().reset();
			}
		}

		friend class c_tmp;

		friend
#ifdef _MSC_VER
		typename
#else
		class
#endif
		independent_implement_type::friend_type;
	};


	typedef handler_common_base
	<
		boost::mpl::void_
	>	tmp_base_c;

	/**
	 *	Class for temporary variable be copied by c_virtual or c_static.
	 *	Or handled as like weak_ptr to c_virtual or c_static.
	 *	Instance of this class never release handling object.
	 */
	class c_tmp : public tmp_base_c
	{
	public:

		typedef tmp_base_c			base;
		typedef c_virtual			virtual_type;
		typedef c_static			static_type;
		typedef c_tmp				this_type;
		typedef c_tmp				tmp_type;
		typedef typename base::handled_type
									handled_type;

		c_tmp(const c_tmp& ins):
			base(ins.get_handle())
		{
		}

		c_tmp(const c_static ins):
			base(ins.get_handle())
		{
		}
#if 0
		c_tmp(typename base::virtual_type* rhs):
			base((reinterpret_cast<c_static*>(rhs))->handle)
		{
			assert(typeid(rhs).before(typeid(c_static)));
		}
#endif

		c_tmp(const typename independent_implement_type::generic_type& rhs):
			//handle
			base
			(
				reinterpret_cast<const c_virtual*>
				(
					static_cast
					<
						const typename independent_implement_type::virtual_type*
					>(rhs)
				)->get_handle()
			)
		{
			assert
			(
				typeid
				(
					*static_cast
					<
						const typename independent_implement_type::virtual_type*
					>(rhs)
				) == (typeid(c_virtual))
			);
		}

		c_tmp(const c_alone& ins):
			/*handle*/base(ins.get_handle())
		{
		}

	// private member function must be used by only friend class
	private:

		c_tmp()
		{
		}


		c_tmp(const handled_type& handle):
			base(handle)
		{
		}

	/*
		operator const handled_type* () const
		{
			return &this->get_handle();
		}

		operator handled_type* ()
		{
			return &this->get_handle();
		}
	*/

	public:

		~c_tmp()
		{
		}

		bool operator < (const this_type& rhs) const
		{
			return &this->get_handle() < &rhs.get_handle();
		}

		//	template<class> friend class handler<HandleImplement>::c;
		friend class c_static;
		friend class c_virtual;
		friend class c_alone;

		friend
#ifdef _MSC_VER
		typename
#else
		class
#endif
		independent_implement_type::friend_type;

	protected:
	//		handled_type* handle;
	};

	typedef	handler_common_base
	<
		boost::mpl::void_
	>	alone_base_c;

	class c_alone : public alone_base_c, boost::noncopyable
	{
	public:

		typedef alone_base_c		base;
		typedef c_tmp				tmp_type;
		typedef alone_base_c		base;

	private:
		typedef
			std::map<typename base::handler_type, std::size_t>
			used_handle_counter_map_type;

		static used_handle_counter_map_type& get_used_handle_count_map()
		{
			static used_handle_counter_map_type obj;

			return obj;
		}

	public:

		c_alone(tmp_type ins):
			base(ins.get_handle())
		{
			if(!ins)
			{
				return;
			}

			typename
			used_handle_counter_map_type::iterator i
			=
			get_used_handle_count_map().find(&ins.get_handle());

			if(i==get_used_handle_count_map().end())
			{
				get_used_handle_count_map().insert
				(
					typename
					used_handle_counter_map_type::value_type
					(
						&ins.get_handle(),
						1
					)
				);

				return;
			}

			i->second++;
		}

		~c_alone()
		{
			if(!*this)
			{
				return;
			}

			typename
			used_handle_counter_map_type::iterator i
			=
			get_used_handle_count_map().find(&this->get_handle());

			assert(i != get_used_handle_count_map().end());

			if(i->second > 1)
			{
				i->second--;
				return;
			}

			base::release(&this->get_handle());

			get_used_handle_count_map().erase(i);
		}

		friend class c_tmp;
	};

};

}	// end of namespace gpuppur

#endif
