/**
 *	@file	
 *	@brief	Physics engine class
 *	@author	Tomohiro Matsumoto
 */

#include "stdafx.h"
//#include "nxphysics.hpp"

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <boost/filesystem/operations.hpp>
#include <fstream>
#include <NxActorDesc.h>
#include <NxBoxShapeDesc.h>
#include <NxMaterial.h>
#include <NxPhysicsSDK.h>
#include <NxPlaneShapeDesc.h>
#include <NxRay.h>
#include <NxSphereShapeDesc.h>
#include <NxSceneDesc.h>
#include <NxTriangleMeshDesc.h>
#include <NxTriangleMeshShapeDesc.h>
#include <NxVec3.h>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/physx/stream.hpp>
#include <gpuppur/utility/frame_per_second.hpp>
#include <gpuppur/utility/script.hpp>
#include <gpuppur/3dfileloader/3dfileLoader.hpp>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace gpuppur;

namespace
{
	const float cube_size = 1.0f;
}

void physics_implement::uninitialize()
{
	if(this->cook)
	{
		this->cook->NxCloseCooking();
		this->cook = NULL;
	}

	if(this->scene_query)
	{
		this->scene->releaseSceneQuery(*this->scene_query);
		this->scene_query = NULL;
	}

	if(this->scene)
	{
		this->physics_sdk->releaseScene(*this->scene);
		physics_implement::get_valid_actor_set().erase(this->scene);
		this->scene = NULL;
	}

	physics_implement::get_valid_mesh_set().erase(this);
}

bool physics_implement::initialize()
{
	this->uninitialize();

	if(!this->physics_sdk.initialize())
		return false;

	bool has_hard;
	if(this->physics_sdk->getHWVersion() == NX_HW_VERSION_NONE)
	{
		std::cout << "No PhysX hardware" << std::endl;
		has_hard = false;
	}else
	{
		std::cout << "PhysX hardware available" << std::endl;
		has_hard = true;
	}

	NxSceneDesc scene_desc;
	scene_desc.simType = has_hard ? NX_SIMULATION_HW : NX_SIMULATION_SW;
	scene_desc.flags |= has_hard ? NX_SF_RESTRICTED_SCENE : 0;
	scene_desc.flags = NX_SF_DISABLE_COLLISIONS;

 	this->scene = this->physics_sdk->createScene(scene_desc);
	if(!this->scene)
	{
		this->physics_sdk.release();
		return false;
	}

	physics_implement::get_valid_actor_set().insert
	(
		actor_set_per_scene::value_type(this->scene, std::set<const NxActor*>())
	);

	physics_implement::get_valid_mesh_set().insert
	(
		mesh_set_per_physics::value_type
		(
			this,
			mesh_set_per_physics::value_type::second_type()
		)
	);

#if 0
	NxMaterial * defaultMaterial = this->scene->getMaterialFromIndex(0); 
	defaultMaterial->setRestitution(0.3f);
	defaultMaterial->setStaticFriction(0.8f);
	defaultMaterial->setDynamicFriction(0.9f);

	NxPlaneShapeDesc PlaneDesc;
	NxActorDesc ActorDesc;
	PlaneDesc.normal = NxVec3(0.0f, 1.0f, 0.0f);
	PlaneDesc.d = 0.0f;
	PlaneDesc.userData = (void*)new vector3(0.5f, 0.5f, 1.0f);
	ActorDesc.shapes.pushBack(&PlaneDesc);
//	this->actors.push_back(
		this->scene->createActor(ActorDesc);
//	);
#endif

	return true;
}

#if 0
//GL_TEXTURE_RECTANGLE_NVを使う場合、この変数に画像の実際のサイズを入れる。 
const GLfloat tex_width = 32;
const GLfloat tex_height = 32;

static void draw_cude()
{
	const float half_size = cube_size;

	glBegin(GL_TRIANGLE_STRIP);
//	glBegin(GL_LINE_STRIP);
		// front face
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f((float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, tex_height);
		glNormal3f((float)M_SQRT1_3, -(float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(half_size, -half_size, half_size);

		//bottom face
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f((float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(half_size, -half_size, -half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, -half_size);

		//left face
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, tex_height);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, -half_size);

		//top face
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f((float)M_SQRT1_3, (float)M_SQRT1_3, (float)M_SQRT1_3);
		glVertex3f(half_size, half_size, half_size);
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, 0.0f);
		glNormal3f((float)M_SQRT1_3, (float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(half_size, half_size, -half_size);

		//right face
		glMultiTexCoord2f(GL_TEXTURE0, 0.0f, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, 0.0f, tex_height);
		glNormal3f((float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(half_size, -half_size, -half_size);

		//back face
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, 0.0f);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, 0.0f);
		glNormal3f(-(float)M_SQRT1_3, (float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, half_size, -half_size);
		glMultiTexCoord2f(GL_TEXTURE0, tex_width, tex_height);
		glMultiTexCoord2f(GL_TEXTURE1, tex_width, tex_height);
		glNormal3f(-(float)M_SQRT1_3, -(float)M_SQRT1_3, -(float)M_SQRT1_3);
		glVertex3f(-half_size, -half_size, -half_size);
	glEnd();
}
#endif

void physics_implement::process()
{
	assert(this->physics_sdk);
	assert(this->scene != NULL);
#if 0
	float mat[16];

	std::vector<NxActor*>::iterator i = this->actors.begin();
	for(;i!=this->actors.end(); ++i)
	{
		(*i)->getGlobalPose().getColumnMajor44(mat);

		glPushMatrix();
		glMultMatrixf(mat);
		draw_cude();
		glPopMatrix();
	}
#endif

//	NxU32 num = this->scene->getNbActors();
//	num = this->scene->getTotalNbShapes();

	this->scene->simulate(gpuppur::frame_per_second::get_spf());
	this->scene->flushStream();
	this->scene->fetchResults(NX_RIGID_BODY_FINISHED, true);
}

physics_implement::mesh_tmp physics_implement::load_mesh
(
	const std::vector<vector3>&			vertices,
	const std::vector<unsigned short>&	triangles
)
{
	memory_stream strm;

	if(!this->write_cooked_mesh(vertices, triangles, strm))
	{
		return physics_implement::mesh_tmp();
	}

	strm.begin_read();
	return
	physics_implement::mesh_tmp
	(
		*this->create_triangle_mesh(strm)
	);
}

physics_implement::mesh_tmp
physics_implement::load_mesh_from_wavefront
(
	const std::string& filename
)
{
	const static std::string cooked_file_extension=".cooked";
	const std::string cooked_filename = filename+cooked_file_extension;

	if
	(
		filesystem::exists(cooked_filename)
		&&
		this->is_cooked_by_current_version_physx()
	)
	{
		/*file_stream*/UserStream  file(cooked_filename.c_str(), true);
		return *this->create_triangle_mesh(file);
	}

	Model model;

	if(!model.load(filename))
	{
		return physics_implement::mesh_tmp();
	}

	// only use model.objects[0]
	vector<vector3> vertices(model.objects[0].verts.size());

	int n=0;
	for
	(
		vector<Vector3d>::iterator i=model.objects[0].verts.begin();
		i!=model.objects[0].verts.end();
		++i
	)
	{
		vertices[n++]
		=
		vector3
		(
			static_cast<float>((*i)[0]),
			float((*i)[1]),
			float((*i)[2])
		);
	}

	{
		std::size_t num_index = model.objects[0].faces.size()*3;
		Object::Faces::iterator i	=	model.objects[0].faces.begin();
		Object::Faces::iterator end	=	model.objects[0].faces.end();

		if(num_index > 0x7fff)
		{
			vector<unsigned int> triangles(num_index);

			n=0;
			for(;i!=end;++i)
			for(size_t j=0; j<3; ++j)
			{
				triangles[n++] = static_cast<unsigned int>(i->vertIndices[j]);
			}

			return this->load_mesh(vertices, triangles, cooked_filename);
		}else
		{
			vector<unsigned short> triangles(num_index);

			n=0;
			for(;i!=end;++i)
			for(size_t j=0; j<3; ++j)
			{
				triangles[n++] = static_cast<unsigned short>(i->vertIndices[j]);
			}

			return this->load_mesh(vertices, triangles, cooked_filename);
		}
	}
}

physics_implement::mesh_tmp
physics_implement::load_mesh_from_cooked
(
	const std::string& filename
)
{
	assert(this->physics_sdk != NULL);

	/*file_stream*/UserStream  file(filename.c_str(), true);

	if(!file)
	{
		return physics_implement::mesh_tmp();
	}

	return
	physics_implement::mesh_tmp
	(
		*this->create_triangle_mesh(file)
	);
}

NxActor*
physics_implement::create_mesh
(
	const physics_implement::mesh_tmp handle,
	const vector3& pos,
	void* user_data
)
{
	assert(this->physics_sdk);
	assert(this->scene != NULL);

	if(!handle)
	{
		return NULL;
	}

	NxTriangleMeshShapeDesc mesh_desc;

	assert(handle.get_handle().getCount(0, NX_ARRAY_TRIANGLES) > 0);

	//handled mesh will never be modified by PhysX, maybe...
	mesh_desc.meshData		= &const_cast<mesh_tmp&>(handle).get_handle();
//	mesh_desc.userData		= user_data;

	NxActorDesc actor_desc;
	actor_desc.shapes.pushBack(&mesh_desc);
	// Initialize as dynamic actor, because static actor and kinematic actor cannot be set pose.
	NxBodyDesc				body_desc;
	body_desc.flags			= NX_BF_DISABLE_GRAVITY;//NX_BF_FROZEN;//NX_BF_KINEMATIC;
	actor_desc.body			= &body_desc;
	actor_desc.density		= 10.0f;
	actor_desc.flags
	=
	NX_AF_DISABLE_COLLISION | NX_AF_DISABLE_RESPONSE | NX_AF_FLUID_DISABLE_COLLISION;
	actor_desc.userData		= user_data;

	actor_desc.globalPose.t = *reinterpret_cast<NxVec3*>(const_cast<vector3*>(&pos));

	NxActor* tmp = this->scene->createActor(actor_desc);
	assert(tmp->userData == actor_desc.userData);

	get_valid_actor_set()[this->scene].insert(tmp);

	return tmp;
}

NxActor*
physics_implement::create_sphere
(
	const float	radius,
	const vector3& pos,
	void* user_data
)
{
	assert(this->physics_sdk != NULL);
	assert(this->scene != NULL);

	NxSphereShapeDesc	sphere_desc;
	sphere_desc.radius = radius;
	sphere_desc.userData = user_data;

	NxActorDesc actor_desc;
	actor_desc.shapes.pushBack(&sphere_desc);
	NxBodyDesc				body_desc;
	body_desc.flags			= NX_BF_DISABLE_GRAVITY;
	actor_desc.body			= &body_desc;
	actor_desc.density		= 10.0f;
	actor_desc.flags		= NX_AF_DISABLE_COLLISION | NX_AF_DISABLE_RESPONSE | NX_AF_FLUID_DISABLE_COLLISION;
	actor_desc.globalPose.t = *reinterpret_cast<NxVec3*>(const_cast<vector3*>(&pos));

	NxActor* tmp = this->scene->createActor(actor_desc);

	get_valid_actor_set()[this->scene].insert(tmp);

	return tmp;
}

NxActor*
physics_implement::cast_ray
(
	const ray& casted_ray,
	float max_t,
	vector3& pos,
	vector3& normal
) const
{
	assert(this->physics_sdk != NULL);
	assert(this->scene != NULL);


//	static NxShape* test_cache[128];	//This variable must be member variable of this class.

	this->count_sampling++;

	NxRaycastHit hit;

	NxShape* shape = /**NULL;/*/this->scene->raycastClosestShape
	(
		*reinterpret_cast<NxRay*>(&const_cast<ray&>(casted_ray)),
		NX_ALL_SHAPES,
		hit,
		0xffffffff,
		max_t,
		NX_RAYCAST_SHAPE | NX_RAYCAST_FACE_NORMAL,//| NX_RAYCAST_NORMAL,
		NULL
	//	, test_cache
	);//*/
	if(shape == NULL)
	{
		pos = 
			normal = vector3(0.0f, 0.0f, 0.0f);

		return NULL;
	}

//	pos = *reinterpret_cast<vector3*>(&hit.worldImpact);
	pos = to_vec3(hit.worldImpact);
//	normal = *reinterpret_cast<vector3*>(&hit.worldNormal);
	normal = to_vec3(hit.worldNormal);

	return &shape->getActor();
//	return reinterpret_cast<const user_data_base*>(shape->userData)->get_user_data();

/*
	//Check cache.
	// result:
	// Always cache size == 1
	// cache使ってもほとんど速くなってない.   
	static int cnt;

	if(++cnt == 10000)
	{
		for(int i=0; i<1024; ++i)
		{
			if(test_cache[i] == NULL)
			{
				std::cout << "cache size = " << i << std::endl;
				break;
			}
		}

		cnt = 0;
	}
*/
}

NxActor*
physics_implement::cast_ray
(
	const ray& casted_ray,
	float max_t
) const
{
	assert(this->physics_sdk != NULL);
	assert(this->scene != NULL);

	this->count_sampling++;

	NxRaycastHit hit;

	NxShape* shape = this->scene->raycastClosestShape
	(
		*reinterpret_cast<NxRay*>(&const_cast<ray&>(casted_ray)),
		NX_ALL_SHAPES,
		hit,
		0xffffffff,
		max_t,
		NX_RAYCAST_SHAPE
	);

	if(shape == NULL)
	{
		return NULL;
	}

	return &shape->getActor();

//	return	boost::make_tuple(instance, shape->userData);
}

void physics_implement::create_cube(const vector3& pos, const vector3& initial_velocity)
{
	assert(this->physics_sdk);
	assert(this->scene != NULL);

	// Create body
	NxBodyDesc BodyDesc;
	BodyDesc.angularDamping		= 0.5f;
//	BodyDesc.maxAngularVelocity	= 10.0f;
	vector3 vel					= initial_velocity;
	BodyDesc.linearVelocity		= *reinterpret_cast<NxVec3*>(&vel);

	NxBoxShapeDesc BoxDesc;
	BoxDesc.dimensions			= NxVec3(cube_size, cube_size, cube_size);
	BoxDesc.userData			= (void*)new vector3(0.5f, 1.0f, 0.5f);

	NxActorDesc ActorDesc;
	ActorDesc.userData			= NULL;
	ActorDesc.shapes.pushBack(&BoxDesc);
	ActorDesc.body				= &BodyDesc;
	ActorDesc.density			= 10.0f;
	vector3 tmp_pos				= pos;
	ActorDesc.globalPose.t		= *reinterpret_cast<NxVec3*>(&tmp_pos);

	NxActor* tmp = this->scene->createActor(ActorDesc);
	tmp->userData = NULL;
//	this->actors.push_back(tmp);
}

bool physics_implement::make_tower_scene(std::size_t height)
{
	if(!this->initialize())
	{
		return false;
	}

	const int h = static_cast<int>(height);
	const float offset=cube_size/100000.0f;
	const float hoffset=cube_size/100000.0f;
	const vector3 pos(0.0f, 1.0f, -20.0f);
	for(int k=0; k<h; ++k)
	for(int j=0; j<k+1; ++j)
	for(int i=0; i<k+1; ++i)
	{
		float x = (i-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float z = (j-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float y = k*(cube_size*2+hoffset);
		this->create_cube(vector3(x, y, z) + pos, vector3(0.0f, 0.0f, 0.0f));
	}

	for(int k=h-2; k>=0; --k)
	for(int j=0; j<k+1; ++j)
	for(int i=0; i<k+1; ++i)
	{
		float x = (i-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float z = (j-k/2)*(cube_size*2.0f+offset)-(k&1)*(cube_size+offset/2.0f);
		float y = (h*2-k-2)*(cube_size*2+hoffset);
		this->create_cube(vector3(x, y, z) + pos, vector3(0.0f, 0.0f, 0.0f));
	}

	for(int k=0; k<h*3; ++k)
	{
		float y = (h*2+k-1)*(cube_size*2+hoffset);
		this->create_cube(vector3(0.0f, y, 0.0f) + pos, vector3(0.0f, 0.0f, 0.0f));
	}

	return true;
}

//箱の数とレイキャストに掛かる時間は比例している! in physx ver2.5 or ver2.4...
void physics_implement::add_stack()
{
	const size_t h = 5;
	static float x = 0.0f;

	for(size_t i=0; i<h; ++i)
	{
		this->create_cube(vector3(x, i*cube_size*2+cube_size*0.1f+1.0f, 0.0f), vector3(0.0f, 0.0f, 0.0f));
	}

	x += cube_size*2.0f;
}

bool physics_implement::add_tube(unsigned short height)
{
	std::vector<vector3> v;
	const unsigned short num_round = 8;
	const vector3 pos(0.0f, 0.0f, 0.0f);

//	v.push_back(vector3(0.0f, 1.0f, 0.0f));
//	v.push_back(vector3(-1.0f, 0.0f, -1.0f));
//	v.push_back(vector3(1.0f, 0.0f, -1.0f));
//	v.push_back(vector3(0.0f, 0.0f, 1.0f));

	v.push_back(pos);

	for(NxU16 i=0; i<height+1; ++i)
	for(NxU16 j=0; j<num_round; ++j)
	{
		const float sita = 2*(float)M_PI*j/num_round;
		float x = cos(sita) + sin((float)i/1.0f);
		float z = sin(sita);
		float y = static_cast<float>(i);

		v.push_back(pos+vector3(x, y, z));
	}

	v.push_back(vector3(pos[0], pos[1]+height, pos[2]));

	std::vector<unsigned short> t;

//	t.push_back(0), t.push_back(2), t.push_back(1);
//	t.push_back(0), t.push_back(1), t.push_back(3);
//	t.push_back(0), t.push_back(3), t.push_back(2);
//	t.push_back(3), t.push_back(1), t.push_back(2);

	// bottom
	for(unsigned short i=0; i<num_round; ++i)
	{
		t.push_back(0);
		t.push_back(i+1);
		t.push_back((i+1)%num_round+1);
	}

	unsigned short base = 1;
	// side
	for(unsigned short i=0; i<height; ++i)
	{
		for(unsigned short j=0; j<num_round; ++j)
		{
			t.push_back(base+j);
			t.push_back(base+j+num_round);
			t.push_back(base+(j+1)%num_round+num_round);

			t.push_back(base+j);
			t.push_back(base+(j+1)%num_round+num_round);
			t.push_back(base+(j+1)%num_round);
		}

		base += num_round;
	}

	const unsigned short offset = 1+num_round*height;
	const unsigned short last = offset+num_round;
	// bottom
	for(unsigned short i=0; i<num_round; ++i)
	{
		t.push_back(last);
		t.push_back(offset+(i+1)%num_round);
		t.push_back(offset+i);
	}

	physics_implement::mesh_static handle = this->load_mesh(v, t, "../data/mesh");

	if(handle)
	{
		cerr << "Failed to load mesh" << endl;
		return false;
	}

	this->create_mesh(handle, vector3(0.0f, 3.0f, -8.0f), new gpuppur::material(vector3(0.7f, 0.5f, 0.2f)));

	return true;
}

bool physics_implement::is_cooked_by_current_version_physx() const
{
	static bool has_checked = false;
	static bool ret;

	if(has_checked)
	{
		return ret;
	}

	assert(this->physics_sdk);
	NxU32 apiRev, descRev, branchId, ver;
	ver = this->physics_sdk->getInternalVersion(apiRev, descRev, branchId);

	gpuppur::script sc;
	const static string date_filename("../data/mesh/date.lua");

	if(sc.initialize(date_filename))
	{
		if
		(
			apiRev == sc.get_value<NxU32>("apiRev")
			&&
			descRev == sc.get_value<NxU32>("descRev")
			&&
			branchId == sc.get_value<NxU32>("branchId")
			&&
			ver == sc.get_value<NxU32>("ver")
		)
		{
			has_checked = true;
			ret = true;
			return true;
		}
	}

	has_checked = true;
	ret = false;

	fstream f(date_filename.c_str(), ios_base::out | ios_base::trunc);
	assert(f.is_open());

	f
	<< "apiRev=" << apiRev << '\n'
	<< "descRev=" << descRev << '\n'
	<< "branchId=" << branchId << '\n'
	<< "ver=" << ver << '\n';

	return false;
}

/*
bool physics_implement::write_cooked_mesh
(
	const std::vector<vector3>&			vertices,
	const std::vector<unsigned short>&	triangles,
	NxStream&							stream
)
{
	if(this->cook == NULL)
	{
		this->cook = NxGetCookingLib(NX_PHYSICS_SDK_VERSION);

		if(this->cook == NULL)
		{
			return false;
		}

		if(!this->cook->NxInitCooking())
		{
			return false;
		}

		NxCookingParams param;
		param.hintCollisionSpeed = true;
		if(!this->cook->NxSetCookingParams(param))
		{
			return false;
		}
	}

	NxTriangleMeshDesc desc;

	desc.numVertices			= static_cast<NxU32>(vertices.size());
	desc.numTriangles			= static_cast<NxU32>(triangles.size()/3);
	desc.pointStrideBytes		= sizeof(vector3);
	desc.triangleStrideBytes	= sizeof(unsigned short)*3;
	desc.points					= &vertices[0];
	desc.triangles				= &triangles[0];
	desc.flags					= NX_MF_16_BIT_INDICES;

	if(!this->cook->NxCookTriangleMesh(desc, stream))
//	if(!NxCookTriangleMesh(desc, stream))
	{
		return false;
	}

	return true;
}
*/

bool physics_implement::write_cooked_mesh_impl
(
	const std::vector<vector3>&	vertices,
	const void*					triangles,
	std::size_t					num_triangles,
	bool						is_16bit_indices,
	NxStream&					stream
)
{
	if(this->cook == NULL)
	{
		this->cook = NxGetCookingLib(NX_PHYSICS_SDK_VERSION);

		if(this->cook == NULL)
		{
			return false;
		}

		if(!this->cook->NxInitCooking())
		{
			return false;
		}
/*
		NxCookingParams param;
		param.hintCollisionSpeed = true;
		if(!this->cook->NxSetCookingParams(param))
		{
			return false;
		}
*/
	}

	NxTriangleMeshDesc desc;

	desc.numVertices			= static_cast<NxU32>(vertices.size());
	desc.numTriangles			= static_cast<NxU32>(num_triangles);
	desc.pointStrideBytes		= sizeof(vector3);
	desc.points					= &vertices[0];
	desc.triangles				= triangles;

	if(is_16bit_indices)
	{
		desc.triangleStrideBytes	= sizeof(unsigned short)*3;
		desc.flags					= NX_MF_16_BIT_INDICES;
	}else
	{
		//32bit index
		desc.triangleStrideBytes	= sizeof(unsigned int)*3;
		desc.flags					= 0;
	}

	assert(desc.isValid());

	if(!this->cook->NxCookTriangleMesh(desc, stream))
	{
		return false;
	}

	return true;
}

