#ifndef NXPHYSICS_HPP
#define NXPHYSICS_HPP

/**
 *	@file	
 *	@brief	Physics class
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <algorithm>
#include <map>
#include <set>
#include <boost/tuple/tuple.hpp>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/call_traits.hpp>
#include <boost/type_traits.hpp>
#include <boost/filesystem/operations.hpp>
#include <NxTriangleMesh.h>
#include <NxActor.h>
#include <NxCooking.h>
#include <NxScene.h>
#include <PhysXLoader.h>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include "../handle_generic.hpp"
#include "../handler.hpp"
#include <gpuppur/physx/stream.hpp>

#include <gpuppur/mesh/index_buffer.hpp>

#define NX_SAFE_RELEASE(p) if((p)!=NULL){(p)->release();(p)=NULL;}
#define	NX_SAFE_DELETE(p) if((p)!=NULL){delete (p);(p)=NULL;}

namespace gpuppur
{

class physics_implement
{
private:

	#include "mesh.hpp"

public:

	typedef gpuppur::handler_tmpl<mesh_implement>::c_static			mesh_static;
	typedef gpuppur::handler_tmpl<mesh_implement>::c_tmp			mesh_tmp;
	typedef gpuppur::handler_tmpl<mesh_implement>::c_alone			mesh_alone;

	physics_implement():
		scene(NULL),
		cook(NULL),
		count_sampling(0),
		scene_query(NULL)
	//	is_initialized_cook(false)
	{
	}

	~physics_implement()
	{
	}

	void uninitialize();
	bool initialize();
	void process();
	mesh_tmp load_mesh
	(
		const std::vector<vector3>& vertices,
		const std::vector<unsigned short>& triangles
	);

	template<typename T>
	mesh_tmp load_mesh
	(
		const std::vector<vector3>&		vertices,
		const std::vector<T>&			triangles,
		const std::string&				filename,
		typename boost::enable_if_c
		<
			boost::is_same<T, unsigned short>::value
			||
			boost::is_same<T, unsigned int>::value
		>::type* =0
	)
	{
		if
		(
			!boost::filesystem::exists(filename)
			||
			!this->is_cooked_by_current_version_physx()
		)
		{
			/*file_stream*/UserStream  file(filename.c_str(), false);

			if(!this->write_cooked_mesh(vertices, triangles, file))
			{
				return physics_implement::mesh_tmp();
			}
		}

		/*file_stream*/UserStream  file(filename.c_str(), true);
		return mesh_tmp(*this->create_triangle_mesh(file));
	}

	mesh_tmp load_mesh_from_wavefront(const std::string& filename);
	mesh_tmp load_mesh_from_cooked(const std::string& filename);

	NxActor* create_mesh
	(
		const mesh_tmp handle,
		const vector3& pos,
		void* user_data
	);

	NxActor* create_sphere
	(
		const float	radius,
		const vector3& pos,
		void* user_data
	);

	const int get_count_sampling() const
	{
		int ret = this->count_sampling;
		this->count_sampling = 0;
		return ret;
	}

	NxActor* cast_ray
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal
	) const;

	NxActor* cast_ray
	(
		const ray& casted_ray,
		float max_t
	) const;

	void cast_ray_batch
	(
		const	ray& casted_ray,
		float	max_t,
		int		id
	) const
	{
		assert
		(
			this->scene_query
			&&
			"call set_casted_ray_func() before calling this function!"
		);

		NxRaycastHit hit;

		this->scene_query->raycastClosestShape
		(
			*reinterpret_cast<NxRay*>(&const_cast<ray&>(casted_ray)),
			NX_ALL_SHAPES,
			hit,
			0xffffffff,
			max_t,
			NX_RAYCAST_SHAPE | NX_RAYCAST_FACE_NORMAL,//| NX_RAYCAST_NORMAL,
			NULL,
			NULL,	//cache,
			reinterpret_cast<void*>(id)		//userData
		);
	}

	void cast_ray_execute()
	{
		assert(this->physics_sdk);
		assert(this->scene);
		assert(this->scene_query);
//		assert(this->query_reported.has_data());

		this->scene_query->execute();

		this->scene_query->finish(true);
	}

	void set_casted_ray_func(NxSceneQueryReport* query_reported)
	{
		assert(this->physics_sdk);
		assert(this->scene);

		if(this->scene_query)
		{
			bool ret = this->scene->releaseSceneQuery(*this->scene_query);
			assert(ret && "I thought this->scene->releaseSceneQuery never fail...");
		}

		NxSceneQueryDesc	desc;
		desc.report			= query_reported;
		desc.executeMode	= NX_SQE_ASYNCHRONOUS;
		assert(desc.isValid() && "barairo no zinsei");
		this->scene_query = this->scene->createSceneQuery(desc);
		assert
		(
			this->scene_query
			&&
			"I thought this->scene->createSceneQuery() never fail..."
		);
	}

	boost::tuple<std::size_t, NxActor**>
	get_actors()
	{
		return boost::make_tuple
		(
			this->scene->getNbActors(),
			this->scene->getActors()
		);
	}

	void create_cube(const vector3& pos, const vector3& initial_velocity);
	bool make_tower_scene(std::size_t height);
	void add_stack();
	bool add_tube(unsigned short height);
	bool is_cooked_by_current_version_physx() const;

protected:

	static vector3 to_vec3(const NxVec3& nxvec)
	{
		return vector3(nxvec.x, nxvec.y, nxvec.z);
	}

	NxScene* get_nxscene()
	{
		return this->scene;
	}

	typedef std::map<const NxScene*, std::set<const NxActor*> > actor_set_per_scene;

	static actor_set_per_scene&	get_valid_actor_set()
	{
		static actor_set_per_scene	asps;

		return asps;
	}

	typedef std::map
	<
		const physics_implement*,
		std::set<const NxTriangleMesh*>
	> mesh_set_per_physics;

	static mesh_set_per_physics& get_valid_mesh_set()
	{
		static mesh_set_per_physics mspp;

		return mspp;
	}

private:

	bool write_cooked_mesh_impl
	(
		const std::vector<vector3>&	vertices,
		const void*					triangles,
		std::size_t					num_triangles,
		bool						is_16bit_indices,
		NxStream&					stream
	);

	bool write_cooked_mesh
	(
		const std::vector<vector3>&			vertices,
		const std::vector<unsigned short>&	triangles,
		NxStream&							stream
	)
	{
		return this->write_cooked_mesh_impl
		(
			vertices,
			&triangles[0],
			triangles.size()/3,
			true,
			stream
		);
	}

	bool write_cooked_mesh
	(
		const std::vector<vector3>& vertices,
		const std::vector<unsigned int>& triangles,
		NxStream& stream
	)
	{
		return this->write_cooked_mesh_impl
		(
			vertices,
			&triangles[0],
			triangles.size()/3,
			false,
			stream
		);
	}

	NxTriangleMesh* create_triangle_mesh(const NxStream& stream) 
	{
		assert(this->physics_sdk);

		NxTriangleMesh* ret =
		this->physics_sdk->createTriangleMesh(stream);

		if(ret)
		{
			get_valid_mesh_set()[this].insert(ret);
		}

		return ret;
	}

	class nxphysics_sdk
	{
	private:

		nxphysics_sdk(NxPhysicsSDK* sdk):
			physics_sdk(sdk)
		{
		}

	public:

		nxphysics_sdk():
			physics_sdk(NULL)
		{
		}

		~nxphysics_sdk()
		{
			this->release();
		}

		void release()
		{
			if(*this)
			{
				//Physics SDK is reference counted.
				NxReleasePhysicsSDK(physics_sdk);
				this->physics_sdk = NULL;
			}
		}

		bool initialize()
		{
			if(*this)
			{
				return true;
			}

			NxPhysicsSDKDesc sdkDesc;
		//	sdkDesc.flags |= NX_SDKF_NO_HARDWARE;	//Disable Hardware acceleration
			this->physics_sdk =
			NxCreatePhysicsSDK(NX_PHYSICS_SDK_VERSION, 0, 0, sdkDesc);

			return *this;
		}

		operator bool() const
		{
			return this->physics_sdk != NULL;
		}

		NxPhysicsSDK* operator ->() const
		{
			return this->physics_sdk;
		}

		NxPhysicsSDK*	physics_sdk;
	};

	nxphysics_sdk				physics_sdk;
	NxScene*					scene;

	NxCookingInterface*			cook;
//	bool						is_initialized_cook;

	mutable int					count_sampling;
	NxSceneQuery*				scene_query;
};	// end of class physics_implement

/**
 *	@brief	Physics class
 *
 *	You can create/destory 3D object in this class.
 *	And this class can raytracing at them.
 */
template<typename UserDataType>
class physics : public physics_implement
{
private:

	class nxphysics_sdk;
//	class user_data_base;
//	template<typename Data>
//	class user_data_storage;

	#include "instance3d.hpp"

	typedef physics_implement									base;

public:

	typedef instance3d_implement<UserDataType>					handle_to_instance3d;
	typedef typename gpuppur::handler_tmpl<handle_to_instance3d::template c>::c_static
																instance3d_static;
	typedef typename gpuppur::handler_tmpl<handle_to_instance3d::template c>::c_tmp
																instance3d_tmp;
private:
	typedef boost::function
	<
		void
		(
			const instance3d_tmp,
			const vector3&	pos,
			const vector3&	normal,
			const int		id
		)
	>	casted_ray_function;
	
public:

	~physics()
	{
		this->uninitialize();
	}

	void uninitialize()
	{
		if(base::get_nxscene())
		{
			NxU32 num_actor = base::get_nxscene()->getNbActors();
			NxActor** actors = base::get_nxscene()->getActors();
			for(NxU32 i = 0; i<num_actor; ++i)
			{
				uninitialize_actor(actors[i]);
			}
		}

		base::uninitialize();
	}

	bool initialize()
	{
		this->uninitialize();

		return base::initialize();
	}

	instance3d_tmp create_mesh
	(
		const mesh_tmp		handle,
		const vector3&		pos,
		const UserDataType&	user_data
	)
	{
		return
		*base::create_mesh
		(
			handle, pos,
			new UserDataType(user_data)
		);
	}

	instance3d_tmp create_sphere
	(
		const float			radius,
		const vector3&		pos,
		const UserDataType&	user_data
	)
	{
		return
		*base::create_sphere
		(
			radius, pos,
			new UserDataType(user_data)
		);
	}

	instance3d_tmp cast_ray
	(
		const ray& casted_ray,
		float max_t,
		vector3& pos,
		vector3& normal
	) const
	{
		return
		*base::cast_ray
		(
			casted_ray,
			max_t,
			pos, normal
		);
	}

	instance3d_tmp 
	cast_ray
	(
		const ray& casted_ray,
		float max_t
	) const
	{
		return
		*base::cast_ray
		(
			casted_ray,
			max_t
		);
	}

	std::vector<instance3d_tmp>
	get_all_instances()
	{
		boost::tuple<std::size_t, NxActor**> actors
		=
		base::get_actors();

		std::vector<instance3d_tmp> ret;
		NxActor** p_actors = boost::get<1>(actors);

		for(std::size_t i=0; i<boost::get<0>(actors); ++i, ++p_actors)
		{
			ret.push_back(instance3d_tmp(**p_actors));
		}

//		std::copy(p_actors, p_actors+ret.size(), ret.begin());

		return ret;
	}

	void set_casted_ray_func(const casted_ray_function& func)
	{
		this->query_reported.func = func;
		base::set_casted_ray_func(&this->query_reported);
	}

	const casted_ray_function get_casted_ray_func() const
	{
		return this->query_reported.func;
	}

private:

	static void uninitialize_actor(NxActor* const actor)
	{
		delete reinterpret_cast<UserDataType*>(actor->userData);
	}

	class query_report : public NxSceneQueryReport
	{
	public:

		virtual ~query_report()
		{
		}

		NxQueryReportResult
		onBooleanQuery (void* /*userData*/, bool/* result*/)
		{
			assert(false && "this function is not used currently");

			return NX_SQR_ABORT_ALL_QUERIES;
		}

		NxQueryReportResult
		onRaycastQuery (void *userData, NxU32 nbHits, const NxRaycastHit *hits)
		{
			assert(!this->func.empty());

			for(NxU32 i=0; i<nbHits; ++i)
			{
				const NxRaycastHit& hit = hits[i];

				union{int i; void* v;} id;
				id.v = userData;

				this->func
				(
					physics::to_instance3d(&hit.shape->getActor()),
					physics::to_vec3(hit.worldImpact),
					physics::to_vec3(hit.worldNormal),
					id.i
				);
			}

			return NX_SQR_CONTINUE;
		}

		NxQueryReportResult
		onShapeQuery (void* /*userData*/, NxU32 /*nbHits*/, NxShape** /*hits*/)
		{
			assert(false && "this function is not used currently");

			return NX_SQR_ABORT_ALL_QUERIES;
		}

		NxQueryReportResult
		onSweepQuery (void* /*userData*/, NxU32 /*nbHits*/, NxSweepQueryHit* /*hits*/)
		{
			assert(false && "this function is not used currently");

			return NX_SQR_ABORT_ALL_QUERIES;
		}

		bool has_data()
		{
			return !this->func.empty();
		}

		casted_ray_function	func;
	};

	friend class	query_report;
	query_report	query_reported;

	static const instance3d_tmp to_instance3d(NxActor* handler)
	{
		return instance3d_tmp(*handler);
	}

//	template<class, class> friend class instance3d_implement<UserDataType>::template c;
/*
	class user_data_base
	{
	public:

		virtual void* get_user_data() = 0;

		virtual ~user_data_base()
		{
		}
	};

	template<typename Data>
	class user_data_storage : public user_data_base
	{
	public:

		user_data_storage
		(
				typename
				boost::call_traits<Data>::param_type
				user_data
		):
			data(user_data)
		{
		}

		~user_data_storage()
		{
		}

		void* get_user_data()
		{
			return &this->data;
		}

		UserData get_user_data() const
		{
			return this->data;
		}

	private:

		Data data;
	};
*/

};	// end of class physics
}	// end of namespace gpuppur


#endif
