#ifndef GPUPPURUT_CAMERA_HPP
#define GPUPPURUT_CAMERA_HPP

/**
 *	@file	
 *	@brief	Camera class
 *	@author	Tomohiro Matsumoto
 */

#include <cmath>
#include <gpuppur/3dmath/vectorNd.hpp>
#include <gpuppur/3dmath/matrixRxC.hpp>

namespace gpuppur
{

/**
 *	@brief	Camera class
 *
 *	This class have state of camera.
 *	It is up vector, front vector and position of camera.
 *	But this class doesnt have state about how scene is rendered to image.
 */
class camera
{
public:
	typedef VectorNd<float, 3> vector3;

	camera():
		up(0.0f, 1.0f, 0.0f),
		front(0.0f, 0.0f, -1.0f),
		pos(0.0f, 0.0f, 0.0f),
		base_speed(0.6f),
		speed(0.0f),
		angle_speed(0.1f),
		base_acceleration(0.05f),
		acceleration(0.0f),

		is_pitch_uping(false),
		is_pitch_downing(false),
		is_yaw_lefting(false),
		is_yaw_righting(false),

		base_mouse_x(0), base_mouse_y(0),
		is_mouse_moving(false)
	{
	}

	camera(float x, float y, float z):
		up(0.0f, 1.0f, 0.0f),
		front(0.0f, 0.0f, -1.0f),
		pos(x, y, z),
		base_speed(0.6f),
		speed(0.0f),
		angle_speed(0.1f),
		base_acceleration(0.05f),
		acceleration(0.0f),

		is_pitch_uping(false),
		is_pitch_downing(false),
		is_yaw_lefting(false),
		is_yaw_righting(false),

		base_mouse_x(0), base_mouse_y(0),
		is_mouse_moving(false)
	{
	}

	vector3 get_up() const
	{
		return this->up;
	}

	vector3 get_front() const
	{
		return this->front;
	}

	vector3 get_pos() const
	{
		return this->pos;
	}

	void set_speed(float speed)
	{
		this->base_speed = speed;
	}

	void move_mouse(int x, int y)
	{
		if(!this->is_mouse_moving)
		{
			return;
		}

		int m_x = x - this->base_mouse_x;
		int m_y = y - this->base_mouse_y;

		this->base_mouse_x = x;
		this->base_mouse_y = y;

		float phi = m_x/400.0f;
		float theta = -m_y/400.0f;
		vector3 rot_x(-sin(phi)*cos(theta), sin(theta), cos(phi)*cos(theta));
		vector3 rot_y(sin(theta)*sin(phi), cos(theta), -cos(phi)*sin(theta));

		vector3 left = this->up.outerProduct(this->front);
		vector3 tmp_front = left*rot_x[0] + this->up*rot_x[1]+ this->front*rot_x[2];
		this->up = left*rot_y[0] + this->up*rot_y[1] + this->front*rot_y[2];

		this->front = tmp_front.getNormalized();;
	}

	void begin_moving(int x, int y)
	{
		this->base_mouse_x = x;
		this->base_mouse_y = y;
		this->is_mouse_moving =  true;
	}

	void end_moving()
	{
		this->is_mouse_moving = false;
	}
/*
	void forward()
	{
		this->pos += this->front * this->speed;
	}

	void backward()
	{
		this->pos -= this->front * this->speed;
	}

	void left()
	{
		this->pos += this->up.outerProduct(this->front) * this->speed;
	}

	void right()
	{
		this->pos -= this->up.outerProduct(this->front) * this->speed;
	}

	void rise()
	{
		this->pos += this->up * this->speed;
	}

	void descent()
	{
		this->pos -= this->up * this->speed;
	}
*/
	void pitch_up()
	{
		this->rotate_pitch(this->angle_speed);
	}
		
	void pitch_down()
	{
		this->rotate_pitch(-this->angle_speed);
	}

	void yaw_left()
	{
		this->rotate_yaw(-this->angle_speed);
	}

	void yaw_right()
	{
		this->rotate_yaw(this->angle_speed);
	}

	void keyboard(int key)
	{
		switch(key)
		{
		case 'W':
			this->acceleration[2] = this->base_acceleration;
			break;
		case 'w':
			this->speed[2] = this->base_speed;
			break;
		case 'S':
			this->acceleration[2] = -this->base_acceleration;
			break;
		case 's':
			this->speed[2] = -this->base_speed;
			break;

		case 'A':
			this->acceleration[0] = this->base_acceleration;
			break;
		case 'a':
			this->speed[0] = this->base_speed;
			break;
		case 'D':
			this->acceleration[0] = -this->base_acceleration;
			break;
		case 'd':
			this->speed[0] = -this->base_speed;
			break;

		case 'E':
			this->acceleration[1] = this->base_acceleration;
			break;
		case 'e':
			this->speed[1] = this->base_speed;
			break;
		case 'C':
			this->acceleration[1] = -this->base_acceleration;
			break;
		case 'c':
			this->speed[1] = -this->base_speed;
			break;

		case gpuppur::GPUPPURUTBase::UP:
			this->is_pitch_uping	= true;
			break;
		case gpuppur::GPUPPURUTBase::DOWN:
			this->is_pitch_downing	= true;
			break;
		case gpuppur::GPUPPURUTBase::LEFT:
			this->is_yaw_lefting	= true;
			break;
		case gpuppur::GPUPPURUTBase::RIGHT:
			this->is_yaw_righting	= true;
			break;
		}
	}

	void keyboard_up(int key)
	{
		switch(key)
		{
		case 'W':
		case 'w':
			this->acceleration[2] = 0;
			this->speed[2] = 0.0f;
			break;
		case 'S':
		case 's':
			this->acceleration[2] = 0;
			this->speed[2] = 0.0f;
			break;

		case 'A':
		case 'a':
			this->acceleration[0] = 0;
			this->speed[0] = 0.0f;
			break;
		case 'D':
		case 'd':
			this->acceleration[0] = 0;
			this->speed[0] = 0.0f;
			break;

		case 'E':
		case 'e':
			this->acceleration[1] = 0;
			this->speed[1] = 0.0f;
			break;
		case 'C':
		case 'c':
			this->acceleration[1] = 0;
			this->speed[1] = 0.0f;
			break;

		case gpuppur::GPUPPURUTBase::UP:
			this->is_pitch_uping	= false;
			break;
		case gpuppur::GPUPPURUTBase::DOWN:
			this->is_pitch_downing	= false;
			break;
		case gpuppur::GPUPPURUTBase::LEFT:
			this->is_yaw_lefting	= false;
			break;
		case gpuppur::GPUPPURUTBase::RIGHT:
			this->is_yaw_righting	= false;
			break;
		}
	}

	/**	Keep a camera horizon
	 *
	 */
	void keep_horizon()
	{
		if(this->is_mouse_moving)
		{
			return;
		}

		const float speed = 0.05f;

		vector3 left = this->up.outerProduct(this->front);

		this->up += left * speed * left[1];
		this->up.normalize();
	}

	/**	Process camera movement.
	 *
	 *	This member function should be called in every rendering frame.
	 */
	void process()
	{
		this->keep_horizon();

		this->speed += this->acceleration;

		matrixRxC<float, 3, 3> mat
		(
			this->up.outerProduct(this->front),
			this->up,
			this->front,
			false
		);
		this->pos +=  mat * this->speed;
/*
		if(this->is_forwarding)
		{
			this->forward();
		}

		if(this->is_backwarding)
		{
			this->backward();
		}

		if(this->is_lefting)
		{
			this->left();
		}

		if(this->is_righting)
		{
			this->right();
		}

		if(this->is_rising)
		{
			this->rise();
		}

		if(this->is_descenting)
		{
			this->descent();
		}
*/
		if(this->is_pitch_uping)
		{
			this->pitch_up();
		}

		if(this->is_pitch_downing)
		{
			this->pitch_down();
		}

		if(is_yaw_lefting)
		{
			this->yaw_left();
		}

		if(is_yaw_righting)
		{
			this->yaw_right();
		}
	}

	void rotate_pitch(float angle_rad)
	{
		rotate_2vectors(this->front, this->up, angle_rad);
	//	this->up = rotate_vector(this->up, this->front, -angle_rad);
	//	this->front = rotate_vector(this->front, this->up, angle_rad);
	}

	void rotate_yaw(float angle_rad)
	{
		vector3 left = this->up.outerProduct(this->front);
		this->front = rotate_vector(this->front, left, -angle_rad);
	}

protected:
	vector3 up, front;
	vector3 pos;
	float base_speed;
	vector3 speed;
	float angle_speed;
	float base_acceleration;
	vector3 acceleration;

	bool is_pitch_uping;
	bool is_pitch_downing;
	bool is_yaw_lefting;
	bool is_yaw_righting;

	int base_mouse_x, base_mouse_y;
	bool is_mouse_moving;
};

}	//end of namespace gpuppur

#endif
