#ifndef GPUPPURUT_BASE_H
#define GPUPPURUT_BASE_H

#include <string>
#include <boost/function.hpp>
#include <boost/utility.hpp>

#ifdef __CYGWIN__
	#include <cwchar>
	#include <iostream>

	namespace std
	{
		typedef std::basic_string<wchar_t> wstring;

		basic_stringbuf<wchar_t> wstrbuf;
		basic_ostream<wchar_t> wcerr(&wstrbuf);
	}
#endif

#ifndef GLUT_KEY_F1
// copy from openglut_std.h in openglut.
#define  GLUT_KEY_F1                        0x0001
#define  GLUT_KEY_F2                        0x0002
#define  GLUT_KEY_F3                        0x0003
#define  GLUT_KEY_F4                        0x0004
#define  GLUT_KEY_F5                        0x0005
#define  GLUT_KEY_F6                        0x0006
#define  GLUT_KEY_F7                        0x0007
#define  GLUT_KEY_F8                        0x0008
#define  GLUT_KEY_F9                        0x0009
#define  GLUT_KEY_F10                       0x000A
#define  GLUT_KEY_F11                       0x000B
#define  GLUT_KEY_F12                       0x000C
#define  GLUT_KEY_LEFT                      0x0064
#define  GLUT_KEY_UP                        0x0065
#define  GLUT_KEY_RIGHT                     0x0066
#define  GLUT_KEY_DOWN                      0x0067
#define  GLUT_KEY_PAGE_UP                   0x0068
#define  GLUT_KEY_PAGE_DOWN                 0x0069
#define  GLUT_KEY_HOME                      0x006A
#define  GLUT_KEY_END                       0x006B
#define  GLUT_KEY_INSERT                    0x006C

#endif

namespace gpuppur
{

class GPUPPURUTBase : public boost::noncopyable
{
protected:
		static const int	glutSpecialKeyOffset = 128;

public:

	//各関数の戻り値
	struct ReturnCode
	{
		enum returnCode
		{
			OK = 0,				//No Error
			UnknownErr,			//未知のエラーが発生した。 
			InvalidCall,		//関数の無効な呼び出し。基本的に関数の引数や呼び出すタイミングが悪いとこのエラーがでる。 
			NotInitialized,		//このオブジェクトが初期化されていないのにメンバ関数が呼ばれた。 
			Invisible,			//画面に表示されていないor表示できない状態。 
			Visible,			//画面に表示されている状態。 
			FailCreatingWindow,	//ウィンドウの作成に失敗した。 

			INCORRECTVERSION,	//違うヴァージョンのランタイムがインストールされている。 
			CREATING3DDEVICE,	//3D deviceの初期化に失敗した。 
			NORUNTIME,			//ランタイムがインストールされていない。 
			FailedUserInit		//ユーザの初期化関数がfalseを返した。 
		};

		typedef unsigned int uint;

		ReturnCode(uint v):v(v){}

		operator uint()
		{
			return this->v;
		}

	private:
		uint v;
	};

	#define ENUMKEYCODE(key) key=GLUT_KEY_##key+GPUPPURUTBase::glutSpecialKeyOffset

	enum keycode
	{
		ENUMKEYCODE(F1),
		ENUMKEYCODE(F2),
		ENUMKEYCODE(F3),
		ENUMKEYCODE(F4),
		ENUMKEYCODE(F5),
		ENUMKEYCODE(F6),
		ENUMKEYCODE(F7),
		ENUMKEYCODE(F8),
		ENUMKEYCODE(F9),
		ENUMKEYCODE(F10),
		ENUMKEYCODE(F11),
		ENUMKEYCODE(F12),
		ENUMKEYCODE(PAGE_UP),
		ENUMKEYCODE(PAGE_DOWN),
		ENUMKEYCODE(HOME),
		ENUMKEYCODE(END),
		ENUMKEYCODE(LEFT),
		ENUMKEYCODE(UP),
		ENUMKEYCODE(RIGHT),
		ENUMKEYCODE(DOWN),
		ENUMKEYCODE(INSERT)
	};

#undef ENUMKEYCODE

	enum key_press
	{
		NONE = 0,
		KEY_DOWN,
		KEY_UP
	};

	enum mouse_button
	{
		MOUSE_LEFT = 1,
		MOUSE_RIGHT,
		MOUSE_MIDDLE
	};

	GPUPPURUTBase():
		isClosed(true)
	{
	}	

	bool	get_is_closed() const
	{
		return this->isClosed;
	}

protected:
	bool				isClosed;
	int					old_x, old_y, old_w, old_h;	///old_x, old_y, old_w and old_h is position of window and size of window respectively, before window change to fullscreen mode.
};

class gpuppurut_virtual : public GPUPPURUTBase
{
public:

	virtual ~gpuppurut_virtual(){};

	virtual ReturnCode	initialize(int width, int height, const std::string& title) = 0;
	virtual ReturnCode	initialize(int width, int height, const std::wstring& title) = 0;

	virtual void		uninitialize() = 0;

	/// 描画処理を開始します。
	/// Begin drawing.
	// If is_clear==false, this function doesnt clear color buffer and depth buffer.
	virtual bool		begin_scene(bool is_clear=true) = 0;
	/// 描画処理を終了し、画面へ出力します。
	/// End drawing and output to display.
	virtual ReturnCode	end_scene() = 0;

	virtual void		redraw() = 0;

	virtual bool		set_fullscreen(bool isFull) = 0;
	virtual bool		get_is_fullscreen() const = 0;

	virtual int			get_width() const = 0;
	virtual int			get_height() const = 0;

	/// You have to call handle_event() to update return value of isVisible()
	virtual ReturnCode	get_is_visible() const = 0;

	/*
	 * Return 3D graphics context.(e.g. IDirect3DDevice)
	 */
	virtual void*		get_context() const = 0;
};


class gpuppurut : public
#ifdef NO_VIRTUAL
	GPUPPURUTBase
#else
	gpuppurut_virtual
#endif
{
public:
	typedef boost::function<void (gpuppurut&)>										RedrawFunc;
	typedef boost::function<void (gpuppurut&, int width, int height)>				ResizeFunc;
	typedef	boost::function<void (gpuppurut&)>										Uninit3DContextFunc;
	typedef boost::function<bool (gpuppurut&)>										Init3DContextFunc;
	typedef boost::function<void (gpuppurut&, int key, int mouse_x, int mouse_y)>	KeyPressingFunc;
	typedef boost::function<void (gpuppurut&, int key, int mouse_x, int mouse_y)>	KeyUpFunc;
	typedef boost::function<void (gpuppurut&, int button, int state, int x, int y)>	MouseFunc;

	gpuppurut():
		redrawFunc(&emptyRedrawFunc),
		mouseFunc(&emptyMouseFunc),
		keyPressingFunc(&emptyKeyPressingFunc),
		keyUpFunc(&emptyKeyUpFunc),
		resizeFunc(&emptyResizeFunc),
		uninit3DContextFunc(&emptyUninit3DContextFunc),
		init3DContextFunc(&emptyInit3DContextFunc),
		isUninitializing(false)
	{
	}

	~gpuppurut()
	{
	}

	//この同じような関数が並んでいるのをなんとかできないものか。。#define使おうか・・・
	void	setRedrawFunc(RedrawFunc func)
	{
		this->redrawFunc = func;
	}

	void	setResizeFunc(ResizeFunc func)
	{
		this->resizeFunc = func;
	}

	void	setUninit3DContextFunc(Uninit3DContextFunc func)
	{
		this->uninit3DContextFunc = func;
	}

	void	setInit3DContextFunc(Init3DContextFunc func)
	{
		this->init3DContextFunc = func;
	}

	void	setKeyPressingFunc(KeyPressingFunc func)
	{
		this->keyPressingFunc = func;
	}

	void	setKeyUpFunc(KeyUpFunc func)
	{
		this->keyUpFunc = func;
	}

	void	setMouseFunc(MouseFunc func)
	{
		this->mouseFunc = func;
	}

protected:

	void uninit3DContext()
	{
		/*
			uninit3DContextFuncで呼び出される関数が、gpuppurutを持つクラスのデストラクタを呼ぶかもしれない。
			その時、gpuppurutのデストラクタが呼び出され、この関数が再び呼び出されるわけである。
		*/
		this->isUninitializing = true;
		this->uninit3DContextFunc(*this);
		this->isUninitializing = false;
	}

	bool getIsUninitializing() const
	{
		return this->isUninitializing;
	}

	RedrawFunc			redrawFunc;
	ResizeFunc			resizeFunc;
	Init3DContextFunc	init3DContextFunc;
	KeyPressingFunc		keyPressingFunc;
	KeyUpFunc			keyUpFunc;
	MouseFunc			mouseFunc;

private:

	Uninit3DContextFunc	uninit3DContextFunc;
	bool				isUninitializing;

private:
	static void emptyRedrawFunc(gpuppurut&){}
	static void emptyResizeFunc(gpuppurut&, int, int){}
	static void emptyUninit3DContextFunc(gpuppurut&){}
	static bool emptyInit3DContextFunc(gpuppurut&){return true;}
	static void emptyKeyPressingFunc(gpuppurut&, int, int, int){}
	static void emptyKeyUpFunc(gpuppurut&, int, int, int){}
	static void emptyMouseFunc(gpuppurut&, int, int, int, int){}
};

}	// end of namespace gpuppur

#endif

