#ifndef GPUPPUR_MESH_INDEX_BUFFER_HPP
#define GPUPPUR_MESH_INDEX_BUFFER_HPP

/**
 *	@file	
 *	@brief	Index buffer class
 *	@author	Tomohiro Matsumoto
 */
#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <boost/type.hpp>
#include <vector>
#include <gpuppur/utility/end_suppress_warnings.hpp>

namespace gpuppur
{
namespace index_buf_implement
{
	template<typename SrcType, typename DstType>
	void copy_to_index_buf
	(
		const void*			src,
		boost::type<SrcType>,
		unsigned int		stride,
		unsigned int		num_index,
		DstType*			dst,
		const unsigned int	num_vertices_per_triangle = 3
	)
	{
		DstType*	p_dst = dst;
		const char*	p_src = reinterpret_cast<const char*>(src);

		for(unsigned int i=0; i<num_index/num_vertices_per_triangle; ++i)
		{
			for(unsigned int j=0; j<num_vertices_per_triangle; ++j)
			{
				p_dst[j]
				=
				static_cast<DstType>
				(
					reinterpret_cast<const SrcType*>(p_src)[j]
				);
			}

			p_dst += num_vertices_per_triangle;
			p_src += stride;
		}
	}
}

class index_buffer
{
private:

public:

	index_buffer():
		num_index(0)
	{
	}

	template<typename SrcType>
	void load_data
	(
		unsigned int	num_vertices,
		const void*		src,
		boost::type<SrcType>,
		unsigned int	stride,
		unsigned int	num_index
	)
	{
		if(num_vertices > 0x7fff)
		{
			this->bytes_per_index = 4;
		}else
		{
			this->bytes_per_index = 2;
		}

		this->num_index = num_index;
		this->index_buf.resize(this->bytes_per_index*num_index);

		switch(this->bytes_per_index)
		{
		case 2:
			index_buf_implement::copy_to_index_buf
			(
				src, boost::type<SrcType>(), stride, num_index,
				reinterpret_cast<unsigned short*>(&this->index_buf[0])
			);
			break;

		case 4:
			index_buf_implement::copy_to_index_buf
			(
				src, boost::type<SrcType>(), stride, num_index,
				reinterpret_cast<unsigned int*>(&this->index_buf[0])
			);
			break;
		}
	}

	void* get_p_buf()
	{
		assert(!this->index_buf.empty());

		return &this->index_buf[0];
	}

	const void* get_p_buf() const
	{
		assert(!this->index_buf.empty());

		return &this->index_buf[0];
	}

	const unsigned int get_num_index() const
	{
		assert(!this->index_buf.empty());

		return this->num_index;
	}

	const unsigned int get_bytes_per_index() const
	{
		assert(!this->index_buf.empty());

		return this->bytes_per_index;
	}

	std::vector<unsigned char>	index_buf;
	unsigned int				num_index;
	unsigned int				bytes_per_index;
};

}	// end of namespace gpuppur

#endif

