#ifndef GPUPPUR_MESH_DIRECTX_HPP
#define GPUPPUR_MESH_DIRECTX_HPP

/**
 *	@file	
 *	@brief	Direct3D mesh class
 *	@author	Tomohiro Matsumoto
 */

#ifndef D3D_SDK_VERSION

namespace gpuppur
{
	class mesh_directx
	{
	public:

		typedef char sub_context_type;
	};
}

#else

#include <boost/call_traits.hpp>
#include <gpuppur/error_manage/error_handler.hpp>
#include <gpuppur/error_manage/error_report_dx.hpp>
#include <gpuppur/mesh/index_buffer.hpp>
#include <gpuppur/shader/hlsl.hpp>

namespace gpuppur
{

/**
 *	@brief Mesh manage class implemented with directx
 *
 */
class mesh_directx
{
public:

	struct dx_shader
	{
		dx_shader():
			is_initialized(false)
		{
		}

		~dx_shader()
		{
			this->uninitialize();
		}

		void uninitialize()
		{
			if(!this->is_initialized)
				return;

			this->hlsl.uninitialize();
		}

		bool initialize
		(
			IDirect3DDevice9*  context,
			const std::string& vert_shader_fname,
			const std::string& frag_shader_fname
		)
		{
			this->uninitialize();

			if
			(
				!this->hlsl.initialize
				(
					vert_shader_fname,
					frag_shader_fname,
					context
				)
			)
			{
				return false;
			}

			if
			(
				!(
					this->vert_mat_handle
					=
					this->hlsl.get_uniform_handle("vert_mat")
				)
			)
			{
				return false;
			}
				
			if
			(
				!(
					this->normal_mat_handle
					=
					this->hlsl.get_uniform_handle("normal_mat")
				)
			)
			{
				return false;
			}

			if
			(
				!(
					this->diffuse_handle
					=
					this->hlsl.get_uniform_handle("diffuse")
				)
			)
			{
				return false;
			}

			return this->is_initialized = true;
		}

		void bind_trans_matrix(const matrix4x4& model_mat)
		{
			matrix4x4 model_view_mat(this->view_mat*model_mat);

			this->hlsl.set_uniform
			(
				this->vert_mat_handle,
				this->proj_mat * model_view_mat
			);

			this->hlsl.set_uniform
			(
				this->normal_mat_handle,
				model_view_mat
			);
		}

		void bind_material(const vector3& mtrl)
		{
			this->hlsl.set_uniform
			(
				this->diffuse_handle,
				&mtrl[0]
			);
		}

		HLSL 					hlsl;
		HLSL::uniform_handle	vert_mat_handle;
		HLSL::uniform_handle	normal_mat_handle;
		HLSL::uniform_handle	diffuse_handle;
		matrix4x4				proj_mat;
		matrix4x4				view_mat;

	private:

		bool					is_initialized;
	};

	typedef dx_shader sub_context_type;
	typedef boost::call_traits<sub_context_type>::param_type
			sub_context_param_type;

	struct vert_element
	{
		vector3 pos;
		vector3 normal;
	};

	~mesh_directx()
	{
		this->release();
	}

	void release()
	{
		if(!this->has_mesh)
		{
			return;
		}

		SAFE_RELEASE(this->vert_buf);
		SAFE_RELEASE(this->index_buf);
		SAFE_RELEASE(this->vert_decl);

		this->has_mesh = false;
	}

	template<class MeshFeeder>
	mesh_directx
	(
		const MeshFeeder&		mesh_feeder,
		IDirect3DDevice9*		context,
		sub_context_param_type
	):
		vert_buf(0),
		index_buf(0),
		vert_decl(0),
		has_mesh(false)
	{
		assert(context);

		if(!mesh_feeder)
		{
			return;
		}

		std::vector<vector3>			vertices;
		std::vector<vector3>			normals;
		std::vector<vector2>			uv0_coords;
		gpuppur::index_buffer			indices;

		mesh_feeder.get_mesh_data(vertices, normals, uv0_coords, indices);

		assert
		(
			(indices.get_bytes_per_index() == 2
			||
			indices.get_bytes_per_index() == 4)
			&&
			"Only 16bits and 32bits index are supported by mesh_opengl"
		);

		//Not yet support texture coordinate...

		std::size_t numvert = vertices.size();

		this->num_vertices = static_cast<UINT>(numvert);
		this->num_triangles = static_cast<UINT>(indices.get_num_index()/3);

		HRESULT hr;

		hr = context->CreateVertexBuffer
		(
			static_cast<UINT>
			(
				sizeof(vert_element)*vertices.size()
			),
			D3DUSAGE_WRITEONLY,
			0,	//non-FVF
			D3DPOOL_MANAGED,
			&this->vert_buf,
			NULL
		);
		if
		(
			hr == D3DERR_OUTOFVIDEOMEMORY
			||
			hr == E_OUTOFMEMORY
		)
		{
			ERRHNDLR_NOT_ENOUGH_MEMORY_TO_STORE_VERTICES
			(
				sizeof(vert_element)*vertices.size()
			);

			return;
		}
		V(("CreateVertexBuffer", hr));

		vert_element*	pvert_buf;
		V(this->vert_buf->Lock
		(
			0,
			0,
			reinterpret_cast<void**>(&pvert_buf),
		//	D3DLOCK_DISCARD
			0
		));

		for(std::size_t i=0; i<vertices.size(); ++i)
		{
			pvert_buf->pos = vertices[i];
			pvert_buf->normal = normals[i];
			pvert_buf++;
		}

		V(this->vert_buf->Unlock());

		UINT index_buf_size = static_cast<UINT>
		(
			indices.get_num_index()*indices.get_bytes_per_index()
		);
		hr = context->CreateIndexBuffer
		(
			index_buf_size,
			D3DUSAGE_WRITEONLY,
			indices.get_bytes_per_index()==2 ? D3DFMT_INDEX16 : D3DFMT_INDEX32,
			D3DPOOL_MANAGED,
			&this->index_buf,
			NULL
		);
		if
		(
			hr == D3DERR_OUTOFVIDEOMEMORY
			||
			hr == E_OUTOFMEMORY
		)
		{
			ERRHNDLR_NOT_ENOUGH_MEMORY_TO_STORE_TRIANGLE_INDICES(index_buf_size);

			return;
		}else
		if(hr == D3DXERR_INVALIDDATA)
		{
			ERRHNDLR_INVALID_TRIANGLE_MESH();

			return;
		}
		V(("CreateIndexBuffer", hr));

		unsigned short* pindex_buf;
		V(this->index_buf->Lock
		(
			0,
			0,
			reinterpret_cast<void**>(&pindex_buf),
		//	D3DLOCK_DISCARD
			0
		));

		memcpy
		(
			reinterpret_cast<void*>(pindex_buf),
			indices.get_p_buf(),
			index_buf_size
		);

		V(this->index_buf->Unlock());

		D3DVERTEXELEMENT9 vert_elem_decl[] =
		{
			{
				0, 0, D3DDECLTYPE_FLOAT3,
				D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_POSITION, 0
			},
			{
				0, 12, D3DDECLTYPE_FLOAT3,
				D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_NORMAL, 0
			},
			D3DDECL_END()
		};
		V(context->CreateVertexDeclaration
		(
			vert_elem_decl,
			&this->vert_decl
		));

		this->has_mesh = true;
	}

	mesh_directx():
		vert_buf(0),
		index_buf(0),
		vert_decl(0),
		has_mesh(false)
	{
	}

	bool get_has_mesh() const
	{
		return this->has_mesh;
	}

	void bind_mesh
	(
		IDirect3DDevice9*		context,
		sub_context_param_type
	)
	{
		assert(context);
		assert(this->vert_buf);

		V(context->SetVertexDeclaration(this->vert_decl));
		V(context->SetStreamSource(0, this->vert_buf, 0, sizeof(vert_element)));
		V(context->SetIndices(this->index_buf));
	}

	void draw
	(
		const matrix4x4&	mat,
		const vector3&		diffuse,
		IDirect3DDevice9*	context,
		sub_context_type&	shader
	)
	{
		assert(this->vert_buf);
		assert(context);

		shader.bind_trans_matrix(mat);
		shader.bind_material(diffuse);

		V(context->DrawIndexedPrimitive
		(
			D3DPT_TRIANGLELIST,
			0,
			0,
			this->num_vertices,
			0,
			this->num_triangles
		));
/*
		V(context->SetTransform
		(
			D3DTS_VIEW,
			&save_mat
		));*/
	}

	static void set_state
	(
		const vector3& pos,
		const vector3& front,
		const vector3& up,
		IDirect3DDevice9*	context,
		sub_context_type&	shader
	)
	{
		assert(context);

		/*
		D3DXMATRIX mat;

		D3DXMatrixLookAtRH
		(
			&mat,
			reinterpret_cast<const D3DXVECTOR3*>(&pos),
			reinterpret_cast<const D3DXVECTOR3*>(&front),
			reinterpret_cast<const D3DXVECTOR3*>(&up)
		);
		*/

		//Same matrix with Opengl implement:)
		matrixRxC<float, 4, 4, false>
		mat(gpuppur::get_view_matrix<float, false>(pos, front, up));
/*
		V(context->SetTransform
		(
			D3DTS_VIEW,
			reinterpret_cast<D3DXMATRIX*>(&mat)
		));
*/
		shader.view_mat = mat;
		V(context->SetRenderState
		(
			D3DRS_CULLMODE,
			D3DCULL_CW
		));
		
		shader.hlsl.bind_shader();
	}

	static void set_viewport
	(
		int x,
		int y,
		int width,
		int height,
		IDirect3DDevice9*	context,
		sub_context_type&
	)
	{
		assert(context);

		D3DVIEWPORT9 vp = {x, y, width, height, 0.0f, 1.0f};
		V(context->SetViewport(&vp));
	}

	static void set_projection
	(
		IDirect3DDevice9*,
		sub_context_type&				sub_context,
		const gpuppur::lens<GLfloat>&	lns
	)
	{
		matrix4x4 mat;

		if(lns.is_perspective())
		{
			D3DXMatrixPerspectiveOffCenterRH
			(
				reinterpret_cast<D3DXMATRIX*>(&mat),
				lns.get_left(),
				lns.get_right(),
				lns.get_bottom(),
				lns.get_top(),
				lns.get_near(),
				lns.get_far()
			);
		}else
		{
			D3DXMatrixOrthoOffCenterRH
			(
				reinterpret_cast<D3DXMATRIX*>(&mat),
				lns.get_left(),
				lns.get_right(),
				lns.get_bottom(),
				lns.get_top(),
				lns.get_near(),
				lns.get_far()
			);
		}

		sub_context.proj_mat = mat;
	}

protected:
	IDirect3DVertexBuffer9*			vert_buf;
	IDirect3DIndexBuffer9*			index_buf;
	IDirect3DVertexDeclaration9*	vert_decl;
//	GLuint		normal_buf;
	UINT		num_vertices;
	UINT		num_triangles;
	bool		has_mesh;
};

}	// end of namespace gpuppur

#endif
#endif
