#ifndef GPUPPUR_MESH_OPENGL_HPP
#define GPUPPUR_MESH_OPENGL_HPP

/**
 *	@file	
 *	@brief	OpenGL mesh class
 *	@author	Tomohiro Matsumoto
 */
#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <boost/call_traits.hpp>
#include <boost/mpl/int.hpp>
#include <boost/mpl/void.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/3dmath/matrixRxC.hpp>
#include <gpuppur/error_manage/error_handler.hpp>
#include <gpuppur/error_manage/error_report_gl.hpp>
#include <gpuppur/mesh/index_buffer.hpp>
#include <gpuppur/shader/glsl.hpp>
#include <gpuppur/utility/lens.hpp>

namespace gpuppur
{

namespace mesh_opengl_implement
{
	//something local objects...
}

/**
 *	@brief Mesh manage class implemented with OpenGL
 *
 */
class mesh_opengl
{
private:

	bool	load_mesh
	(
		const std::vector<vector3>&		vertices,
		const std::vector<vector3>&		normals,
		const std::vector<vector2>&		uv0_coords,
		const void*						p_index_buf,
		const std::size_t				num_index,
		const std::size_t				byte_per_index
	)
	{
		glGenBuffers(1, &this->vert_buf);
		glGenBuffers(1, &this->index_buf);

		GLsizeiptr vert_size = vertices.size()*sizeof(vector3);
		GLsizeiptr norm_size = normals.size()*sizeof(vector3);
		GLsizeiptr uv0_size  = uv0_coords.size()*sizeof(vector2);

		glBindBuffer(GL_ARRAY_BUFFER, this->vert_buf);
		glBufferData
		(
			GL_ARRAY_BUFFER,
			vert_size + norm_size + uv0_size,
			NULL,
			GL_STATIC_DRAW
		);
		if(!GL_ERROR_CHECK((GL_OUT_OF_MEMORY)))
		{
			ERRHNDLR_NOT_ENOUGH_MEMORY_TO_STORE_VERTICES
			(
				vert_size + norm_size + uv0_size
			);

			this->release();
			return false;
		}
		glBufferSubData(GL_ARRAY_BUFFER, 0, vert_size, &vertices[0]);
		glBufferSubData(GL_ARRAY_BUFFER, vert_size, norm_size, &normals[0]);
		if(uv0_size)
		{
			glBufferSubData
			(
				GL_ARRAY_BUFFER,
				vert_size+norm_size,
				uv0_size,
				&uv0_coords[0]
			);

			this->has_uv0 = true;
		}

		this->vert_size = vert_size;
		this->norm_size = norm_size;

		this->num_indices = static_cast<GLsizei>(num_index);

		glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->index_buf);
		glBufferData
		(
			GL_ELEMENT_ARRAY_BUFFER,
			this->num_indices*byte_per_index,
			p_index_buf,
			GL_STATIC_DRAW
		);
		if(!GL_ERROR_CHECK((GL_OUT_OF_MEMORY)))
		{
			ERRHNDLR_NOT_ENOUGH_MEMORY_TO_STORE_TRIANGLE_INDICES
			(
				this->num_indices*byte_per_index
			);

			this->release();
			return false;
		}

		return true;
	}

public:

	struct gl_shader
	{
		gl_shader():
			is_initialized(false)
		{
		}

		~gl_shader()
		{
			this->uninitialize();
		}

		void uninitialize()
		{
			if(!this->is_initialized)
				return;

			this->glsl.uninitialize();
		}

		bool initialize
		(
			boost::mpl::void_ context,
			const std::string& vert_shader_fname,
			const std::string& frag_shader_fname
		)
		{
			this->uninitialize();

			if
			(
				!this->glsl.initialize
				(
					vert_shader_fname,
					frag_shader_fname,
					context
				)
			)
			{
				return false;
			}

			return this->is_initialized = true;
		}

		GLSL& get_shader()
		{
			return this->glsl;
		}

		const GLSL& get_shader() const
		{
			return this->glsl;
		}

		GLSL 	glsl;

	private:

		bool	is_initialized;
	};

	typedef gl_shader sub_context_type;
	typedef boost::call_traits<sub_context_type>::param_type
			sub_context_param_type;
	typedef gpuppur::mesh_opengl	this_type;

	~mesh_opengl()
	{
		this->release();
	}

	void release()
	{
		if(this->vert_buf)
		{
			glDeleteBuffers(1, &this->vert_buf);
			this->vert_buf = 0;

			GL_ERROR_ASSERT;
		}

		if(this->index_buf)
		{
			glDeleteBuffers(1, &this->index_buf);
			this->index_buf = 0;

			GL_ERROR_ASSERT;
		}
	}

public:

	template<class MeshFeeder>
	mesh_opengl
	(
		const MeshFeeder& mesh_feeder,
		boost::mpl::void_,
		sub_context_param_type
	):
		vert_buf(0),
		index_buf(0),
		has_uv0(false),
		has_mesh(false)
	{
		if(!mesh_feeder)
		{
			return;
		}
/*
		inr<byte_per_index> mesh_loader;
		void*		p_index;
		std::size_t	num_index;
*/
//		mesh_loader.feed_mesh(mesh_feeder, p_index, num_index);

		std::vector<vector3>			vertices;
		std::vector<vector3>			normals;
		std::vector<vector2>			uv0_coords;
		gpuppur::index_buffer			index_buf;

		mesh_feeder.get_mesh_data
		(
			vertices, normals, uv0_coords,
			index_buf
		);

		assert
		(
			(index_buf.get_bytes_per_index() == 2
			||
			index_buf.get_bytes_per_index() == 4)
			&&
			"Only 16bits and 32bits index are supported by mesh_opengl"
		);

		if
		(
			!this->load_mesh
			(
				vertices, normals,
				uv0_coords,
				index_buf.get_p_buf(), index_buf.get_num_index(),
				index_buf.get_bytes_per_index()
			)
		)
		{
			return;
		}

		switch(index_buf.get_bytes_per_index())
		{
		case 2:
			this->index_type=GL_UNSIGNED_SHORT;
			break;

		case 4:
			this->index_type = GL_UNSIGNED_INT;
			break;

		default:
			assert(false && "byte per index must 2 or 4");
			break;
		}

		this->has_mesh = true;
	}

	mesh_opengl():
		vert_buf(0),
		index_buf(0),
		has_uv0(false),
		has_mesh(false)
	{
	}

public:

	bool get_has_mesh() const
	{
		return this->has_mesh;
	}

	void bind_mesh
	(
		boost::mpl::void_,
		sub_context_param_type
	)
	{
	//	glEnable(GL_LIGHTING);
		glBindBuffer(GL_ARRAY_BUFFER, this->vert_buf);
		GL_ERROR_ASSERT;

		glVertexPointer(3, GL_FLOAT, 0, 0);
		GL_ERROR_ASSERT;

		glNormalPointer(GL_FLOAT, 0, (const void*)this->vert_size);
		GL_ERROR_ASSERT;

		if(this->has_uv0)
		{
			glClientActiveTexture(GL_TEXTURE0);
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glTexCoordPointer(2, GL_FLOAT, 0, (const void*)(this->vert_size+this->norm_size));

			GL_ERROR_ASSERT;
		}else
		{
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		}

		glBindBuffer(GL_ARRAY_BUFFER, 0);

		glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->index_buf);
		GL_ERROR_ASSERT;
	}

	void draw
	(
		const matrix4x4&	mat,
		const vector3&		diffuse,
		boost::mpl::void_,
		sub_context_param_type
	)
	{
		glPushMatrix();
		glMultMatrixf(mat.raw_data());

		vector4 dif(diffuse, 1.0f);
		glMaterialfv(GL_FRONT, GL_DIFFUSE, reinterpret_cast<GLfloat*>(&dif));

		glDrawElements(GL_TRIANGLES, this->num_indices, this->index_type, 0);
		GL_ERROR_ASSERT;

		glPopMatrix();
	}

	static void set_state
	(
		const vector3& pos,
		const vector3& front,
		const vector3& up,
		boost::mpl::void_,
		sub_context_type& shader
	)
	{
		matrixRxC<float, 4, 4, false>
		mat(gpuppur::get_view_matrix<float, false>(pos, front, up));

		glMatrixMode(GL_MODELVIEW);
		glLoadMatrixf(mat.raw_data());

		glEnableClientState(GL_VERTEX_ARRAY);
		glEnableClientState(GL_NORMAL_ARRAY);

		glEnable(GL_CULL_FACE);
		glEnable(GL_DEPTH_TEST);

	//	drow in wire frame mode.
	//	glPolygonMode(GL_FRONT, GL_LINE);
		GL_ERROR_ASSERT;

		shader.glsl.bind_shader();
	}

	static void set_viewport
	(
		int x,
		int y,
		int width,
		int height,
		boost::mpl::void_,
		sub_context_param_type
	)
	{
		glViewport(x, y, width, height);
	}

	static void set_projection
	(
		boost::mpl::void_,
		sub_context_param_type,
		const gpuppur::lens<GLfloat>&	lns
	)
	{
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		if(lns.is_perspective())
		{
			glFrustum
			(
				lns.get_left(),
				lns.get_right(),
				lns.get_bottom(),
				lns.get_top(),
				lns.get_near(),
				lns.get_far()
			);
		}else
		{
			glOrtho
			(
				lns.get_left(),
				lns.get_right(),
				lns.get_bottom(),
				lns.get_top(),
				lns.get_near(),
				lns.get_far()
			);
		}

		GL_ERROR_ASSERT;

		glMatrixMode(GL_MODELVIEW);
	}

protected:

	GLuint		vert_buf;
	GLuint		index_buf;
	bool		has_uv0;
	GLsizeiptr	vert_size;
	GLsizeiptr	norm_size;
	GLsizei		num_indices;
	bool		has_mesh;
	GLenum		index_type;
};

}	// end of namespace gpuppur

#endif

