#ifndef GPUPPUR_PHYSX_STREAM
#define GPUPPUR_PHYSX_STREAM

/**
 *	@file	
 *	@brief	Stream class used by PhysX API to serialize data.
 *	@author	Tomohiro Matsumoto
 */

#include <NxStream.h>

/** File stream
 *
 *	This class read/write data to file.
 */
class file_stream : public NxStream
{
public:
	file_stream(const std::string& filename, bool is_read):
		file
		(
			filename.c_str(),
			std::ios_base::binary | (is_read ? std::ios_base::in : (std::ios_base::out | std::ios_base::trunc))
		)
	{
	}

	virtual ~file_stream()
	{
	}

	virtual	NxU8		readByte() const
	{
		NxU8 ret;

		file >> ret;

		return ret;
	}

	virtual NxU16		readWord() const
	{

		NxU16 ret;

		file >> ret;

		return ret;
	}

	virtual NxU32		readDword() const
	{
		NxU32 ret;

		file >> ret;

		return ret;
	}

	virtual NxF32		readFloat() const
	{
		NxF32 ret;

		file >> ret;

		return ret;
	}

	virtual NxF64		readDouble() const
	{
		NxF64 ret;

		file >> ret;

		return ret;
	}

	virtual void		readBuffer(void* buffer, NxU32 size) const
	{
		file.read(reinterpret_cast<char*>(buffer), size);
	}

	virtual NxStream&	storeByte(NxU8 b)
	{
		file << b << std::endl;

		return *this;
	}

	virtual NxStream&	storeWord(NxU16 w)
	{
		file << w << std::endl;

		return *this;
	}

	virtual NxStream&	storeDword(NxU32 d)
	{
		file << d << std::endl;

		return *this;
	}

	virtual NxStream&	storeFloat(NxReal f)
	{
		file << f << std::endl;

		return *this;
	}

	virtual NxStream&	storeDouble(NxF64 f)
	{
		file << f << std::endl;

		return *this;
	}

	virtual NxStream&	storeBuffer(const void* buffer, NxU32 size)
	{
		file.write(reinterpret_cast<const char*>(buffer), size);

		return *this;
	}

protected:
	mutable std::fstream file;
};

/** File stream
 *
 *	This class read/write data to file.
 */
class UserStream : public NxStream
{
public:

	UserStream(const char* filename, bool load) : fp(NULL)
	{
		fp = fopen(filename, load ? "rb" : "wb");
	}

	operator bool() const
	{
		return fp != NULL;
	}

	virtual				~UserStream()
	{
		if(fp)	fclose(fp);
	}

	virtual NxU8 readByte() const
	{
		NxU8 b;
		size_t r = fread(&b, sizeof(NxU8), 1, fp);
		NX_ASSERT(r);
		return b;
	}

	virtual NxU16 readWord() const
	{
		NxU16 w;
		size_t r = fread(&w, sizeof(NxU16), 1, fp);
		NX_ASSERT(r);
		return w;
	}

	virtual NxU32 readDword() const
	{
		NxU32 d;
		size_t r = fread(&d, sizeof(NxU32), 1, fp);
		NX_ASSERT(r);
		return d;
	}

	virtual float readFloat() const
	{
		NxReal f;
		size_t r = fread(&f, sizeof(NxReal), 1, fp);
		NX_ASSERT(r);
		return f;
	}

	virtual double readDouble() const
	{
		NxF64 f;
		size_t r = fread(&f, sizeof(NxF64), 1, fp);
		NX_ASSERT(r);
		return f;
	}

	virtual void readBuffer(void* buffer, NxU32 size)	const
	{
		size_t w = fread(buffer, size, 1, fp);
		NX_ASSERT(w);
	}

	// Saving API
	virtual NxStream& storeByte(NxU8 b)
	{
		size_t w = fwrite(&b, sizeof(NxU8), 1, fp);
		NX_ASSERT(w);
		return *this;
	}

	virtual NxStream& storeWord(NxU16 w)
	{
		size_t ww = fwrite(&w, sizeof(NxU16), 1, fp);
		NX_ASSERT(ww);
		return *this;
	}

	virtual NxStream& storeDword(NxU32 d)
	{
		size_t w = fwrite(&d, sizeof(NxU32), 1, fp);
		NX_ASSERT(w);
		return *this;
	}

	virtual NxStream& storeFloat(NxReal f)
	{
		size_t w = fwrite(&f, sizeof(NxReal), 1, fp);
		NX_ASSERT(w);
		return *this;
	}

	virtual NxStream& storeDouble(NxF64 f)
	{
		size_t w = fwrite(&f, sizeof(NxF64), 1, fp);
		NX_ASSERT(w);
		return *this;
	}

	virtual NxStream& storeBuffer(const void* buffer, NxU32 size)
	{
		size_t w = fwrite(buffer, size, 1, fp);
		NX_ASSERT(w);
		return *this;
	}

	FILE*			fp;
};

/** Memory stream
 *
 *	This class read/write data to memory.
 */
class memory_stream : public NxStream
{
public:

	memory_stream()
	{
	}

	virtual ~memory_stream()
	{
	}

	template<typename T>
	T read() const
	{
		T ret;

		this->readBuffer(&ret, sizeof(ret));

		return ret;
	}

	virtual NxU8 readByte() const
	{
		return *(this->itr++);
	}

	virtual NxU16 readWord() const
	{
		return this->read<NxU16>();
	}

	virtual NxU32 readDword() const
	{
		return this->read<NxU32>();
	}

	virtual NxF32 readFloat() const
	{
		return this->read<NxF32>();
	}

	virtual NxF64 readDouble() const
	{
		return this->read<NxF64>();
	}

	virtual void readBuffer(void* buffer, NxU32 size) const
	{
		memcpy
			(
			buffer,
			reinterpret_cast<const void*>(&(*this->itr)),
			size
			);

		this->itr += size;
	}

	virtual NxStream& storeByte(NxU8 b)
	{
		this->buf.push_back(b);

		return *this;
	}

	virtual NxStream& storeWord(NxU16 w)
	{
		return this->storeBuffer(&w, sizeof(w));
	}

	virtual NxStream& storeDword(NxU32 d)
	{
		return this->storeBuffer(&d, sizeof(d));
	}

	virtual NxStream& storeFloat(NxReal f)
	{
		return this->storeBuffer(&f, sizeof(f));
	}

	virtual NxStream& storeDouble(NxF64 f)
	{
		return this->storeBuffer(&f, sizeof(f));
	}

	virtual NxStream& storeBuffer(const void* buffer, NxU32 size)
	{
		this->buf.resize(this->buf.size()+size);

		memcpy
			(
			reinterpret_cast<void*>(&this->buf[this->buf.size()-size]),
			buffer,
			size
			);

		return *this;
	}

	void begin_read()
	{
		this->itr = this->buf.begin();
	}

protected:
	std::vector<NxU8> buf;
	mutable std::vector<NxU8>::iterator itr;
};

/*
file_stream::file_stream(const std::string& filename, bool is_read):
	file
	(
		filename.c_str(),
		std::ios_base::binary | (is_read ? std::ios_base::in : (std::ios_base::out | std::ios_base::trunc))
	)
{
}

file_stream::~file_stream()
{
}

NxU8		file_stream::readByte() const
{
	NxU8 ret;

	file >> ret;

	return ret;
}

NxU16		file_stream::readWord() const
{
	NxU16 ret;

	file >> ret;

	return ret;
}

NxU32		file_stream::readDword() const
{
	NxU32 ret;

	file >> ret;

	return ret;
}

NxF32		file_stream::readFloat() const
{
	NxF32 ret;

	file >> ret;

	return ret;
}

NxF64		file_stream::readDouble() const
{
	NxF64 ret;

	file >> ret;

	return ret;
}

void		file_stream::readBuffer(void* buffer, NxU32 size) const
{
	file.read(reinterpret_cast<char*>(buffer), size);
}

NxStream&	file_stream::storeByte(NxU8 b)
{
	file << b << std::endl;

	return *this;
}

NxStream&	file_stream::storeWord(NxU16 w)
{
	file << w << std::endl;

	return *this;
}

NxStream&	file_stream::storeDword(NxU32 d)
{
	file << d << std::endl;

	return *this;
}

NxStream&	file_stream::storeFloat(NxReal f)
{
	file << f << std::endl;

	return *this;
}

NxStream&	file_stream::storeDouble(NxF64 f)
{
	file << f << std::endl;

	return *this;
}

NxStream&	file_stream::storeBuffer(const void* buffer, NxU32 size)
{
	file.write(reinterpret_cast<const char*>(buffer), size);

	return *this;
}



UserStream::UserStream(const char* filename, bool load) : fp(NULL)
	{
	fp = fopen(filename, load ? "rb" : "wb");
	}

UserStream::~UserStream()
	{
	if(fp)	fclose(fp);
	}

NxU8 UserStream::readByte() const
	{
	NxU8 b;
	size_t r = fread(&b, sizeof(NxU8), 1, fp);
	NX_ASSERT(r);
	return b;
	}

NxU16 UserStream::readWord() const
	{
	NxU16 w;
	size_t r = fread(&w, sizeof(NxU16), 1, fp);
	NX_ASSERT(r);
	return w;
	}

NxU32 UserStream::readDword() const
	{
	NxU32 d;
	size_t r = fread(&d, sizeof(NxU32), 1, fp);
	NX_ASSERT(r);
	return d;
	}

float UserStream::readFloat() const
	{
	NxReal f;
	size_t r = fread(&f, sizeof(NxReal), 1, fp);
	NX_ASSERT(r);
	return f;
	}

double UserStream::readDouble() const
	{
	NxF64 f;
	size_t r = fread(&f, sizeof(NxF64), 1, fp);
	NX_ASSERT(r);
	return f;
	}

void UserStream::readBuffer(void* buffer, NxU32 size)	const
	{
	size_t w = fread(buffer, size, 1, fp);
	NX_ASSERT(w);
	}

// Saving API
NxStream& UserStream::storeByte(NxU8 b)
	{
	size_t w = fwrite(&b, sizeof(NxU8), 1, fp);
	NX_ASSERT(w);
	return *this;
	}

NxStream& UserStream::storeWord(NxU16 w)
	{
	size_t ww = fwrite(&w, sizeof(NxU16), 1, fp);
	NX_ASSERT(ww);
	return *this;
	}

NxStream& UserStream::storeDword(NxU32 d)
	{
	size_t w = fwrite(&d, sizeof(NxU32), 1, fp);
	NX_ASSERT(w);
	return *this;
	}

NxStream& UserStream::storeFloat(NxReal f)
	{
	size_t w = fwrite(&f, sizeof(NxReal), 1, fp);
	NX_ASSERT(w);
	return *this;
	}

NxStream& UserStream::storeDouble(NxF64 f)
	{
	size_t w = fwrite(&f, sizeof(NxF64), 1, fp);
	NX_ASSERT(w);
	return *this;
	}

NxStream& UserStream::storeBuffer(const void* buffer, NxU32 size)
	{
	size_t w = fwrite(buffer, size, 1, fp);
	NX_ASSERT(w);
	return *this;
	}



NxU8 memory_stream::readByte() const
{
	return *(this->itr++);
}

NxU16 memory_stream::readWord() const
{
	return this->read<NxU16>();
}

NxU32 memory_stream::readDword() const
{
	return this->read<NxU32>();
}

NxF32 memory_stream::readFloat() const
{
	return this->read<NxF32>();
}

NxF64 memory_stream::readDouble() const
{
	return this->read<NxF64>();
}

void memory_stream::readBuffer(void* buffer, NxU32 size) const
{
	memcpy
	(
		buffer,
		reinterpret_cast<const void*>(&(*this->itr)),
		size
	);

	this->itr += size;
}

NxStream& memory_stream::storeByte(NxU8 b)
{
	this->buf.push_back(b);
	
	return *this;
}

NxStream& memory_stream::storeWord(NxU16 w)
{
	return this->storeBuffer(&w, sizeof(w));
}

NxStream& memory_stream::storeDword(NxU32 d)
{
	return this->storeBuffer(&d, sizeof(d));
}

NxStream& memory_stream::storeFloat(NxReal f)
{
	return this->storeBuffer(&f, sizeof(f));
}

NxStream& memory_stream::storeDouble(NxF64 f)
{
	return this->storeBuffer(&f, sizeof(f));
}

NxStream& memory_stream::storeBuffer(const void* buffer, NxU32 size)
{
	this->buf.resize(this->buf.size()+size);

	memcpy
	(
		reinterpret_cast<void*>(&this->buf[this->buf.size()-size]),
		buffer,
		size
	);

	return *this;
}

*/

#endif

