#ifndef SHADER_CG_HPP
#define SHADER_CG_HPP

/**
 *	@file	
 *	@brief	Manage shader programs Using Cg. Currently, implemented for OpenGL.
 *	@author	Tomohiro Matsumoto
 */

#include <string>
#include <Cg/cg.h>
#include <Cg/cgGL.h>

namespace gpuppur
{

class Cg
{
public:

	typedef CGparameter uniform_handle;

	Cg()
	:context(NULL), vert_program(NULL), frag_program(NULL)
	{
	}

	void uninitialize()
	{
		if(this->vert_program != NULL)
		{
			cgDestroyProgram(this->vert_program);
		}

		if(this->frag_program != NULL)
		{
			cgDestroyProgram(this->frag_program);
		}

		if(this->context != NULL)
		{
			cgDestroyContext(this->context);
		}
	}

	bool initialize(const std::string& vert_shader_fname, const std::string& frag_shader_fname)
	{
		this->uninitialize();

		this->context = cgCreateContext();

		if(this->context == NULL)
		{
			return false;
		}

		int a;

		switch(cgGLGetLatestProfile(CG_GL_FRAGMENT))
		{
		case CG_PROFILE_FP40:
			a=10;
			break;
		case CG_PROFILE_ARBFP1:
			a=11;
			break;
		default:
			a=0;
			break;
		}

		cgGLSetOptimalOptions(cgGLGetLatestProfile(CG_GL_FRAGMENT));

		this->frag_program = cgCreateProgramFromFile
							(
								this->context,
								CG_SOURCE,
							//	CG_OBJECT,
								(vert_shader_fname+".cg").c_str(),
								cgGLGetLatestProfile(CG_GL_FRAGMENT),
								"main",
								NULL
							);
		if(this->frag_program == NULL)
		{
			this->printLastError();
			fprintf(stderr, "Failed to compile fragment program!\n");

			return false;
		}
		if(!this->checkLastError())
		{
			return false;
		}

		cgGLLoadProgram(this->frag_program);
		if(!this->checkLastError())
		{
			return false;
		}

		cgGLSetOptimalOptions(cgGLGetLatestProfile(CG_GL_VERTEX));

		this->vert_program = cgCreateProgramFromFile
							(
								this->context,
								CG_SOURCE,
								(frag_shader_fname+".cg").c_str(),
								cgGLGetLatestProfile(CG_GL_VERTEX),
								NULL,
								NULL
							);
		if(this->vert_program == NULL)
		{
			this->printLastError();
			fprintf(stderr, "Failed to compile vertex program!\n");

			return false;
		}
		if(!this->checkLastError())
		{
			return false;
		}

		cgGLLoadProgram(this->vert_program);
		if(!this->checkLastError())
		{
			return false;
		}

		return true;
	}

	void bind_shader()
	{
		cgGLBindProgram(this->vert_program);
		cgGLBindProgram(this->frag_program);

		cgGLEnableProfile(cgGLGetLatestProfile(CG_GL_VERTEX));
		cgGLEnableProfile(cgGLGetLatestProfile(CG_GL_FRAGMENT));
	}

	void unbind_shader()
	{
	}

	template<typename Buffer>
	void set_uniform_texture(const std::string& name, const Buffer& texture)
	{
		this->set_uniform_texture(this->get_uniform_handle(name), texture);
	}

	void set_uniform(const std::string& name, const GLfloat value[3])
	{
		this->set_uniform(this->get_uniform_handle(name), value);
	}

	template<typename Buffer>
	void set_uniform_texture(Cg::uniform_handle handle, const Buffer& texture)
	{
		cgGLSetTextureParameter(handle, texture.GetTextureObjectName());
		cgGLEnableTextureParameter(handle);
	}

	void set_uniform(Cg::uniform_handle handle, const GLfloat value[3])
	{
		cgSetParameterValuefc(handle, 3, value);
	}

	Cg::uniform_handle get_uniform_handle(const std::string& name)
	{
		Cg::uniform_handle ret = cgGetNamedParameter(this->frag_program, name.c_str());

		return ret == NULL ? cgGetNamedParameter(this->vert_program, name.c_str()) : ret;
	}


	static void BindFixedFunction();


protected:

	void printLastError()
	{
		fprintf(stderr, "cgGetLastErrorString is:\n%s\n", cgGetLastErrorString(NULL));
		fprintf(stderr, "cgGetLastListing is:\n%s\n", cgGetLastListing(this->context));
		CGerror err = cgGetError();
		fprintf(stderr, "cgGetError is:\n%s\n", cgGetErrorString(err));
	}

	bool checkLastError()
	{
		CGerror err;

		const char* errString = cgGetLastErrorString(&err);

		if(errString != NULL)
		{
			fprintf(stderr, "cgGetLastListing is:\n%s\n", cgGetLastListing(this->context));
			fprintf(stderr, "cgGetLastErrorString is:\n%s\n", errString);
			return false;
		}

		return true;
	}


	CGcontext context;
	CGprogram vert_program;
	CGprogram frag_program;
};

}	//end of namespace gpuppur

#endif
