#ifndef SHADER_GLSL_HPP
#define SHADER_GLSL_HPP

/**
 *	@file	
 *	@brief	Manage shader program using GLSL language.
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <string>
#include <iostream>
#include <fstream>
#include <boost/mpl/void.hpp>
#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>

#include <gpuppur/utility/utility.hpp>
#include <gpuppur/error_manage/error_report_gl.hpp>
#include <gpuppur/error_manage/error_handler.hpp>
#include <gpuppur/3dmath/matrixRxC.hpp>

namespace gpuppur
{

/**	Manage shader program using GLSL language.
 *
 */
class GLSL
{
public:

	//typedef GLint uniform_handle;

	/** Handle class to uniform value in shader.
	 *
	 */
	struct uniform_handle
	{
		uniform_handle():
			handle(-1)
		{
		}

		uniform_handle(GLint handle):
			handle(handle)
		{
		}

		/** Check whether this handle is valid.
		 *
		 *	Don't use invalid handle to manipulate uniform value.
		 *
		 *	@retval true	:	Valid handle
		 *	@retval	false	:	Invalid handle.
		 */
		operator bool() const
		{
			return this->handle != -1;
		}

		operator GLint() const
		{
			return this->handle;
		}


	protected:

		GLint handle;
	};

	GLSL()
	:program(0)
	{
	}

	~GLSL()
	{
		this->uninitialize();
	}

	void uninitialize()
	{
		if(this->program == 0)
			return;

		glDeleteProgram(this->program);
		GL_ERROR_ASSERT;
		this->program = 0;
	}

	bool initialize
	(
		const std::string& vert_shader_fname,
		const std::string& frag_shader_fname,
		boost::mpl::void_
	)
	{
		this->uninitialize();

		GLuint vert_shader, frag_shader;

		vert_shader = glCreateShader(GL_VERTEX_SHADER);
		frag_shader = glCreateShader(GL_FRAGMENT_SHADER);

		if(vert_shader == 0 || frag_shader == 0)
		{
			std::cerr << "Failed to glCreateShader()" << std::endl;
			return false;
		}

		if(!read_shader(vert_shader, vert_shader_fname + ".vert"))
		{
			return false;
		}

		if(!read_shader(frag_shader, frag_shader_fname + ".frag"))
		{
			return false;
		}

		this->program = glCreateProgram();
		if(this->program == 0)
		{
			std::cerr << "Failed to glCreateProgram()" << std::endl;
			return false;
		}

		glAttachShader(this->program, vert_shader);
		glAttachShader(this->program, frag_shader);
		GL_ERROR_ASSERT;
		glDeleteShader(vert_shader);
		glDeleteShader(frag_shader);
		GL_ERROR_ASSERT;
		glLinkProgram(this->program);

		GLint log_len;
		glGetProgramiv(this->program, GL_INFO_LOG_LENGTH, &log_len);

		if(log_len != 0)
		{
			std::vector<char> buf(log_len);
			char* pbuf = &buf[0];

			glGetProgramInfoLog(this->program, log_len, NULL, pbuf);

			std::cerr << pbuf<< std::endl;
		}

		GLint success;
		glGetProgramiv(this->program, GL_LINK_STATUS, &success);
		if(success == GL_FALSE)
		{
			std::cerr << "Failed to link shader program" << std::endl;
			return false;
		}

		GL_ERROR_ASSERT;

		return true;
	}

	bool initialize
	(
		const std::wstring& vert_shader_fname,
		const std::wstring& frag_shader_fname,
		boost::mpl::void_
	)
	{
		std::vector<char> vert_mbstr;
		std::vector<char> frag_mbstr;

		gpuppur::to_char_from_wchar_t(vert_mbstr, vert_shader_fname.c_str());
		gpuppur::to_char_from_wchar_t(frag_mbstr, frag_shader_fname.c_str());

		return this->initialize(std::string(&vert_mbstr[0]), std::string(&frag_mbstr[0]), boost::mpl::void_());
	}

	void bind_shader()
	{
		glUseProgram(this->program);
		GL_ERROR_ASSERT;
	}

	void unbind_shader()
	{
		bind_fixed_function();
	}

	static void bind_fixed_function()
	{
		glUseProgram(0);
		GL_ERROR_ASSERT;
	}

#ifndef NDEBUG
	static bool get_false()
	{
		return false;
	}

	#define ASSERT_BINDING_SHADER						\
	do													\
	{													\
		GLuint current_shader;							\
		glGetIntegerv									\
		(												\
			GL_CURRENT_PROGRAM,							\
			reinterpret_cast<GLint*>(&current_shader)	\
		);												\
		assert											\
		(												\
			current_shader == this->program				\
			||											\
			!"You have to bind shader "					\
			"before set uniform value of the shader.!"	\
		);												\
	}while(get_false())

#else
		#define ASSERT_BINDING_SHADER
#endif

	template<typename Buffer>
	void set_uniform_texture(const std::string& name, const Buffer& texture)
	{
		this->set_uniform_texture(this->get_uniform_handle(name), texture);
		GL_ERROR_ASSERT;
	}

	void set_uniform(const std::string& name, const GLfloat value[3])
	{
		this->set_uniform(this->get_uniform_handle(name), value);
		GL_ERROR_ASSERT;
	}

	void set_uniform(const std::string& name, const GLfloat value)
	{
		this->set_uniform(this->get_uniform_handle(name), value);
		GL_ERROR_ASSERT;
	}

	template<typename Buffer>
	void set_uniform_texture(GLSL::uniform_handle handle, const Buffer& texture)
	{
		assert(handle);
		ASSERT_BINDING_SHADER;
		glUniform1i(handle, static_cast<GLint>(texture.GetTextureUnitSelector()));
		GL_ERROR_ASSERT;
	}

	void set_uniform(GLSL::uniform_handle handle, const GLfloat value[3])
	{
		assert(handle);
		ASSERT_BINDING_SHADER;
		glUniform3fv(handle, 1, value);
		GL_ERROR_ASSERT;
	}

	void set_uniform(GLSL::uniform_handle handle, GLfloat value)
	{
		assert(handle);
		ASSERT_BINDING_SHADER;
		glUniform1f(handle, value);
		GL_ERROR_ASSERT;
	}

	void set_uniform(GLSL::uniform_handle handle, int value)
	{
		assert(handle);
		ASSERT_BINDING_SHADER;
		glUniform1i(handle, value);
		GL_ERROR_ASSERT;
	}

	template<bool IsRowMajor>
	void set_uniform
	(
		GLSL::uniform_handle						handle,
		const matrixRxC<float, 4, 4, IsRowMajor>&	matrix_reloaded
	)
	{
		assert(handle);
		ASSERT_BINDING_SHADER;
		glUniformMatrix4fv
		(
			handle, 1,
			IsRowMajor ? GL_TRUE : GL_FALSE,
			matrix_reloaded.raw_data()
		);
		GL_ERROR_ASSERT;
	}

	template<bool IsRowMajor>
	void set_uniform
	(
		GLSL::uniform_handle						handle,
		const matrixRxC<float, 3, 3, IsRowMajor>&	matrix_reloaded
	)
	{
		assert(handle);
		ASSERT_BINDING_SHADER;
		glUniformMatrix3fv
		(
			handle, 1,
			IsRowMajor ? GL_TRUE : GL_FALSE,
			matrix_reloaded.raw_data()
		);
		GL_ERROR_ASSERT;
	}

	GLSL::uniform_handle get_uniform_handle(const std::string& name)
	{
		assert(this->program);
		return glGetUniformLocation(this->program, name.c_str());
	}

#undef ASSERT_BINDING_SHADER

protected:
	GLuint program;

	static	bool read_shader(GLuint shader, const std::string& filename)
	{
		std::ifstream ifs(filename.c_str(), std::ios_base::in);

		if(!ifs.good())
		{
			ERRHNDLR_FILE_NOT_FOUND(filename);
			return false;
		}

		std::ifstream::pos_type begin =  ifs.tellg();
		ifs.seekg(0, std::ios_base::end);
		std::ifstream::pos_type end =  ifs.tellg();
		ifs.seekg(0, std::ios_base::beg);

		{
			std::streamsize s = end-begin+1;
			std::vector<char> buf(s);
			char* pbuf = &buf[0];

			ifs.read(pbuf, s);
			buf[s-1] = 0;

			glShaderSource(shader, 1, (const GLchar**)&pbuf, &s);	
		}

		glCompileShader(shader);

		GLint log_len;
		glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &log_len);

		if(log_len != 0)
		{
			std::vector<char> buf(log_len);
			char* pbuf = &buf[0];

			glGetShaderInfoLog(shader, log_len, NULL, pbuf);

			std::cerr << pbuf<< std::endl;
		}

		GLint success;
		glGetShaderiv(shader, GL_COMPILE_STATUS, &success);
		if(success == GL_FALSE)
		{
			ERRHNDLR_FAILED_TO_LOAD_SHADER(filename);
			GL_ERROR_ASSERT;
			return false;
		}

		GL_ERROR_ASSERT;
		return true;
	}

#if 0	//gȂEEE
	template<typename Char>
	class glsl_extension;

	template<>
	class glsl_extension<char>
	{
	public:
		static const char* get_vert_extension()
		{
			return ".vert";
		}

		static const char* get_frag_extension()
		{
			return ".frag";
		}
	};

	template<>
	class glsl_extension<wchar_t>
	{
	public:
		static const wchar_t* get_vert_extension()
		{
			return L".vert";
		}
		static const wchar_t* get_frag_extension()
		{
			return L".frag";
		}
	};
#endif
};


}	//end of namespace gpuppur

#endif
