#ifndef SHADER_HLSL_HPP
#define SHADER_HLSL_HPP

/**
 *	@file	
 *	@brief	Manage shader program using HLSL language.
 *	@author	Tomohiro Matsumoto
 */

#ifndef D3D_SDK_VERSION

namespace gpuppur
{
	class HLSL;
}

#else


#include <string>
#include <gpuppur/error_manage/error_report_dx.hpp>
#include <gpuppur/error_manage/error_handler.hpp>
#include <gpuppur/3dmath/matrixRxC.hpp>

namespace gpuppur
{

class HLSL
{
public:

//	typedef D3DXHANDLE uniform_handle;
	struct uniform_handle
	{
		uniform_handle():
			handle(NULL)
		{
		}

		explicit uniform_handle(D3DXHANDLE handle):
			handle(handle)
		{
		}

		operator bool() const
		{
			return this->handle != NULL;
		}

		operator D3DXHANDLE() const
		{
			return this->handle;
		}

	protected:

		D3DXHANDLE handle;
	};

	HLSL():
		program(NULL),
		vert_shader(NULL)
	{
	}

	~HLSL()
	{
		this->uninitialize();
	}

	void uninitialize()
	{
		if(this->program == NULL)
			return;

		SAFE_RELEASE(this->vert_shader);
		SAFE_RELEASE(this->program);

	//	if(this->program != NULL)
		{
	//		V(this->program->OnResetDevice());
		}
	}

	template<typename Char>
	bool initialize
	(
		const std::basic_string<Char>& vert_shader_fname,
		const std::basic_string<Char>&,
		IDirect3DDevice9*  context
	)
	{
		this->uninitialize();

		ID3DXBuffer* compileErr = NULL;
		HRESULT ret;
		V( ret=create_effect_from_file
		(
			context,
			(vert_shader_fname+HLSL::hlsl_extension<Char>::get_extension()).c_str(),
			/*D3DXFX_NOT_CLONEABLE*/0, NULL,
			&this->program, &compileErr
		));

		// if there are compile erros, E_FAIL will be return.
		if(ret != E_FAIL && ret != S_OK)
		{
			if(ret == D3DXERR_INVALIDDATA)
			{
				ERRHNDLR_FILE_NOT_FOUND(vert_shader_fname);
			}

			ERRHNDLR_FAILED_TO_LOAD_SHADER(vert_shader_fname);

			return false;
		}

		if(compileErr != NULL)
		if(compileErr->GetBufferSize() > 0)
		{
			//Error message is always ascii code, not unicode.
			std::string msg
			(
				(const char*)compileErr->GetBufferPointer(),
				compileErr->GetBufferSize()
			);
		//	DebugPrint(msg.c_str());
			std::cerr << msg << std::endl;
			return false;
		}

		this->tech_handle = HLSL::uniform_handle(this->program->GetTechnique(0));
		if(this->tech_handle == NULL)
		{
			return false;
		}
#if 0
		D3DXHANDLE pass_handle = this->program->GetPass(tech_handle, 0);
		if(pass_handle == NULL)
		{
			return false;
		}
///*
	//	D3DXHANDLE param_handle = this->program->GetParameterByName(pass_handle, "VertexShader");
	//	B_RET(this->program->GetVertexShader(pass_handle, &this->vert_shader));
		B_RET(this->program->GetVertexShader(tech_handle, &this->vert_shader));

	//	param_handle = this->program->GetParameterByName(pass_handle, "PixelShader");
		B_RET(this->program->GetPixelShader(pass_handle, &this->frag_shader));
/*/
		D3DXHANDLE func_handle = this->program->GetFunctionByName("v_main");
		if(func_handle == NULL)
		{
			return false;
		}
		B_RET(this->program->GetVertexShader(func_handle, &this->vert_shader));


		func_handle = this->program->GetFunctionByName("p_main");
		if(func_handle == NULL)
		{
			return false;
		}
		B_RET(this->program->GetPixelShader(func_handle, &this->frag_shader));
//*/
#endif

		UINT npass;
		B_RET( this->program->Begin(&npass, D3DXFX_DONOTSAVESTATE) );
		B_RET( this->program->BeginPass(0) );

		B_RET( context->GetVertexShader(&this->vert_shader));

		B_RET( this->program->EndPass() );
		B_RET( this->program->End() );

		return true;
	}

	void bind_shader() const
	{
	//	IDirect3DDevice9* device;
	//	V( this->program->GetDevice(&device) );

	//	V( device->SetVertexShader(this->vert_shader) );
	//	V( device->SetPixelShader(this->frag_shader) );

	//	SAFE_RELEASE(device);

		//Activate all state(including vertex/pixel shader) of effect.
		//Begin() and End() dont save/restore any state,
		//so states of effect persists after bind_shader().
		UINT npass;
		V( this->program->Begin(&npass, D3DXFX_DONOTSAVESTATE) );
		V( this->program->BeginPass(0) );
		V( this->program->EndPass() );
		V( this->program->End() );
	}

	void unbind_shader()
	{
		IDirect3DDevice9* device;
		V( this->program->GetDevice(&device) );

		V( device->SetVertexShader(NULL) );
		V( device->SetPixelShader(NULL) );

		SAFE_RELEASE(device);
	}

	void begin_pass()
	{
		UINT npass;
		V( this->program->Begin(&npass, D3DXFX_DONOTSAVESTATE) );
		V( this->program->BeginPass(0) );
	}

	void end_pass()
	{
		V( this->program->EndPass() );
		V( this->program->End() );
	}

//	static void bind_fixed_function()
//	{
//	}

	template<typename Buffer>
	void set_uniform_texture(const std::string& name, const Buffer& texture)
	{
		this->set_uniform_texture(this->get_uniform_handle(name), texture);
	}

	void set_uniform(const std::string& name, const GLfloat value[3])
	{
		this->set_uniform(this->get_uniform_handle(name), value);
	}

	template<typename Buffer>
	void set_uniform_texture(HLSL::uniform_handle handle, const Buffer& texture)
	{
		V( this->program->SetTexture(handle, texture.get_texture_interface()) );
		this->update_uniform();
	}

	void set_uniform(HLSL::uniform_handle handle, const float value[3])
	{
		V( this->program->SetFloatArray(handle, value, 3) );
		this->update_uniform();
	}

	void set_uniform(HLSL::uniform_handle handle, float value)
	{
		V( this->program->SetFloat(handle, value) );
		this->update_uniform();
	}

	void set_uniform
	(
		HLSL::uniform_handle handle,
		const matrixRxC<float, 4, 4, false>& matrix
	)
	{
		V
		(
			this->program->SetMatrix
			(
				handle,
				reinterpret_cast<const D3DXMATRIX*>(matrix.raw_data())
			)
		);
		this->update_uniform();
	}

	void set_uniform
	(
		HLSL::uniform_handle handle,
		const matrixRxC<float, 3, 3, false>& matrix
	)
	{
		matrixRxC<float, 4, 4, false> tmp;
		set_sub(tmp, matrix, 0, 0);

		return this->set_uniform(handle, tmp);
	}

	HLSL::uniform_handle get_uniform_handle(const std::string& name)
	{
		return
			HLSL::uniform_handle
			(
				this->program->GetParameterByName(NULL, name.c_str())
			);
	}

protected:

	ID3DXEffect*				program;
	IDirect3DVertexShader9*		vert_shader;
	uniform_handle				tech_handle;

	template<typename Char>
	class hlsl_extension;

	template<>
	class hlsl_extension<char>
	{
	public:
		static const char* get_extension()
		{
			return ".fx";
		}
	};

	template<>
	class hlsl_extension<wchar_t>
	{
	public:
		static const wchar_t* get_extension()
		{
			return L".fx";
		}
	};

	void update_uniform()
	{
		if(this->program->GetCurrentTechnique() != this->tech_handle)
		{
			return;
		}

		this->bind_shader();
	}

	static HRESULT create_effect_from_file
	(
		LPDIRECT3DDEVICE9 context,
		const char* src_file,
		DWORD flags,
		LPD3DXEFFECTPOOL p_pool,
		LPD3DXEFFECT * ppEffect,
		LPD3DXBUFFER * ppCompilationErrors
	)
	{
		return D3DXCreateEffectFromFileA
		(
			context,
			src_file,
			NULL,
			NULL,
			flags,
			p_pool,
			ppEffect,
			ppCompilationErrors
		);
	}

	static HRESULT create_effect_from_file
	(
		LPDIRECT3DDEVICE9 context,
		const wchar_t* src_file,
		DWORD flags,
		LPD3DXEFFECTPOOL p_pool,
		LPD3DXEFFECT * ppEffect,
		LPD3DXBUFFER * ppCompilationErrors
	)
	{
		return D3DXCreateEffectFromFileW
		(
			context,
			src_file,
			NULL,
			NULL,
			flags,
			p_pool,
			ppEffect,
			ppCompilationErrors
		);
	}
};


}	//end of namespace gpuppur

#endif
#endif
