#ifndef TEXBUF_DX_TEX_BUF_HPP
#define TEXBUF_DX_TEX_BUF_HPP

/**
 *	@file	
 *	@brief	Texture class implemented with directx9.
 *	@author	Tomohiro Matsumoto
 */

#ifndef D3D_SDK_VERSION

namespace gpuppur
{
	template<class Group>
	class dx_texbuf;
}

#else

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <vector>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/texbuf/texbuf.hpp>
#include <gpuppur/error_manage/error_report_dx.hpp>

namespace gpuppur
{

class dx_texbuf_common : public texbuf_impl_base
{
protected:

	dx_texbuf_common()
	{
	}

	~dx_texbuf_common()
	{
	}

	void uninitialize()
	{
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector
	)
	{
		this->init_data(width, height, unit_selector);

		return true;
	}

public:

	typedef IDirect3DDevice9*	context_type;

	std::size_t GetTextureUnitSelector() const
	{
		return this->unit_selector;
	}
};

template<class Group>
class dx_texbuf_format_dep;

template<>
class dx_texbuf_format_dep<VectorNd<float, 2> > : public dx_texbuf_common
{
protected:
	static const D3DFORMAT dx_format = D3DFMT_G32R32F;
};

template<>
class dx_texbuf_format_dep<VectorNd<float, 4> > : public dx_texbuf_common
{
protected:
	static const D3DFORMAT dx_format = D3DFMT_A32B32G32R32F;
};

template<>
class dx_texbuf_format_dep<VectorNd<unsigned short, 4> > : public dx_texbuf_common
{
protected:
	static const D3DFORMAT dx_format = D3DFMT_A16B16G16R16;
};

template<>
class dx_texbuf_format_dep<VectorNd<unsigned short, 2> > : public dx_texbuf_common
{
protected:
	static const D3DFORMAT dx_format = D3DFMT_G16R16;
};

template<>
class dx_texbuf_format_dep<VectorNd<unsigned short, 1> > : public dx_texbuf_common
{
protected:
	static const D3DFORMAT dx_format = D3DFMT_L16;
};

template<class Group>
class dx_texbuf : public dx_texbuf_format_dep<Group>
{
protected:

	typedef dx_texbuf_format_dep<Group> base;

	dx_texbuf():
		texture(NULL)
	{
	}

	~dx_texbuf()
	{
	}

	void uninitialize()
	{
		base::uninitialize();
		SAFE_RELEASE(this->texture);
	}

	bool initialize
	(
		std::size_t			width,
		std::size_t			height,
		std::size_t			unit_selector,
		IDirect3DDevice9*	context
	)
	{
		if(!base::initialize(width, height, unit_selector))
		{
			return false;
		}

		B_RET
		(
			context->CreateTexture
			(
				static_cast<UINT>(width), static_cast<UINT>(height),
				1, D3DUSAGE_DYNAMIC, base::dx_format,
				D3DPOOL_DEFAULT, &this->texture, NULL
			)
		);

		return true;
	}

	void activate(IDirect3DDevice9*	context)
	{
		V
		(
			context->SetTexture
			(
				static_cast<DWORD>(this->unit_selector), this->texture
			)
		);
	}

	bool lock_for_write()
	{
		D3DLOCKED_RECT locked_rect;
		B_RET
		(
			this->texture->LockRect(0, &locked_rect, NULL, D3DLOCK_DISCARD)
		);

		this->p_buf = this->p_buf_begin
		=
		reinterpret_cast<Group*>(locked_rect.pBits);

		return true;
	}

	bool unlock()
	{
		B_RET(this->texture->UnlockRect(0));

		return true;
	}

	void set(const Group& vec)
	{
		assert
		(
			this->p_buf - this->p_buf_begin
			<
			static_cast<signed int>(this->width*this->height)
		);
		*this->p_buf = vec;
		++this->p_buf;
	}

	Group* get_buf_ptr()
	{
		return this->p_buf_begin;
	}

public:
	LPDIRECT3DBASETEXTURE9	get_texture_interface() const
	{
		return this->texture;
	}

protected:

	IDirect3DTexture9*	texture;
	Group *p_buf, *p_buf_begin;
};


}	//end of namespace gpuppur

#endif
#endif
