#ifndef TEXBUF_GL_TEX_BUF_HPP
#define TEXBUF_GL_TEX_BUF_HPP

/**
 *	@file	
 *	@brief	Texture class implemented with OpenGL.
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/utility/begin_suppress_warnings_from_others_code.hpp>
#include <vector>
#include <boost/mpl/void.hpp>
#include <gpuppur/utility/end_suppress_warnings.hpp>

#include <gpuppur/texbuf/texbuf.hpp>
#include <gpuppur/error_manage/error_report_gl.hpp>

namespace gpuppur
{

//This maybe needed to rewrite when new type buffer will be available in openGL2.x or 3.x
class gl_texbuf_common : public texbuf_impl_base
{
protected:

	GLenum tex_target;
	GLuint texture;

	gl_texbuf_common():
		tex_target
		(
			GL_TEXTURE_2D
			/*/
			GL_TEXTURE_RECTANGLE_EXT
			*/
		),
		texture(0)
	{
	}

	~gl_texbuf_common()
	{
	}

	void uninitialize()
	{
		if(this->texture == 0)
		{
			return;
		}

		GL_ERROR_ASSERT;
		glDeleteTextures(1, &this->texture);
		GL_ERROR_ASSERT;

		this->texture = 0;
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector
	)
	{
		this->init_data(width, height, unit_selector);

		return true;
	}

	static void activeTexture(std::size_t unit_selector)
	{
		static GLenum table[] = {
								GL_TEXTURE0,
								GL_TEXTURE1,
								GL_TEXTURE2,
								GL_TEXTURE3,
								GL_TEXTURE4,
								GL_TEXTURE5,
								GL_TEXTURE6,
								GL_TEXTURE7,
								GL_TEXTURE8,
								GL_TEXTURE9,
								GL_TEXTURE10,
								GL_TEXTURE11,
								GL_TEXTURE12,
								GL_TEXTURE13,
								GL_TEXTURE14,
								GL_TEXTURE15,
								GL_TEXTURE16,
								GL_TEXTURE17,
								GL_TEXTURE18,
								GL_TEXTURE19,
								GL_TEXTURE20,
								GL_TEXTURE21,
								GL_TEXTURE22,
								GL_TEXTURE23,
								GL_TEXTURE24,
								GL_TEXTURE25,
								GL_TEXTURE26,
								GL_TEXTURE27,
								GL_TEXTURE28,
								GL_TEXTURE29,
								GL_TEXTURE30,
								GL_TEXTURE31};


		assert(unit_selector >= 0 && unit_selector <= sizeof(table)/sizeof(GLenum));

		glActiveTexture(table[unit_selector]);
		GL_ERROR_ASSERT;
	}

public:

	typedef boost::mpl::void_	context_type;

	GLuint GetTextureObjectName() const
	{
		return this->texture;
	}

	std::size_t GetTextureUnitSelector() const
	{
		return this->unit_selector;
	}

	void attach_frame_buffer_object() const
	{
		::glFramebufferTexture2DEXT
		(
			GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT,
			this->tex_target,
			this->texture,
			0
		);
		::glViewport
		(
			0, 0,
			static_cast<GLsizei>(this->width), static_cast<GLsizei>(this->height)
		);

 		GL_ERROR_ASSERT;
	}

	void set_linear_filter()
	{
		glTexParameteri(this->tex_target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(this->tex_target, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	}
};

template<class Group>
class gl_texbuf_format_dep;

template<>
class gl_texbuf_format_dep<VectorNd<GLfloat, 2> > : public gl_texbuf_common
{
protected:

	static const GLenum gl_internal_format = GL_LUMINANCE16_ALPHA16;
	static const GLenum gl_type = GL_FLOAT;
	static const GLenum gl_format = GL_LUMINANCE_ALPHA;
};

template<>
class gl_texbuf_format_dep<VectorNd<GLfloat, 3> > : public gl_texbuf_common
{
protected:

	static const GLenum gl_internal_format = GL_RGB16;
	static const GLenum gl_type = GL_FLOAT;
	static const GLenum gl_format = GL_RGB;
};

template<>
class gl_texbuf_format_dep<VectorNd<GLfloat, 4> > : public gl_texbuf_common
{
protected:

	static const GLenum gl_internal_format = GL_RGBA16;
	static const GLenum gl_type = GL_FLOAT;
	static const GLenum gl_format = GL_RGBA;
};

template<>
class gl_texbuf_format_dep<VectorNd<unsigned short, 4> > : public gl_texbuf_common
{
protected:

	static const GLenum gl_internal_format = GL_RGBA16;
	static const GLenum gl_type = GL_UNSIGNED_SHORT;
	static const GLenum gl_format = GL_RGBA;
};

template<>
class gl_texbuf_format_dep<VectorNd<unsigned short, 2> > : public gl_texbuf_common
{
protected:

	static const GLenum gl_internal_format = GL_LUMINANCE16_ALPHA16;
	static const GLenum gl_type = GL_UNSIGNED_SHORT;
	static const GLenum gl_format = GL_LUMINANCE_ALPHA;
};

template<>
class gl_texbuf_format_dep<VectorNd<unsigned short, 1> > : public gl_texbuf_common
{
protected:

	static const GLenum gl_internal_format = GL_ALPHA16;
	static const GLenum gl_type = GL_UNSIGNED_SHORT;
	static const GLenum gl_format = GL_ALPHA;
};

/*
template<>
class gl_texbuf_format_dep<VectorNd<half, 3> > : public gl_texbuf_common
{
protected:

	static const GLenum gl_internal_format = GL_FLOAT_RGBA16_NV;
	static const GLenum gl_type = GL_HALF_FLOAT_NV;
	static const GLenum gl_format = GL_RGB;
};
*/

class gl_base_texture
{

};

/**
 *	Geforce7600GT, ForceWare version 85.96
 *	ではgl_texbufとgl_pbo_bufはほぼ同じ速さ。
 *	gl_pbo_sync_bufはgl_texbufより1.6倍ほど遅い。
 */
template<class Group>
class gl_texbuf : public gl_texbuf_format_dep<Group>
{
protected:

	typedef gl_texbuf_format_dep<Group> base;

	gl_texbuf()
	{
	}

	~gl_texbuf()
	{
	}

	void uninitialize()
	{
		base::uninitialize();
		this->buffer.clear();
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector,
		boost::mpl::void_			//This argument exists for compatibility with directx version tex_buf.
	)
	{
		if(!base::initialize(width, height, unit_selector))
		{
			return false;
		}

		this->buffer.resize(width*height);
		activeTexture(this->unit_selector);
		glGenTextures(1, &this->texture);
		GL_ERROR_ASSERT;

		glBindTexture(this->tex_target, this->texture);
		glTexImage2D
		(
			this->tex_target, 0, base::gl_internal_format,
			static_cast<GLsizei>(width), static_cast<GLsizei>(height),
			0, base::gl_format, base::gl_type, &this->buffer[0]
		);
		GL_ERROR_ASSERT;

	//	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	//	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	//	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	//	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

		return true;
	}

	void activate(boost::mpl::void_)
	{
		activeTexture(this->unit_selector);
		glBindTexture(this->tex_target, this->texture);

	//	glPixelTransferf(GL_RED_SCALE, 0.5);
	//	glPixelTransferf(GL_RED_BIAS, 0.5);
	//	glPixelTransferf(GL_GREEN_SCALE, 0.5);
	//	glPixelTransferf(GL_GREEN_BIAS, 0.5);
	//	glPixelTransferf(GL_BLUE_SCALE, 0.5);
	//	glPixelTransferf(GL_BLUE_BIAS, 0.5);
	}

	bool lock_for_write()
	{
		this->itr = this->buffer.begin();
		return true;
	}

	bool unlock()
	{
		activeTexture(this->unit_selector);
		glBindTexture(this->tex_target, this->texture);
		GL_ERROR_ASSERT;

		glTexSubImage2D
		(
			this->tex_target, 0, 0, 0,
			static_cast<GLsizei>(this->width), static_cast<GLsizei>(this->height),
			base::gl_format, base::gl_type, &this->buffer[0]
		);
		GL_ERROR_ASSERT;

		return true;
	}

	void set(const Group& vec)
	{
		*this->itr = vec;
		++this->itr;
	}

	Group* get_buf_ptr()
	{
		return &this->buffer[0];
	}

	typedef std::vector<Group>	buffer_type;
	typedef typename buffer_type::iterator	buffer_itr;
	buffer_type	buffer;
	buffer_itr	itr;
};

// PBOとはPixel Buffer Objectの略なのだ
// PBO is stand for Pixel Buffer Object.
// Need GL_ARB_pixel_buffer_object or OpenGL 2.1 or higher.
template<class Group>
class gl_pbo_buf : public gl_texbuf_format_dep<Group>
{
protected:

	typedef gl_texbuf_format_dep<Group> base;

#if 0
	// プロセスの使用メモリの変化をタスクマネージャで確認したいとき、
	// この関数でプログラムを任意の場所で一定時間停止して、メモリ使用量を確認する。 
	static void count_down(const char* msg, std::size_t count)
	{
		do
		{
			printf("%s %d\n", msg, count);
			Sleep(1000);
		}while(--count);
	}
#endif

	gl_pbo_buf()
	{
	}

	~gl_pbo_buf()
	{
	}

	void uninitialize()
	{
		base::uninitialize();
		glDeleteBuffersARB(1, &this->pixel_buffer);
		GL_ERROR_ASSERT;
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector,
		boost::mpl::void_					//This argument exists for compatibility with directx version tex_buf.
	)
	{
		if(!base::initialize(width, height, unit_selector))
		{
			return false;
		}

		base::activeTexture(unit_selector);
		glGenTextures(1, &this->texture);
		glBindTexture(this->tex_target, this->texture);
		std::vector<Group> tmp_buf(this->width*this->height);
		glTexImage2D
		(
			this->tex_target, 0,
			base::gl_internal_format,
			static_cast<GLsizei>(width), static_cast<GLsizei>(height),
			0, base::gl_format, base::gl_type,
			&tmp_buf[0]
		);
		GL_ERROR_ASSERT;

		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	//	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	//	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);


		glGenBuffersARB(1, &this->pixel_buffer);
		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, this->pixel_buffer);
		glBufferDataARB
		(
			GL_PIXEL_UNPACK_BUFFER_ARB,
			width*height*sizeof(Group),
			NULL, GL_DYNAMIC_DRAW_ARB
		);
		GL_ERROR_ASSERT;


		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, 0);
		GL_ERROR_ASSERT;

		return true;
	}

	bool lock_for_write()
	{
		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, this->pixel_buffer);
		GL_ERROR_ASSERT;
		this->p_buf = this->p_buf_begin =
		reinterpret_cast<Group*>
		(
			glMapBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, GL_WRITE_ONLY_ARB)
		);

		GL_ERROR_ASSERT;

		if(this->p_buf_begin == NULL)
		{
			return false;
		}

		return true;
	}

	bool unlock()
	{
		glUnmapBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB);
		GL_ERROR_ASSERT;

		glBindTexture(this->tex_target, this->texture);

		GL_ERROR_ASSERT;

		glTexSubImage2D
		(
			this->tex_target, 0, 0, 0,
			static_cast<GLsizei>(this->width), static_cast<GLsizei>(this->height),
			base::gl_format, base::gl_type, NULL
		);

		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, 0);
		GL_ERROR_ASSERT;

		return true;
	}

	void set(const Group& vec)
	{
		assert
		(
			this->p_buf - this->p_buf_begin
			<
			static_cast<signed int>(this->width*this->height)
		);
		*this->p_buf = vec;
		++this->p_buf;
	}

	Group* get_buf_ptr()
	{
		return this->p_buf_begin;
	}

	void activate(boost::mpl::void_)
	{
		activeTexture(this->unit_selector);

		glBindTexture(this->tex_target, this->texture);
		GL_ERROR_ASSERT;
	}

protected:
	GLuint pixel_buffer;
	Group *p_buf, *p_buf_begin;
};

// PBOとはPixel Buffer Objectの略なのだ
// PBO is stand for Pixel Buffer Object.
// Need GL_ARB_pixel_buffer_object or OpenGL 2.1 or higher.
template<class Group>
class gl_pbo_sync_buf : public gl_texbuf_format_dep<Group>
{
protected:

	typedef gl_texbuf_format_dep<Group> base;

	gl_pbo_sync_buf()
	{
	}

	~gl_pbo_sync_buf()
	{
	}

	void uninitialize()
	{
		base::uninitialize();
		glDeleteBuffersARB(1, &this->pixel_buffer);
		GL_ERROR_ASSERT;
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector,
		boost::mpl::void_						//This argument exists for compatibility with directx version tex_buf.
	)
	{
		if(!base::initialize(width, height, unit_selector))
		{
			return false;
		}

		base::activeTexture(unit_selector);
		glGenTextures(1, &this->texture);
		glBindTexture(this->tex_target, this->texture);

		this->buffer.resize(width*height);
		glTexImage2D
		(
			this->tex_target, 0, base::gl_internal_format,
			static_cast<GLsizei>(width), static_cast<GLsizei>(height),
			0, base::gl_format, base::gl_type,
			NULL/*&this->buffer[0]*/
		);
		GL_ERROR_ASSERT;


		glGenBuffersARB(1, &this->pixel_buffer);
		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, this->pixel_buffer);
		glBufferDataARB
		(
			GL_PIXEL_UNPACK_BUFFER_ARB,
			width*height*sizeof(Group),
			NULL, GL_DYNAMIC_DRAW_ARB/*/GL_STREAM_DRAW_ARB*/
		);
		GL_ERROR_ASSERT;

	/*
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	/*/
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	//*/
		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, 0);
		GL_ERROR_ASSERT;

		return true;
	}

	bool lock_for_write()
	{
		this->itr = this->buffer.begin();
		return true;
	}

	bool unlock()
	{
		//test whether PBO translate data to texture properly.
	/*	float v = 0.0;
		for
		(
			this->itr = this->buffer.begin();
			this->itr != this->buffer.end();
			++this->itr
		)
		{
			*(this->itr) = Group(static_cast<Group::element_type>(v));
			v+= 65536.0f / (this->width*this->height);
		}
	*/

		glBindTexture(this->tex_target, this->texture);
		GL_ERROR_ASSERT;

		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, this->pixel_buffer);
		GL_ERROR_ASSERT;

		glBufferSubData
		(
			GL_PIXEL_UNPACK_BUFFER_ARB,
			0,	this->width*this->height*sizeof(Group),
			&this->buffer[0]
		);
		GL_ERROR_ASSERT;

	//This might be bug of driver in ForceWare93.71
	//	glPixelStorei(GL_UNPACK_ALIGNMENT, 8);
	//	glPixelStorei(GL_UNPACK_ROW_LENGTH, this->width);
	//	glPixelStorei(GL_UNPACK_IMAGE_HEIGHT, this->height*2);
		glTexSubImage2D
		(
			this->tex_target, 0, 0, 0,
			static_cast<GLsizei>(this->width), static_cast<GLsizei>(this->height),
			base::gl_format, base::gl_type, NULL
		);

		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, 0);
		GL_ERROR_ASSERT;

		return true;
	}

	void set(const Group& vec)
	{
		*this->itr = vec;
		++this->itr;
	}

	Group* get_buf_ptr()
	{
		return &this->buffer[0];
	}

	void activate(boost::mpl::void_)
	{
		activeTexture(this->unit_selector);

		glBindTexture(this->tex_target, this->texture);
		GL_ERROR_ASSERT;
	}

protected:

	GLuint pixel_buffer;
//	Group *p_buf, *p_buf_begin;

	typedef std::vector<Group>	buffer_type;
	typedef typename buffer_type::iterator	buffer_itr;
	buffer_type	buffer;
	buffer_itr	itr;
};

/*
// PDRとはPixel Data Rangeの略なのだ
class PDRBuffer : public Buffer
{
protected:
	GLuint pixel_buffer;

protected:
	PDRBuffer();
	~PDRBuffer();

public:

	bool Initialize(int width, int height, int unit_selector);
	void Uninitialize();
	void Activate();
};
*/

}	//end of namespace gpuppur

#endif
