#ifndef TEXBUF_TEXBUF_HPP
#define TEXBUF_TEXBUF_HPP

/**
 *	@file	
 *	@brief	Texture class.
 *	@author	Tomohiro Matsumoto
 */

#include <cassert>
#include <gpuppur/3dmath/vectorNd.hpp>
#include <gpuppur/utility/utility.hpp>

namespace gpuppur
{

class texbuf_impl_base
{
protected:
	std::size_t	width;
	std::size_t	height;
	std::size_t	unit_selector;
	bool		has_resource;

	texbuf_impl_base():
		has_resource(false)
	{
	}

	void init_data
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector
	)
	{
		this->width = width;
		this->height = height;
		this->unit_selector = unit_selector;
	}

	/**
	 *	Default implementation to lock_for_write()
	 */
	bool lock_for_write()
	{
		return true;
	}

	/**
	 *	Default implementation to unlock()
	 */
	bool unlock()
	{
		return true;
	}
};

template<class Group, template<class> class Implement>
class texbuf : public Implement<Group>
{
protected:

#ifndef NDEBUG
	bool locked;
#endif

#ifndef NDEBUG
	void set_lock(bool is_locked)
	{
		this->locked = is_locked;
	}

	#define check_locked(must_locked)	assert(this->locked == must_locked)

#else
	void set_lock(bool){}
	void check_locked(bool){}
#endif

public:

	typedef Implement<Group>	base;
	typedef Group				group_type;

	texbuf()
	{
		this->set_lock(false);
	}

	~texbuf()
	{
		this->uninitialize();
	}

	void uninitialize()
	{
		check_locked(false);

		if(!this->has_resource)
			return;

		base::uninitialize();

		this->has_resource = false;
		this->set_lock(false);
	}

	bool initialize
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector
	)
	{
		this->uninitialize();

		if(!base::initialize(width, height, unit_selector))
		{
			return false;
		}

		this->has_resource = true;

		return true;
	}

	template<typename Context>
	bool initialize
	(
		std::size_t width,
		std::size_t height,
		std::size_t unit_selector,
		Context		context
	)
	{
		this->uninitialize();

		if(!base::initialize(width, height, unit_selector, context))
		{
			return false;
		}

		this->has_resource = true;

		return true;
	}

	bool lock_for_write()
	{
		assert(this->has_resource);
		check_locked(false);

		if(!base::lock_for_write())
		{
			return false;
		}

		this->set_lock(true);

		return true;
	}

	bool unlock()
	{
		assert(this->has_resource);
		check_locked(true);

		if(!base::unlock())
		{
			return false;
		}

		this->set_lock(false);

		return true;
	}

	void set(const Group& vec)
	{
		assert(this->has_resource);
		check_locked(true);

		base::set(vec);
	}

	Group* get_buf_ptr()
	{
		assert(this->has_resource);
		check_locked(true);

		return base::get_buf_ptr();
	}

	template<typename Context>
	void activate(Context context)
	{
		assert(this->has_resource);
		check_locked(false);

		base::activate(context);
	}

	std::size_t get_width() const
	{
		return this->width;
	}

	std::size_t get_height() const
	{
		return this->height;
	}
};

}	// end of namespace gpuppur

#endif
