#ifndef UTILITY_LENS_HPP
#define UTILITY_LENS_HPP

/**
 *	@file	
 *	@brief	Lens class which define view frustum.
 *	@author	Tomohiro Matsumoto
 */

#include <gpuppur/3dmath/vectorNd.hpp>

namespace gpuppur
{

template<typename T=float>
class lens
{
public:

	lens():
		l(static_cast<T>(-1.0)), r(1.0), b(-1.0), t(1.0), n(1.0), f(256.0),
		s(0.0)
	{
	}

	lens(T left, T right, T bottom, T top, T neear, T faar, bool is_perspective=true):
		l(left), r(right), b(bottom), t(top), n(neear), f(faar),
		s(static_cast<T>
		(
			is_perspective ? 0.0 : 1.0
		))
	{
	}

	template<std::size_t N>
	const VectorNd<T, N> get_ray_dir
	(
		const VectorNd<T, N>&	front, 
		const VectorNd<T, N>&	side,
		const VectorNd<T, N>&	up,
		const T					x,
		const T					y
	)const
	{
		return VectorNd<T, N>
		(
				this->n*front
			+
					(
						((this->r-this->l)*x+this->r+this->l)/2*side
						+ ((this->t-this->b)*y+this->t+this->b)/2*up
					)
				*
					(1-this->s)
		).getNormalized();
	}

	template<std::size_t N>
	const VectorNd<T, N> get_ray_local_pos
	(
		const VectorNd<T, N>&	,//front, 
		const VectorNd<T, N>&	side,
		const VectorNd<T, N>&	up,
		const T					x,
		const T					y
	)const
	{
	//	return VectorNd<T, N>(0.0);

		return
		(
			(
				(this->r-this->l)*x+this->r+this->l
			)/2*side
			+
			(
				(this->t-this->b)*y+this->t+this->b
			)/2*up
		)
		*
		this->s;
	}

	void set_perspective(bool is_perspective=true)
	{
		this->s = is_perspective ? 0.0 : 1.0;
	}

	const T get_left() const
	{
		return this->l;
	}

	const T get_right() const
	{
		return this->r;
	}

	const T get_bottom() const
	{
		return this->b;
	}

	const T get_top() const
	{
		return this->t;
	}

	const T get_near() const
	{
		return this->n;
	}

	const T get_far() const
	{
		return this->f;
	}

	const bool is_perspective() const
	{
		return this->s == 0.0;
	}

protected:

	T l, r, b, t, n, f;
	T s;				//Scale view_point. If s==0, this is perspective. If s==1, this is orthographic.
};

}

#endif
