#ifndef GPUPPUR_UTILITY_SCRIPT_HPP

/**
 *	@file	
 *	@brief
 *	@author	Tomohiro Matsumoto
 */

#include <string>
#include <boost/type_traits.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/tuple/tuple.hpp>

extern "C"
{
#include "lua.h"
#include "lualib.h"
#include "lauxlib.h"
}

namespace gpuppur
{
	namespace scripts
	{
		namespace detail
		{
			template<typename T>
			inline T get_value(lua_State* lua, int i=-1, T default_value=T())
			{
				if(!lua_isnumber(lua, i))
				{
					return default_value;
				}

				return static_cast<T>(lua_tonumber(lua, i));
			}

			inline const std::string get_value
			(
				lua_State* lua, int i=-1, const std::string& default_value=std::string()
			)
			{
				return std::string();
			}
		}
	}

	class script
	{
	public:

		template<typename T>
		T get_value
		(
			const std::string& identifier,
			T default_value=T(),
			typename boost::enable_if<boost::is_arithmetic<T> >::type* = 0
		)
		{
			assert(state);

			this->load_value(identifier);
		/*
			if(!lua_isnumber(this->state, -1))
			{
				return default_value;
			}

			return static_cast<T>(lua_tonumber(this->state, -1));
		*/
			return
			gpuppur::scripts::detail::get_value<T>(this->state, -1, default_value);
		}

		bool get_value
		(
			const std::string& identifier,
			bool default_value=true
		);

		script():
			state(NULL),
			has_initialized(false)
		{
		}

		~script()
		{
			this->uninitialize();
		}

		void uninitialize()
		{
			if(!this->has_initialized)
			{
				return;
			}

			this->has_initialized = false;

			if(!this->state)
			{
				return;
			}

			lua_close(this->state);
		}

		bool initialize(const std::string& filename)
		{
			this->uninitialize();

			lua_State*	s = luaL_newstate();
			if(!s)
			{
				return false;
			}

			luaL_openlibs(s);

			if(luaL_dofile(s, filename.c_str()))
			{
				return false;
			}

			this->state = s;
			this->filename = filename;
			this->has_initialized = true;

			return true;
		}

		bool get_has_initialized() const
		{
			return this->has_initialized;
		}

	private:

		void load_value(const std::string& identifier)
		{
			assert(this->state);

			lua_getfield(this->state, LUA_GLOBALSINDEX, identifier.c_str());
		}

		lua_State*		state;
		std::string		filename;
		bool			has_initialized;
	};

	inline script& get_global_script()
	{
		static script sc;

		if(!sc.get_has_initialized())
		{
			bool ret = sc.initialize("../data/test_param.lua");
			assert(ret);
		}

		return sc;
	}

	template<class Tuple>
	class lua_funcs
	{
		typedef lua_funcs<Tuple> this_type;

		template<class Function>
		struct func_entry
		{
			typedef Function func_type;
			std::string name;
			Function	func;

			func_entry(const std::string name, const Function& func):
				name(name), func(func)
			{
			}
		};

		template<int N>
		static int lua_func(lua_State* lua)
		{
			template<typename T1, typename T2>
			struct funcer
			{
				template<class Function>
				static void call(const Function& func, lua_State* lua)
				{
					return func
					(
						gpuppur::script::detail::get_value<T1>(lua, 1),
						gpuppur::script::detail::get_value<T2>(lua, 2)
					);
				}
			};

			template<typename T1>
			struct funcer<T1, void>
			{
				template<class Function>
				static void call(const Function& func, lua_State* lua)
				{
					return func
					(
						gpuppur::script::detail::get_value<T1>(lua, 1)
					);
				}
			};

			template<>
			struct funcer<void, void>
			{
				template<class Function>
				static void call(const Function& func, lua_State* lua)
				{
					return func();
				}
			};

			typedef typename
			boost::tuples::element<N, this_type::func_list>::type::func_type
			function;

			funcer<function::arg1_type, function::arg2_type>::call
			(
				this_type::func_list.get<N>().func,
				lua
			);
		}

	private:

		struct setter
		{
		public:

			template<typename Arg>
			void operator()(Arg a, int index)
			{
				lua_register
				(
					this->lua,
					this_type::func_list.get<index>().name.c_str(),
					&this_type::lua_func<index>
				);
			}

			setter(lua_State* lua):
				lua(lua)
			{
			}

		private:

			lua_State* lua;
		};

		lua_funcs(const Tuple& func_list, const std::string& filename):
			has_err(false)
		{
			this_type::func_list = func_list;

			lua_State*	s = luaL_newstate();
			if(!s)
			{
				return this->has_err=true, void();
			}

			luaL_openlibs(s);

			gpuppur::tuple_for_each_with_index(this_type::func_list, setter(s));

			if(luaL_dofile(s, filename.c_str()))
			{
				return this->has_err=true, void();
			}

			lua_close(s);
		}

		bool get_has_err() const
		{
			return this->has_err;
		}

		bool has_err;
		static Tuple func_list;
	};

}	// end of namespace gpuppur

#endif
