#ifndef UTILITY_UTILITY_HPP
#define UTILITY_UTILITY_HPP

/**
 *	@file	
 *	@brief	Collection of a small useful class and functions.
 *	@author	Tomohiro Matsumoto
 */

#include <boost/mpl/void.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/tuple/tuple.hpp>

namespace gpuppur
{

template<class Base>
class nil : public Base
{
};

template<class InterfaceBase=boost::mpl::void_>
class tail : public InterfaceBase
{
protected:
	tail():
		is_initialized(false)
	{
	}

	~tail()
	{
		this->uninitialize();
	}

	bool failed()
	{
		return this->is_initialized = false;
	}

	void uninitialize()
	{
		this->is_initialized = false;
	}

	bool initialize()
	{
		this->is_initialized = true;
		return true;
	}

	bool initialize(std::size_t, std::size_t)
	{
		this->is_initialized = true;
		return true;
	}

	bool is_initialized;
};

#include <cwchar>
#include <vector>

inline void to_char_from_wchar_t(std::vector<char>& dst, const wchar_t* src)
{
		// get needed size to store multibyte string.
#if !defined _WIN32 || !defined _MSC_VER
	const int maxCharLen = 65535;
	std::size_t size = wcsrtombs(NULL, &src, maxCharLen, NULL);
#else
	std::size_t size;
	wcsrtombs_s(&size, NULL, 0, &src, _TRUNCATE, NULL);
	--size;	//this size include null terminate character.
#endif
	dst.resize(size+1);
	dst.back() = '\0';

#if !defined _WIN32 || !defined _MSC_VER
	size_t ret = wcsrtombs(&dst[0], &src, maxCharLen, NULL);
	assert(ret != static_cast<size_t>(-1));
#else
	errno_t ret = wcsrtombs_s(NULL, &dst[0], size+1, &src, _TRUNCATE, NULL);
	assert(ret == 0);
#endif
}

namespace tuple_for_each_impl
{
namespace detail
{
/**	class to implement gpuppur::tuple_for_each and gpuppur::tuple_for_each_with_index
 *
 */
template<class Function>
struct conv
{
	typedef conv<Function> this_type;
	conv(Function& f):
		f(f)
	{
	}

	template<typename T>
	void call(T& t)
	{
		this->f(t.get_head());

		this->call(t.get_tail());
	}

	template<typename T>
	void call(T& t, int n)
	{
		this->f(t.get_head(), n);

		this->call(t.get_tail(), n+1);
	}

	void call(boost::tuples::null_type)
	{
	}

	void call(boost::tuples::null_type, int)
	{
	}

private:
	this_type operator = (const this_type&);

	Function& f;
};
}	// end of namespace detail
}	// end of namespace tuple_for_each

/** Call function f() for each element in tuple.
 *
 *	This function works like
 *	f(t.get<0>());
 *	f(t.get<1>());
 *	f(t.get<2>());
 *	...
 *
 *	@param[in,out]	t	Instance of boost::tuple.
 *	@param[in]		f	Function object. This function must have all signature which can aceppt each element of t.
 */
template<class Tuple, class Function>
void tuple_for_each(Tuple& t, Function& f)
{
	gpuppur::tuple_for_each_impl::detail::conv<Function>(f).call(t);
}

/** Call function f() for each element in tuple.
 *
 *	This function works like
 *	f(t.get<0>(), 0);
 *	f(t.get<1>(), 1);
 *	f(t.get<2>(), 2);
 *	...
 *
 *	@param[in,out]	t	Instance of boost::tuple.
 *	@param[in]		f	Function object. This function must have all signature which can aceppt each element of t and a int value.
 */
template<class Tuple, class Function>
void tuple_for_each_with_index(Tuple& t, Function& f)
{
	gpuppur::tuple_for_each_impl::detail::conv<Function>(f).call(t, 0);
}

#if 0
#if 0
/** Force to increase reference count of boost::shared_ptr
 *	This functions must be joke.
 */
template<typename T>
boost::shared_ptr<T> fucking_force_to_increase_reference_count(const boost::shared_ptr<T>& ptr)
{
	void* tmp = malloc(sizeof(boost::shared_ptr<T>));

	new(tmp) boost::shared_ptr<T>(ptr);

	free(tmp);

	return boost::shared_ptr<T>(ptr);
}

#endif
#endif

}	//end of namespace gpuppur

#endif
