package osm.jp.osmtracker.tool;
import javax.xml.parsers.*;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.*;
import org.xml.sax.*;

import java.io.*;

public class Gpx2osm {

	/** メイン
	 * java -jar gpx2osm.jar
	 * 		変換テンプレートファイルを"temp.xml"として、カレントディレクトリにあるすべてのGPXファイルを変換します。
	 *
	 * java -jar gpx2osm.jar <temp.xml>
	 * 		指定された変換テンプレートファイルを<temp.xml>を使って、カレントディレクトリにあるすべてのGPXファイルを変換します。
	 *
	 * java -jar gpx2osm.jar <temp.xml> <GPXファイル1> <GPXファイル2> <GPXファイル3> ...
	 * 		指定された変換テンプレートファイルを<temp.xml>を使って、指定されたGPXファイルを変換します。
	 *
	 * @throws TransformerException
	 * @throws IOException
	 * @throws SAXException
	 * @throws ParserConfigurationException */
	public static void main(String[] argv) throws ParserConfigurationException, SAXException, IOException, TransformerException
	{
		File tempfile = new File("temp.xml");
		if (argv.length > 0) {
			tempfile = new File(argv[0]);
		}

		int counter = 0;
		if (argv.length > 1) {
			for (int i=1; i < argv.length; i++) {
				File iFile = new File(argv[i]);
				if (checkFile(iFile)) {
					counter++;
					Gpx2osm.proc(iFile, tempfile);
				}
			}
		}
		else {
			File dir = new File(".");
			File[] files = dir.listFiles();
			for (File iFile : files) {
				if (checkFile(iFile)) {
					counter++;
					Gpx2osm.proc(iFile, tempfile);
				}
			}
		}
		System.out.println("["+ counter +"]つのファイルを処理しました。");
	}

	public static Document document;
	public static void proc (File iFile, File tempfile) throws ParserConfigurationException, SAXException, IOException, TransformerException {
		int iCounter = 0;
		DocumentBuilderFactory factory;
		DocumentBuilder        builder;
		Node root;
		way = null;

		String fileName = iFile.getName();
		String iStr = fileName.substring(0, fileName.length() - 4);

		File outputFile = new File(iStr +".osm");
		System.out.println(iStr + " => "+ outputFile.getName());

		try {
			factory = DocumentBuilderFactory.newInstance();
			builder = factory.newDocumentBuilder();
			factory.setIgnoringElementContentWhitespace(true);
			factory.setIgnoringComments(true);
			factory.setValidating(true);

			// GPX file --> Node root
			root    = builder.parse(iFile).getFirstChild();

			// TEMP XML file --> Node temp
			Node temp    = builder.parse(tempfile).getFirstChild();

			// OSM file --> Node osmnode
			DOMImplementation domImpl = builder.getDOMImplementation();
			document = domImpl.createDocument("", "osm", null);
			Element osmnode = document.getDocumentElement();
			NamedNodeMap nodeMap = temp.getAttributes();
			if (null != nodeMap) {
				for (int j=0; j < nodeMap.getLength(); j++ ) {
					osmnode.setAttribute(nodeMap.item(j).getNodeName(), nodeMap.item(j).getNodeValue());
				}
			}

			iCounter = showNodes(osmnode, temp, root, iCounter, "");

			// 出力
			DOMSource source = new DOMSource(osmnode);
			FileOutputStream os = new FileOutputStream(outputFile);
			StreamResult result = new StreamResult(os);
			TransformerFactory transFactory = TransformerFactory.newInstance();
			Transformer transformer = transFactory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.transform(source, result);
		}
		finally {
			System.out.println("WPT数["+ iCounter +"]");
		}
	}

	public static int showNodes(Element osmnode, Node temp, Node node, int iCounter, String space) throws IOException {
		NodeList nodes = node.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("wpt")) {
				iCounter = showWPT(osmnode, temp, node2, iCounter);
			}
			else {
				iCounter = showNodes(osmnode, temp, node2, iCounter, space + "    ");
			}
		}
		return iCounter;
	}

	public static Element way = null;


	/**
	 * <wpt/> ノードを処理する
	 *
	 * 	<wpt lat="35.44804477" lon="139.6348507">
	 * 		<ele>48.70000076293945</ele>
	 * 		<time>2012-03-20T00:26:30Z</time>
	 * 		<name><![CDATA[Picture (6.0m)]]></name>
	 * 		<link href="2012-03-20_09-26-30.jpg">
	 * 			<text>2012-03-20_09-26-30.jpg</text>
	 * 		</link>
	 * 		<sat>9</sat>
	 * 	</wpt>
	 *
	 * @param w
	 * @param node
	 * @throws IOException
	 */
	public static int showWPT(Element osmnode, Node temp, Node node, int iCounter) throws IOException {
		String nameStr = "";
		String latStr = "";
		String lonStr = "";
		String timeStr = "";

		NamedNodeMap nodeMap = node.getAttributes();
		if ( null != nodeMap ) {
			for (int j=0; j<nodeMap.getLength(); j++ ) {
				if (nodeMap.item(j).getNodeName().equals("lat")) {
					latStr = nodeMap.item(j).getNodeValue();
				}
				else if (nodeMap.item(j).getNodeName().equals("lon")) {
					lonStr = nodeMap.item(j).getNodeValue();
				}
			}
		}

		NodeList nodes = node.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("time")) {
				timeStr = node2.getTextContent();
			}
			else if (node2.getNodeName().equals("name")) {
				nameStr = node2.getTextContent();
			}
		}

		if (nameStr.startsWith("Picture")) {
			System.out.println("'"+ nameStr +"' : スキップ");
		}
		else if (nameStr.startsWith("Voice recording")) {
			System.out.println("'"+ nameStr +"' : スキップ");
		}
		else {
			System.out.println("name : '"+ nameStr +"'");
			iCounter--;

			Element onode;
			if ((onode = getOsmNode(temp, nameStr)) != null) {
				System.out.println("'"+ nameStr +"' : <"+ onode.getNodeName() +" id=\""+ iCounter +"\" timestamp=\""+ timeStr +"\" lat=\""+ latStr +"\" lon=\""+ lonStr +"\">");
				if (onode.getNodeName().equals("node")) {
					// Template.xml からの回答が NODE の場合
					onode.setAttribute("id", Integer.toString(iCounter));
					onode.setAttribute("timestamp", timeStr);
					onode.setAttribute("lat", latStr);
					onode.setAttribute("lon", lonStr);
					onode.appendChild(getSourceTag(timeStr));
					osmnode.appendChild(onode);
					if (nameStr.startsWith("W:")) {
						if (way != null) {
							Element nd = document.createElement("nd");
							nd.setAttribute("ref", Integer.toString(iCounter));
							way.appendChild(getCopy(nd));
						}
					}
				}
				else {
					// Template.xml からの回答が WAY の場合
					// ノードを作成して、出力する。
					Element nod = document.createElement("node");
					nod.setAttribute("id", Integer.toString(iCounter));
					nod.setAttribute("timestamp", timeStr);
					nod.setAttribute("lat", latStr);
					nod.setAttribute("lon", lonStr);
					osmnode.appendChild(nod);

					// WAYを作成する
					iCounter--;
					if (way == null) {
						if (nameStr.equals("W:END")) {
							// WAY が記録されていないときに「W:END」を押しても何もしない
							way = null;
						}
						else {
							// 「W:END」以外が押された場合は、WAYエレメントを新規に作成して
							// WAYに先のNDノードを登録する
							way = document.createElement("way");
							way.setAttribute("id", Integer.toString(iCounter));
							Element nd = document.createElement("nd");
							nd.setAttribute("ref", Integer.toString(iCounter + 1));
							way.appendChild(getCopy(nd));
							way.appendChild(getSourceTag(timeStr));

							// 回答ノードの内容をWAYに転記する
							NodeList nlist = onode.getChildNodes();
							for (int i=0; i < nlist.getLength(); i++) {
								way.appendChild(getCopy(nlist.item(i)));
							}
						}
					}
					else {
						way.setAttribute("id", Integer.toString(iCounter));
						Element nd = document.createElement("nd");
						nd.setAttribute("ref", Integer.toString(iCounter + 1));
						way.appendChild(getCopy(nd));

						if (updateWay(osmnode, onode)) {
							iCounter--;
							way = document.createElement("way");
							way.setAttribute("id", Integer.toString(iCounter));
							way.appendChild(getCopy(nd));
							way.appendChild(getSourceTag(timeStr));
							NodeList nlist = onode.getChildNodes();
							for (int i=0; i < nlist.getLength(); i++) {
								way.appendChild(getCopy(nlist.item(i)));
							}
						}
						else {
							NodeList nlist = onode.getChildNodes();
							for (int i=0; i < nlist.getLength(); i++) {
								way.appendChild(getCopy(nlist.item(i)));
							}
						}

						if (nameStr.equals("W:END")) {
							osmnode.appendChild(getCopy(way));
							way = null;
						}
					}
				}
			}
		}
		showWay();

		return iCounter;
	}

	static void showWay() {
		if (way == null) {
			System.out.println("way = null");
			return;
		}
		NodeList nlist = way.getChildNodes();
		System.out.println("<"+ way.getNodeName() +">");
		for (int i=0; i < nlist.getLength(); i++) {
			Node node2 = nlist.item(i);
			System.out.print("  <"+ node2.getNodeName() +" ");
			NamedNodeMap nodeMap = node2.getAttributes();
			if (null != nodeMap) {
				for (int j = 0; j < nodeMap.getLength(); j++ ) {
					System.out.print(nodeMap.item(j).getNodeName() +"='"+ nodeMap.item(j).getNodeValue() +"' ");
				}
			}
			System.out.println("/>");
		}
		System.out.println("</"+ way.getNodeName() +">");
		System.out.println();
	}

	/**
	 * 既存のWAYエレメントの状態を見ながら、WAYにNODEの内容を転記する。
	 * 以下の場合は、WAYをOSMに出力して、NODEを新たなWAYとする。
	 * ・'bridge=yes'属性のON／OFF
	 * @return WAY が切り替わるときはTRUE。そのままWAYを継続するときはFALSE。
	 * @param osmnode
	 * @param way
	 * @param node
	 */
	public static boolean updateWay(Element osmnode, Element onode) {
		if (isBridge(onode) && !isBridge(way)) {
			osmnode.appendChild(getCopy(way));
			return true;
		}
		String highway1 = getHighway(onode);
		String highway2 = getHighway(way);
		if (!highway1.equals("") && !highway1.equals(highway2)) {
			osmnode.appendChild(getCopy(way));
			return true;
		}

		NodeList nlist = onode.getChildNodes();
		for (int i=0; i < nlist.getLength(); i++) {
			Node node2 = nlist.item(i);
			if (node2.getNodeName().equals("tag")) {
				NamedNodeMap nodeMap = node2.getAttributes();
				if (null != nodeMap) {
					String k = "";
					String v = "";
					for (int j = 0; j < nodeMap.getLength(); j++ ) {
						if (nodeMap.item(j).getNodeName().equals("k")) {
							k =  nodeMap.item(j).getNodeValue();
							continue;
						}
						if (nodeMap.item(j).getNodeName().equals("v")) {
							v =  nodeMap.item(j).getNodeValue();
						}
					}
					String wv = getTag(way, k, v);
					if (wv != null) {
						osmnode.appendChild(getCopy(way));	//
						return true;
					}
				}
			}
		}

		return false;
	}

	public static boolean isBridge(Element node) {
		boolean bridge = false;
		boolean yes = false;
		NodeList nlist = node.getChildNodes();
		for (int i=0; i < nlist.getLength(); i++) {
			Node node2 = nlist.item(i);
			if (node2.getNodeName().equals("tag")) {
				NamedNodeMap nodeMap = node2.getAttributes();
				if (null != nodeMap) {
					for (int j = 0; j < nodeMap.getLength(); j++ ) {
						if (nodeMap.item(j).getNodeName().equals("k") && nodeMap.item(j).getNodeValue().equals("bridge")) {
							bridge = true;
							continue;
						}
						if (nodeMap.item(j).getNodeName().equals("v") && nodeMap.item(j).getNodeValue().equals("yes")) {
							yes = true;
						}
					}
				}
			}
		}
		if (bridge && yes) {
			return true;
		}
		else {
			return false;
		}
	}

	public static String getHighway(Element node) {
		NodeList nlist = node.getChildNodes();
		for (int i=0; i < nlist.getLength(); i++) {
			Node node2 = nlist.item(i);
			if (node2.getNodeName().equals("tag")) {
				NamedNodeMap nodeMap = node2.getAttributes();
				if (null != nodeMap) {
					boolean highway = false;
					String v = "";
					for (int j = 0; j < nodeMap.getLength(); j++ ) {
						if (nodeMap.item(j).getNodeName().equals("k") && nodeMap.item(j).getNodeValue().equals("highway")) {
							highway = true;
						}
						if (nodeMap.item(j).getNodeName().equals("v")) {
							v = nodeMap.item(j).getNodeValue();
						}
					}
					if (highway) {
						return v;
					}
				}
			}
		}
		return "";
	}

	/**
	 * 更新が必要ないならNULL
	 * @param node
	 * @param key
	 * @param value
	 * @return
	 */
	public static String getTag(Element node, String key, String value) {
		NodeList nlist = node.getChildNodes();
		for (int i=0; i < nlist.getLength(); i++) {
			Node node2 = nlist.item(i);
			if (node2.getNodeName().equals("tag")) {
				NamedNodeMap nodeMap = node2.getAttributes();
				if (null != nodeMap) {
					boolean kk = false;
					String v = "";
					for (int j = 0; j < nodeMap.getLength(); j++ ) {
						if (nodeMap.item(j).getNodeName().equals("k") && nodeMap.item(j).getNodeValue().equals(key)) {
							kk = true;
						}
						if (nodeMap.item(j).getNodeName().equals("v")) {
							v = nodeMap.item(j).getNodeValue();
						}
					}
					if (kk) {
						if (!v.equals(value)) {
							return v;
						}
						else {
							return null;
						}
					}
				}
			}
		}
		return value;
	}

	public static Element getOsmNode(Node temp, String name) {
		NodeList nodes = temp.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			NamedNodeMap nodeMap = node2.getAttributes();
			if (null != nodeMap) {
				for (int j=0; j < nodeMap.getLength(); j++ ) {
					if (nodeMap.item(j).getNodeName().equals("gpxname")) {
						if (nodeMap.item(j).getNodeValue().equals(name)) {
							return getCopy(node2);
						}
					}
				}
			}
		}
		return null;
	}

	public static Element getSourceTag(String timestampStr) {
		int i;
		if ((i = timestampStr.indexOf("T")) >= 0) {
			timestampStr = timestampStr.substring(0, i);
		}
		Element tag = document.createElement("tag");
		tag.setAttribute("k", "source");
		tag.setAttribute("v", "survey "+ timestampStr);
		return getCopy(tag);
	}

	public static Element getCopy(Node node) {
		Element root = document.createElement(node.getNodeName());

		NamedNodeMap nodeMap = node.getAttributes();
		if (null != nodeMap) {
			for (int j=0; j < nodeMap.getLength(); j++ ) {
				root.setAttribute(nodeMap.item(j).getNodeName(), nodeMap.item(j).getNodeValue());
			}
		}

		NodeList nodes = node.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeType() == Node.ELEMENT_NODE) {
				root.appendChild(getCopy(node2));
			}
		}

		return root;
	}

	static boolean checkFile(File f) {
		String name = f.getName();
		if (!name.endsWith(".gpx")) {
			return false;
		}
		return true;
	}


}