/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005-2006 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2006/06/06 15:30:29
 * 
 */
package jp.grain.spike.xpath;

import jp.grain.spike.Document;
import jp.grain.spike.Node;

public class CoreFunctionLibrary implements XPathFunctionLibrary{

    /*
     *  Trasration functions
     */
    public static final String FUNC_NUMBER = "number";
    public static final String FUNC_STRING = "string";    
    public static final String FUNC_BOOLEAN = "boolean";    
    
    /*
     *  Nodeset functions
     */
    public static final String FUNC_COUNT = "count";
    public static final String FUNC_ID = "id";

    /*
     * String functions
     */ 
    public static final String FUNC_CONCAT = "concat";
    public static final String FUNC_STARTS_WITH = "starts-with";
    public static final String FUNC_CONTAINS = "contains";
    public static final String FUNC_SUBSTRING_BEFORE = "substring-before";
    public static final String FUNC_SUBSTRING_AFTER = "substring-after";
    public static final String FUNC_SUBSTRING = "substring";
    public static final String FUNC_STRING_LENGTH = "string-length";
    public static final String FUNC_NORMALIZE_SPACE = "normalize-space";
    public static final String FUNC_TRANSLATE = "translate";
    
    public static Node[] nodeset(Object obj) {
        if(obj instanceof Node[]) {
            return (Node[])obj;
        }
        return new Node[0];
    }
    
    public static int number(Object obj) {
        if (obj instanceof Node[]) {
            obj = string(obj);
        }
        if (obj instanceof Integer) {
            return ((Integer)obj).intValue();
        } else if (obj instanceof String) {
            try {
                return Integer.parseInt(((String)obj).trim());
            } catch (NumberFormatException e) {
            }
        } else if (obj instanceof Boolean) {
            return ((Boolean)obj).booleanValue() ? 1 : 0;
        }
        return 0;
    }
    
    public static String string(Object obj) {
        if (obj instanceof Node[]) {
            Node[] nodeset = ((Node[])obj);
            if (nodeset.length == 0) return "";
            return nodeset[0].getSimpleContent();            
        }
        return String.valueOf(obj);
    }
    
    public static boolean bool(Object obj) {
        if (obj instanceof Integer) {
            if (((Integer)obj).intValue() == 0) return false;
        } else if (obj instanceof Node[]) {
            if (((Node[])obj).length == 0) return false;
        } else if (obj instanceof String) {
            if (((String)obj).length() == 0) return false;
        } else if (obj instanceof Boolean) {
            return ((Boolean)obj).booleanValue();
        }
        return true;
    }
    
    public boolean call(String funcName, XPathEvaluator.Frame f, int argc) {
        f.initBuffer();
        
        if (FUNC_NUMBER.equals(funcName)) {
            if (argc <= 0) {
                f.storeNumber(number(f._ctxnode));
            } else {
                f.loadArgs(1);
                f.storeNumber(number(f.getArg(0)));                
            }
        } else if (FUNC_STRING.equals(funcName)) {
            if (argc <= 0) {
                f.store(string(f._ctxnode));
            } else {
                f.loadArgs(1);
                f.store(string(f.getArg(0)));                
            }
        } else if (FUNC_BOOLEAN.equals(funcName)) {
            f.loadArgs(1);
            f.storeBoolean(bool(f.getArg(0)));

        } else if (FUNC_COUNT.equals(funcName)) {
            f.loadArgs(1);
            f.storeNumber(((Node[])f.getArg(0)).length);
        } else if (FUNC_ID.equals(funcName)) {
            f.loadArgs(1);
            Document doc = f._ctxnode.getDocument();
            Node node = doc.findNodeById(string(f.getArg(0)));
            f.initBuffer();
            if (node != null) f.put(node);
            f.storeNodeset();
        } else if (FUNC_CONCAT.equals(funcName)) {
            f.loadArgs(argc);
            StringBuffer buf = new StringBuffer();
            for (int i = 0; i < argc; ++i) {
                buf.append(string(f.getArg(i)));
            }
            f.store(buf.toString());
        } else if (FUNC_STARTS_WITH.equals(funcName)) {
            f.loadArgs(2);
            f.storeBoolean(string(f.getArg(0)).startsWith(string(f.getArg(1))));
        } else if(FUNC_CONTAINS.equals(funcName)){        	
        	f.loadArgs(2);
        	String str1 = string(f.getArg(0));
            String str2 = string(f.getArg(1));
         	f.storeBoolean(str1.indexOf(str2) < 0 ? false : true);        	
        } else if(FUNC_SUBSTRING_BEFORE.equals(funcName)){
        	f.loadArgs(2);
            String result = Node.EMPTY_STRING;
        	String str1 = string(f.getArg(0));
            String str2 = string(f.getArg(1));
            int index = str1.indexOf(str2);
            if (index != -1) result = str1.substring(0, index);
            f.store(result);
        } else if(FUNC_SUBSTRING_AFTER.equals(funcName)){
            f.loadArgs(2);
            String result = Node.EMPTY_STRING;
            String str1 = string(f.getArg(0));
            String str2 = string(f.getArg(1));
            int index = str1.indexOf(str2);
            if (index != -1) result = str1.substring(index + str2.length());
            f.store(result);
        } else if(FUNC_SUBSTRING.equals(funcName)){
        	f.loadArgs(argc);
        	String str = string(f.getArg(0));
        	int start = number(f.getArg(1)) - 1;
            int end = (argc > 2) ? start + number(f.getArg(2)) : str.length();
            if (start >= end) {
                f.store(Node.EMPTY_STRING);
            } else {
                if (start < 0) start = 0;
                if (end > str.length()) end = str.length();
                f.store(str.substring(start, end));
            }
        } else if (FUNC_STRING_LENGTH.equals(funcName)) {
            String str;
            if (argc > 0) {
                f.loadArgs(1);
                str = string(f.getArg(0));
            } else {
                str = string(new Node[]{f._ctxnode});
            }
        	f.storeNumber(str.length());
        } else if (FUNC_NORMALIZE_SPACE.equals(funcName)) {
            String str;
            if (argc > 0) {
                f.loadArgs(1);
                str = string(f.getArg(0));                
            } else {
                str = string(new Node[]{f._ctxnode});
            }
            StringBuffer buf = new StringBuffer();            
            boolean header = true;
            boolean s = false;
            for (int i = 0; i < str.length(); ++i) {
                char ch = str.charAt(i);
                switch (ch) {
                case Node.S_CR:
                case Node.S_LF:
                case Node.S_TAB:
                case Node.S_WHITESPACE:
                    if (header) {
                        continue;
                    } else {
                        s = true;
                    }
                    break;
                default:
                    if (header) header = false;
                    if (s) {
                        buf.append(Node.S_WHITESPACE);
                        s = false;
                    }
                    buf.append(ch);
                }
            }
            f.store(buf.toString());
        } else if (FUNC_TRANSLATE.equals(funcName)) {
            f.loadArgs(3);
            String str = string(f.getArg(0));
            char[] sch = string(f.getArg(1)).toCharArray();
            char[] rch = string(f.getArg(2)).toCharArray();
            StringBuffer buf = new StringBuffer();
            
            for (int i = 0; i < str.length(); ++i) {
                char ch = str.charAt(i);
                boolean replaced = false;
                for (int n = 0; n < sch.length; ++n) {
                    if (ch != sch[n]) continue;
                    if (n < rch.length) {
                        buf.append(rch[n]);
                    }
                    replaced = true;
                    break;
                }
                if (!replaced) buf.append(ch);
            }            
            f.store(buf.toString());
        } else {
            return false;
        }
        return true;
    }
}

