/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 * 
 * Created on 2005/05/07
 *
 */
package jp.haw.grain.framework.xml;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.CharArrayReader;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.log4j.Logger;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * gbXML(oCiXML)K؂ȃXg[ɏo͂
 * 
 * @version $Id: BinaryXMLOutputter.java 268 2006-05-02 06:06:02Z go $
 * @author Go Takahashi
 *
 */
public class BinaryXMLOutputter {
	
    public static final int XML_DECL_MAX_LENGTH = 100;    
	private static final Logger log = Logger.getLogger(BinaryXMLOutputter.class);
    
    private InputSource source;
    private String encoding = "Shift-JIS";
    private String lineSeparator = "\n";
    
    public BinaryXMLOutputter(byte[] bs) {
		try {
			String header = new String(bs, 0, bs.length > 50 ? 50 : bs.length, "ISO-8859-1");
			parseEncoding(header);
		} catch (UnsupportedEncodingException e) {
		}
        this.source= new InputSource(new ByteArrayInputStream(bs));
        log.debug("encoding="+this.encoding);
    }    

    public BinaryXMLOutputter(char[] cs) {
        this.source= new InputSource(new CharArrayReader(cs));
        String head = new String(cs, 0, cs.length > 50 ? 50 : cs.length); 
        parseEncoding(head);
		log.debug("encoding="+this.encoding);
    }
    
    public BinaryXMLOutputter(InputStream is) {
        this.source = new InputSource(new BufferedInputStream(is));
        parseEncoding(this.source.getByteStream());
        log.debug("encoding="+this.encoding);
    }

    public BinaryXMLOutputter(Reader reader) {
        this.source = new InputSource(new BufferedReader(reader));
        parseEncoding(this.source.getCharacterStream());
        log.debug("encoding="+this.encoding);
    }
    
    public BinaryXMLOutputter(String src) {
        this.source= new InputSource(new StringReader(src));
        parseEncoding(src);
		log.debug("encoding="+this.encoding);
    }
    
    private void parseEncoding(InputStream is) {
        is.mark(XML_DECL_MAX_LENGTH);
        try {
            StringBuffer head = new StringBuffer();
            for (int i = 0; i < XML_DECL_MAX_LENGTH; ++i) {
                int b = is.read();
                if (b < 0) break;
                head.append((char)b);
                if (i > 0 && head.substring(i - 1).equals("?>")) break;
            }
            parseEncoding(head.toString());
        } catch (IOException e) {
        } finally {
            try {
                is.reset();
            } catch (IOException e) {
            }
        }
    }

    private void parseEncoding(Reader reader) {
        try {
            reader.mark(XML_DECL_MAX_LENGTH);
            StringBuffer head = new StringBuffer();
            for (int i = 0; i < XML_DECL_MAX_LENGTH; ++i) {
                int b = reader.read();
                if (b < 0) break;
                head.append((char)b);
                if (i > 0 && head.substring(i - 1).equals("?>")) break;
            }
            parseEncoding(head.toString());
        } catch (IOException e) {
        } finally {
            try {
                reader.reset();
            } catch (IOException e) {
            }
        }        
    }
    
    private void parseEncoding(String head) {
        if (!head.startsWith("<?xml")) return;
        Matcher m = Pattern.compile("encoding=\"([^\"]*)\"").matcher(head);
        if (m.find() && m.groupCount() > 0) setTextEncoding(m.group(1));
    }
    
    public void setLineSeparator(String separator) {
        this.lineSeparator = separator;
    }
    
    public void setTextEncoding(String encoding) {
        this.encoding = encoding;
    }
    
    
    public void writeTo(OutputStream os) throws IOException, ParseException {
        try {
            SAXParserFactory factory = SAXParserFactory.newInstance();
            factory.setNamespaceAware(true);
            SAXParser parser = factory.newSAXParser();
            log.debug("SAXParser: " + parser.getClass().getName());
            log.debug("char stream: " + this.source.getCharacterStream());
            log.debug("output stream: " + os.getClass().getName());
            BinaryXMLEncoder enc = new BinaryXMLEncoder(this.encoding);
            parser.parse(this.source, enc);
            enc.writeTo(os);
        } catch (SAXException e) {
            log.debug(e.toString(), e);
            throw new ParseException(e);
        } catch (ParserConfigurationException e) {
            log.debug(e.toString(), e);
            throw new ParseException(e);
        } catch (FactoryConfigurationError e) {
            log.debug(e.toString(), e);
            throw new ParseException(e);
        }
    }
    
    public static void main(String[] args) {
        if (args.length != 3) {
            System.err.println("ussage: encoding infile(xml) outfile(gbxml)");
            System.exit(1);
        }
        InputStream is = null;
        OutputStream os = null;
        try {
            is = new BufferedInputStream(new FileInputStream(args[1]));
            BinaryXMLOutputter bxo = new BinaryXMLOutputter(is);
            bxo.setTextEncoding(args[0]);
            os = new BufferedOutputStream(new FileOutputStream(args[2]));
            bxo.writeTo(os);
        } catch (IOException e) {
            e.printStackTrace();
        } catch (ParseException e) {
            e.printStackTrace();
        } finally {
            try {
                if (is != null) is.close();
            } catch (IOException e) {
                e.printStackTrace();
            } finally {
                try {
                    if (os != null) os.close();
                } catch (IOException ie) {
                    ie.printStackTrace();
                }
            }
        }
    }

}
