package jp.haw.grain.framework.servlet;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;

import jp.haw.grain.framework.xml.BinaryXMLEncoderTest;
import jp.haw.grain.framework.xml.Util;

import org.apache.cactus.FilterTestCase;
import org.apache.cactus.WebRequest;
import org.apache.log4j.Logger;

import com.meterware.httpunit.WebResponse;

/**
 * @author nakajo
 */
public class BinaryXMLFilterExtendHeaderTest extends FilterTestCase {

	public static final String APPLICATION_GBXML = "application/gbxml";
	private static final Logger log = Logger.getLogger(BinaryXMLFilterExtendHeaderTest.class);

	/**
	 * gwb_X-expect-FO: gudݒ肳ĂƂgudɕϊB
	 * @param request
	 */
	public void beginExtendFOHeader(WebRequest request) {
		String extendHeader = "X-expect-FO: gud\n" + "\n";
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			out.write(extendHeader.getBytes());
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		ByteArrayInputStream stream = new ByteArrayInputStream(out.toByteArray());
		request.setUserData(stream);
		request.setContentType(APPLICATION_GBXML);
	}

	public void testExtendFOHeader() throws ServletException, IOException {
		assertEquals("POST", request.getMethod());
		assertEquals(APPLICATION_GBXML, request.getContentType());

		BinaryXMLFilter filter = new BinaryXMLFilter();
		filter.init(config);
		MockFilterChain chain = new WriterChain(BinaryXMLEncoderTest.TEST_TEXT_GUD_NS) {
			public void readRequest(ServletRequest request) throws IOException {

				assertEquals(((HttpServletRequest)request).getHeader("X-expect-FO"), "gud");
			}
		};
		filter.doFilter(request, response, chain);
	}

	public void endExtendFOHeader(WebResponse response) throws IOException {
		assertEquals(response.getContentType(), APPLICATION_GBXML);
	}

	/**
	 * gwb_X-expect-FO:ĂgudłȂΕϊȂ
	 * @param request
	 */
	public void beginNotGudFO(WebRequest request) {
		String extendHeader = "X-expect-FO: test\n" + "\n";
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			out.write(extendHeader.getBytes());
			out.write(BinaryXMLEncoderTest.TEST_BIN_STREAM_NS);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		ByteArrayInputStream stream = new ByteArrayInputStream(out.toByteArray());
		request.setUserData(stream);
		request.setContentType(APPLICATION_GBXML);
	}

	public void testNotGudFO() throws ServletException, IOException {
		assertEquals("POST", request.getMethod());
		assertEquals(APPLICATION_GBXML, request.getContentType());

		BinaryXMLFilter filter = new BinaryXMLFilter();
		filter.init(config);
		MockFilterChain chain = new WriterChain(BinaryXMLEncoderTest.TEST_TEXT_NS) {
			public void readRequest(ServletRequest request) throws IOException {

				assertEquals(((HttpServletRequest)request).getHeader("X-expect-FO"), "test");
			}
		};
		filter.doFilter(request, response, chain);
	}

	public void endNotGudFO(WebResponse response) throws IOException {
		assertEquals(response.getContentType(), APPLICATION_GBXML);
		InputStream is = response.getInputStream();
		for (int i = 0; i < BinaryXMLEncoderTest.TEST_BIN_STREAM_NS.length; ++i) {
			assertEquals(BinaryXMLEncoderTest.TEST_BIN_STREAM_NS[i], (byte)is.read());
		}

	}

	/**
	 * CDATÃeXg
	 * @param request
	 */
	public void beginCDATA(WebRequest request) {
		String extendHeader = "X-expect-FO: null\n" + "\n";
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			out.write(extendHeader.getBytes());
			out.write(BinaryXMLEncoderTest.TEST_CDATA);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		ByteArrayInputStream stream = new ByteArrayInputStream(out.toByteArray());
		request.setUserData(stream);
		request.setContentType(APPLICATION_GBXML);
	}

	public void testCDATA() throws ServletException, IOException {
		assertEquals("POST", request.getMethod());
		assertEquals(APPLICATION_GBXML, request.getContentType());

		BinaryXMLFilter filter = new BinaryXMLFilter();
		filter.init(config);
		MockFilterChain chain = new WriterChain(BinaryXMLEncoderTest.TEST_TEXT_NS) {
			public void readRequest(ServletRequest request) throws IOException {
				BufferedReader reader = request.getReader();
				String buf = null;
				StringBuffer result = new StringBuffer();
				while((buf = reader.readLine()) != null) {
					result.append(buf);
				}
				Pattern p = Pattern.compile("<!\\[CDATA\\[([\\w=]+)\\]\\]>");
				Matcher match = p.matcher(result.toString());
				if(match.find()) {
					String m = match.group(1);
					try {
					assertEquals("cdata test", "test", new String(Util.decodeBase64(m)));
					} catch (Exception e) {
						fail(e.toString());
					}
				} else {
					fail("not found cdata");
				}
			}
		};
		filter.doFilter(request, response, chain);
	}

	public void endCDATA(WebResponse response) throws IOException {

	}



	/**
	 * gwb_ĂrequestcontentTypeGBXMLłȂΉ߂ȂB
	 * @param request
	 */
	public void beginNotGudRequest(WebRequest request) {
		String extendHeader = "X-expect-FO: test\n" +
			"X-Divide-Transfer: start\n" +
			"X-Divide-Max-Count: 3\n" +
			"X-Divide-Count: 1\n" +
			"X-Divide-Protocol-Version: 1.0\n" + "\n";
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			out.write(extendHeader.getBytes());
			out.write(BinaryXMLEncoderTest.TEST_BIN_STREAM_NS);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		ByteArrayInputStream stream = new ByteArrayInputStream(out.toByteArray());
		request.setUserData(stream);
		request.setContentType(APPLICATION_GBXML);
	}

	public void testNotGudRequest() throws ServletException, IOException {
		assertEquals("POST", request.getMethod());
		assertEquals(APPLICATION_GBXML, request.getContentType());

		BinaryXMLFilter filter = new BinaryXMLFilter();
		filter.init(config);
		MockFilterChain chain = new WriterChain(BinaryXMLEncoderTest.TEST_TEXT_NS) {
			public void readRequest(ServletRequest request) throws IOException {

				assertNull(((HttpServletRequest)request).getHeader("X-expect-FO"));
				assertNull(((HttpServletRequest)request).getHeader("X-Divide-Transfer"));
				assertNull(((HttpServletRequest)request).getHeader("X-Divide-Max-Count"));
				assertNull(((HttpServletRequest)request).getHeader("X-Divide-Count"));
				assertNull(((HttpServletRequest)request).getHeader("X-Divide-Protocol-Version"));
			}
		};
		filter.doFilter(request, response, chain);
	}

	public void endNotGudRequest(WebResponse response) throws IOException {
	}

	/**
	 * gwb_X-Divide-Transfer: startsessionIDAĂ
	 * response͋ۂɂȂĂ
	 * @param request
	 */
	public void beginXDivideStart(WebRequest request) {
		String extendHeader = "X-Divide-Transfer: start\n" + "X-Divide-Max-Count: 3\n" + "X-Divide-Count: 1\n" + "\n";
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			out.write(extendHeader.getBytes());
			out.write(BinaryXMLEncoderTest.TEST_BIN_STREAM_NS);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		ByteArrayInputStream stream = new ByteArrayInputStream(out.toByteArray());
		request.setUserData(stream);
		request.setContentType(APPLICATION_GBXML);
	}

	public void testXDivideStart() throws ServletException, IOException {
		assertEquals("POST", request.getMethod());
		assertEquals(APPLICATION_GBXML, request.getContentType());

		BinaryXMLFilter filter = new BinaryXMLFilter();
		filter.init(config);
		MockFilterChain chain = new WriterChain(BinaryXMLEncoderTest.TEST_TEXT_NS) {
			public void readRequest(ServletRequest request) throws IOException {

				assertEquals(((HttpServletRequest)request).getHeader("X-Divide-Transfer"), "start");
			}
		};
		filter.doFilter(request, response, chain);
	}

	public void endXDivideStart(WebResponse response) throws IOException {
		assertNotNull(response.getHeaderField("X-Divide-Session-ID"));
		assertEquals(response.getHeaderField("X-Divide-Protocol-Version"), "1.0");
		InputStream is = response.getInputStream();
		assertEquals(is.read(), -1);
	}

	/**
	 * gwb_ɂĂX-Divide-SessionID
	 * ServletException
	 * @param request
	 */
	public void beginSessionIDError(WebRequest request) {
		String extendHeader = "X-Divide-Session-ID: AAAAAAAAAAAAAAAA\n" + "X-Divide-Count: 2\n" + "\n";
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			out.write(extendHeader.getBytes());
			out.write(BinaryXMLEncoderTest.TEST_BIN_STREAM_NS);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		ByteArrayInputStream stream = new ByteArrayInputStream(out.toByteArray());
		request.setUserData(stream);
		request.setContentType(APPLICATION_GBXML);
	}

	public void testSessionIDError() throws ServletException, IOException {
		assertEquals("POST", request.getMethod());
		assertEquals(APPLICATION_GBXML, request.getContentType());

		BinaryXMLFilter filter = new BinaryXMLFilter();
		filter.init(config);
		MockFilterChain chain = new WriterChain(BinaryXMLEncoderTest.TEST_TEXT_NS) {
			public void readRequest(ServletRequest request) throws IOException {

				assertEquals(((HttpServletRequest)request).getHeader("X-Divide-Session-ID"), "AAAAAAAAAAAAAAAA");
			}
		};
		try {
			filter.doFilter(request, response, chain);
			fail("ServletException has not raised.");
		} catch(ServletException e) {
			assertEquals("message check", e.getMessage(), "session is not found!: sessionId=AAAAAAAAAAAAAAAA");
		}
	}

	public void endSessionIDError(WebResponse response) throws IOException {
	}
	
	/*
	 * FilterChaiñbNAbv
	 */
	abstract class MockFilterChain implements FilterChain {

		String contentType = "application/xml; charset=UTF-8";
		protected String testString;

		MockFilterChain(String testString) {
			this.testString = testString;
		}

		void setContentType(String contentType) {
			this.contentType = contentType;
		}

		public void doFilter(ServletRequest request, ServletResponse response) throws IOException, ServletException {
			if (this.contentType != null)
				response.setContentType(this.contentType);
			readRequest(request);
			writeResponse(response);
		}

		public void init(FilterConfig conf) {
		}

		public void destroy() {
		}

		void writeResponse(ServletResponse response) throws IOException {
		}

		void readRequest(ServletRequest request) throws IOException {
		}
	}

	class WriterChain extends MockFilterChain {

		WriterChain(String testString) {
			super(testString);
		}

		void writeResponse(ServletResponse response) throws IOException {
			PrintWriter writer = response.getWriter();
			writer.print(this.testString);
		}
	}

}
