
#==============================================================================#
# $Id: base.rb,v 1.2 2002/09/18 15:43:59 yuya Exp $
#==============================================================================#

require 'GD'

#====================================================================#

# Graph Module
module Graph

  # GraphBase Class
  class GraphBase

    # Initialize Method

    def initialize(image, x, y, dx, dy)
      @image = image
      @box   = Box.new(x, y, dx, dy)
      @data  = nil

      @frame  = create_attribute_frame
      @title  = create_attribute_title
      @legend = create_attribute_legend
      @chart  = create_attribute_chart

      @frame.setup
      @title.setup
      @legend.setup
      @chart.setup
    end

    # Accessor

    attr_accessor :image, :box, :data
    attr_reader   :frame, :title, :legend, :chart

    # Class Methods

    def self.create(dx, dy, truecolor = false)
      if truecolor
        image = GD::Image.newTrueColor(dx, dy)
        image.filledRectangle(0, 0, dx - 1, dy - 1, image.colorAllocate(255, 255, 255))
      else
        image = GD::Image.new(dx, dy) 
        image.colorAllocate(255, 255, 255)
      end

      return self.new(image, 0, 0, dx, dy)
    end

    # Private Instance Methods

    def create_attribute_frame
      return Attribute::Composite::Frame.new
    end
    private :create_attribute_frame

    def create_attribute_title
      return Attribute::Composite::Title.new
    end
    private :create_attribute_title

    def create_attribute_legend
      return Attribute::Composite::Legend.new
    end
    private :create_attribute_legend

    def create_attribute_chart
      raise NotImplementedError, "don't use this class"
    end
    private :create_attribute_chart

    def build
      return @frame.create_decorator(layout)
    end
    private :build

    def layout
      case [@title.visible, @legend.visible]
      when [true, true];   decorator = layout_title(build_title, layout_legend(build_legend, build_graph))
      when [true, false];  decorator = layout_title(build_title, build_graph)
      when [false, true];  decorator = layout_legend(build_legend, build_graph)
      when [false, false]; decorator = build_graph
      else
        raise 'bug?'
      end

      return decorator
    end
    private :layout

    def layout_title(title, graph)
      case @title.position
      when :top_center
        decorator = Decorator::HorizontalAlignCenter.new(title)
        decorator = Decorator::VerticalSplitUpper.new(decorator, graph)
      when :middle_left
        decorator = Decorator::VerticalAlignCenter.new(title)
        decorator = Decorator::HorizontalSplitLeft.new(decorator, graph)
      when :middle_right
        decorator = Decorator::VerticalAlignCenter.new(title)
        decorator = Decorator::HorizontalSplitRight.new(graph, decorator)
      when :bottom_center
        decorator = Decorator::HorizontalAlignCenter.new(title)
        decorator = Decorator::VerticalSplitLower.new(graph, decorator)
      else
        raise ArgumentError, 'invalid position'
      end

      return decorator
    end

    def layout_legend(legend, graph)
      if @legend.vertical
        case @legend.position
        when :top_left
          decorator = Decorator::VerticalAlignTop.new(legend)
          decorator = Decorator::HorizontalSplitLeft.new(decorator, graph)
        when :top_right
          decorator = Decorator::VerticalAlignTop.new(legend)
          decorator = Decorator::HorizontalSplitRight.new(graph, decorator)
        when :middle_left
          decorator = Decorator::VerticalAlignCenter.new(legend)
          decorator = Decorator::HorizontalSplitLeft.new(decorator, graph)
        when :middle_right
          decorator = Decorator::VerticalAlignCenter.new(legend)
          decorator = Decorator::HorizontalSplitRight.new(graph, decorator)
        when :bottom_left
          decorator = Decorator::VerticalAlignBottom.new(legend)
          decorator = Decorator::HorizontalSplitLeft.new(decorator, graph)
        when :bottom_right
          decorator = Decorator::VerticalAlignBottom.new(legend)
          decorator = Decorator::HorizontalSplitRight.new(graph, decorator)
        else
          raise ArgumentError, 'invalid position'
        end
      else
        case @legend.position
        when :top_left
          decorator = Decorator::HorizontalAlignLeft.new(legend)
          decorator = Decorator::VerticalSplitUpper.new(decorator, graph)
        when :top_center
          decorator = Decorator::HorizontalAlignCenter.new(legend)
          decorator = Decorator::VerticalSplitUpper.new(decorator, graph)
        when :top_right
          decorator = Decorator::HorizontalAlignRight.new(legend)
          decorator = Decorator::VerticalSplitUpper.new(decorator, graph)
        when :bottom_left
          decorator = Decorator::HorizontalAlignLeft.new(legend)
          decorator = Decorator::VerticalSplitLower.new(graph, decorator)
        when :bottom_center
          decorator = Decorator::HorizontalAlignCenter.new(legend)
          decorator = Decorator::VerticalSplitLower.new(graph, decorator)
        when :bottom_right
          decorator = Decorator::HorizontalAlignRight.new(legend)
          decorator = Decorator::VerticalSplitLower.new(graph, decorator)
        else
          raise ArgumentError, 'invalid position'
        end
      end

      return decorator
    end
    private :layout_legend

    def build_title
      return @title.create_decorator(create_title)
    end
    private :build_title

    def build_legend
      return @legend.create_decorator(create_legend)
    end
    private :build_legend

    def build_graph
      return @chart.create_decorator(create_graph)
    end
    private :build_graph

    def create_title
      return Title.new(@title.font.create_font, @title.text)
    end
    private :create_title

    def create_legend
      raise NotImplementedError, "don't use this class"
    end
    private :create_legend

    def create_graph
      raise NotImplementedError, "don't use this class"
    end
    private :create_graph

    # Instance Methods

    def draw
      frame = build()
      frame.draw(@image, @box)
    end

    def png(io)
      self.draw
      @image.png(io)
    end

    def png_file(filepath)
      File.open(filepath, 'w') { |file|
        self.png(file)
      }
    end

    def png_cgi
      STDOUT.print "Content-Type: image/png\n\n"
      self.png(STDOUT)
    end

    def jpeg(io, quality)
      self.draw
      @image.jpeg(io, quality)
    end

    def jpeg_file(filepath, quality)
      File.open(filepath, 'w') { |file|
        self.jpeg(file, quality)
      }
    end

    def jpeg_cgi
      STDOUT.print "Content-Type: image/jpeg\n\n"
      self.jpeg(STDOUT)
    end

    def interlace
      return @image.interlace
    end

    def interlace=(value)
      @image.interlace = value
    end

  end # GraphBase

end # Graph

#====================================================================#
