
#==============================================================================#
# $Id: legend.rb,v 1.2 2002/09/18 15:43:59 yuya Exp $
#==============================================================================#

require 'ggcl/extension/enumerable'

module Graph

  # Legend Class
  class Legend

    # Initialize Method

    def initialize(attribute)
      @attribute = attribute
      @fields    = []

      @spacing   = @attribute.interval
      @vertical  = @attribute.vertical
    end

    # Instance Methods

    def draw(image, box)
      if @vertical
        @fields.inject(0) { |current, field|
          field.draw(image, box.x, box.y + current)
          current + field.dy + @spacing
        }
      else
        @fields.inject(box.x) { |current, field|
          field.draw(image, current, box.y)
          current + field.dx + @spacing
        }
      end
    end

    def dx
      if @vertical
        unless @fields.empty?
          return @fields.collect { |field| field.dx }.max
        else
          return 0
        end
      else
        unless @fields.empty?
          return @fields.inject(0) { |result, field| result + field.dx + @spacing } - @spacing
        else
          return 0
        end
      end
    end

    def dy
      if @vertical
        unless @fields.empty?
          return @fields.inject(0) { |result, field| result + field.dy + @spacing } - @spacing
        else
          return 0
        end
      else
        unless @fields.empty?
          return @fields.collect { |field| field.dy }.max
        else
          return 0
        end
      end
    end

    def add(font, text, style = nil)
      @fields << Field.new(font, text, style, @attribute)
    end

    class Field

      def initialize(font, text, marker, attribute)
        @text      = Text.new(font, text)
        @marker    = marker
        @marker_dx = attribute.marker.width
        @marker_dy = attribute.marker.height
        @spacing   = attribute.marker.spacing
      end

      def draw(image, x, y)
        if @marker
          cy = y + self.dy / 2.0
          @marker.draw(image, Box.new(x, cy - @marker_dy / 2.0, @marker_dx, @marker_dy))
          @text.draw2(image, x + @marker_dx + @spacing, cy - @text.dy / 2.0)
        else
          @text.draw2(image, x, y)
        end

      end

      def dx
        if @marker
          return @text.dx + @spacing + @marker_dx
        else
          return @text.dx
        end
      end

      def dy
        if @marker
          return [@text.dy, @marker_dy].max
        else
          return @text.dy
        end
      end

    end

    module Marker

      class Base
        def draw(image, box)
          raise "Don't use this class"
        end
      end

      class Color < Base
        def initialize(fill_color, line_color)
          @fill_color = fill_color
          @line_color = line_color
        end
        def draw(image, box)
          image.filledRectangle(box.x, box.y, box.x2, box.y2, @fill_color.allocate(image))
          image.rectangle(box.x, box.y, box.x2, box.y2, @line_color.allocate(image))
        end
      end

      class Line < Base
        def initialize(color)
          @color = color
        end
        def draw(image, box)
          color = @color.allocate(image)
          cy    = box.y + box.dy / 2
          image.line(box.x, cy, box.x2, cy, color)
        end
      end

      class LineMarker < Base
        def initialize(color, marker)
          @color  = color
          @marker = marker
        end
        def draw(image, box)
          color = @color.allocate(image)
          cx    = box.x + box.dx / 2
          cy    = box.y + box.dy / 2
          image.line(box.x, cy, box.x2, cy, color)
          @marker.draw(image, cx, cy)
        end
      end

    end

  end

end
