
#==============================================================================#
# gruby/core.rb
# $Id: core.rb,v 1.3 2002/09/19 08:37:07 yuya Exp $
#==============================================================================#

#==============================================================================#
class GRb

  #============================================================================#
  # Initialize Method

  def initialize(image)
    @image = image
  end

  #============================================================================#
  # Class Methods

  def self.create(width, height, truecolor = false)
    if truecolor
      image = GD::Image.newTrueColor(width, height)
      image.filledRectangle(0, 0, width - 1, height - 1, image.colorAllocate(255, 255, 255))
    else
      image = GD::Image.new(width, height)
      image.colorAllocate(255, 255, 255)
    end

    return self.new(image)
  end

  def self.new_from_png(filepath)
    return self.new(GD::Image.newFromPng(filepath))
  end

  def self.new_from_jpeg(filepath)
    return self.new(GD::Image.newFromJpeg(filepath))
  end

  #============================================================================#
  # Private Instance Methods

  def create_brush(size, color)
    red   = @image.red(color)
    green = @image.green(color)
    blue  = @image.blue(color)

    trans_red   = ( red   == 0 ? 255 : 0 )
    trans_green = ( green == 0 ? 255 : 0 )
    trans_blue  = ( blue  == 0 ? 255 : 0 )

    if size == 2
      brush = ::GD::Image.new(2, 2)
      brush.colorAllocate(red, green, blue)
    elsif size >= 3
      brush = ::GD::Image.new(size, size)
      brush.colorAllocate(trans_red, trans_green, trans_blue)
      brush.colorAllocate(red, green, blue)
      brush.colorAllocate(trans_red, trans_green, trans_blue)
      brush.transparent(0)

      case size
      when 3
        brush.setPixel(1, 0, 1)   #  *
        brush.line(0, 1, 2, 1, 1) # ***
        brush.setPixel(1, 2, 1)   #  *
      when 4
        brush.line(1, 0, 2, 0, 1) #  **
        brush.line(0, 1, 3, 1, 1) # ****
        brush.line(0, 2, 3, 2, 1) # ****
        brush.line(1, 3, 2, 3, 1) #  **
      when 5
        brush.setPixel(2, 0, 1)   #   *
        brush.line(1, 1, 3, 1, 1) #  ***
        brush.line(0, 2, 4, 2, 1) # *****
        brush.line(1, 3, 3, 3, 1) #  ***
        brush.setPixel(2, 4, 1)   #   *
      when 6
        brush.line(2, 0, 3, 0, 1) #   **
        brush.line(1, 1, 4, 1, 1) #  ****
        brush.line(0, 2, 5, 2, 1) # ******
        brush.line(0, 3, 5, 3, 1) # ******
        brush.line(1, 4, 4, 4, 1) #  ****
        brush.line(2, 5, 3, 5, 1) #   **
      when 7
        brush.line(2, 0, 4, 0, 1) #   ***
        brush.line(1, 1, 5, 1, 1) #  *****
        brush.line(0, 2, 6, 2, 1) # *******
        brush.line(0, 3, 6, 3, 1) # *******
        brush.line(0, 4, 6, 4, 1) # *******
        brush.line(1, 5, 5, 5, 1) #  *****
        brush.line(2, 6, 4, 6, 1) #   ***
      when 8
        brush.line(3, 0, 4, 0, 1) #    **
        brush.line(1, 1, 6, 1, 1) #  ******
        brush.line(1, 2, 6, 2, 1) #  ******
        brush.line(0, 3, 7, 3, 1) # ********
        brush.line(0, 4, 7, 4, 1) # ********
        brush.line(1, 5, 6, 5, 1) #  ******
        brush.line(1, 6, 6, 6, 1) #  ******
        brush.line(3, 7, 4, 7, 1) #    **
      when 9
        brush.setPixel(4, 0, 1)   #     *
        brush.line(2, 1, 6, 1, 1) #   *****
        brush.line(1, 2, 7, 2, 1) #  *******
        brush.line(1, 3, 7, 3, 1) #  *******
        brush.line(0, 4, 8, 4, 1) # *********
        brush.line(1, 5, 7, 5, 1) #  *******
        brush.line(1, 6, 7, 6, 1) #  *******
        brush.line(2, 7, 6, 7, 1) #   *****
        brush.setPixel(4, 8, 1)   #     *
      when 10
        brush.line(4, 0, 5, 0, 1) #     **
        brush.line(2, 1, 7, 1, 1) #   ******
        brush.line(1, 2, 8, 2, 1) #  ********
        brush.line(1, 3, 8, 3, 1) #  ********
        brush.line(0, 4, 9, 4, 1) # **********
        brush.line(0, 5, 9, 5, 1) # **********
        brush.line(1, 6, 8, 6, 1) #  ********
        brush.line(1, 7, 8, 7, 1) #  ********
        brush.line(2, 8, 7, 8, 1) #   ******
        brush.line(4, 9, 5, 9, 1) #     **
      else
        half = (size / 2.0).floor
        brush.filledArc(half, half, size, size, 0, 360, 1, ::GD::Arc)
      end
    else
      raise 'bug?'
    end

    return brush
  end
  private :create_brush

  def create_pattern(pattern, color)
    array = []

    pattern.each_with_index { |repeat, index|
      repeat.times {
        array << (index % 2 == 0 ? color : GD::Transparent)
      }
    }

    return array
  end
  private :create_pattern

  def set_brush_and_pattern(color, width, pattern)
    with_brush   = (!width.nil?   && width >= 2)
    with_pattern = (!pattern.nil? && pattern.kind_of?(Array))

    if with_brush
      brush = create_brush(width, color)
      @image.setBrush(brush)
    end

    if with_pattern
      style = create_pattern(pattern, color)
      @image.setStyle(*style)
    end

    return case [with_brush, with_pattern]
           when [true, true]   then ::GD::StyledBrushed
           when [true, false]  then ::GD::Brushed
           when [false, true]  then ::GD::Styled
           when [false, false] then color
           else raise 'bug?'
           end
  end
  private :set_brush_and_pattern

  def image2gd(image)
    case image
    when GRb       then image.image
    when GD::Image then image
    else raise TypeError, "GD::Image or GRb required"
    end
  end
  private :image2gd

#  def create_canvas(bgcolor)
#    if self.truecolor?
#      image = GD::Image.newTrueColor(self.width, self.height)
#      cidx = image.colorAllocate(*bgcolor.rgb)
#      image.filledRectangle(0, 0, width - 1, height - 1, cidx)
#      image.transparent(cidx)
#    else
#      image = GD::Image.new(self.width, self.height)
#      image.transparent(image.colorAllocate(*bgcolor.rgb))
#    end
#
#    return self.type.new(image)
#  end

  #============================================================================#
  # Instance Methods

  def image
    return @image
  end

  def interlace
    return @image.interlace
  end

  def interlace=(value)
    @image.interlace = value
  end

  def truecolor?
    return @image.is_trueColor?
  end

  def width
    return @image.width
  end

  def height
    return @image.height
  end

  def destroy
    @image.destroy
  end

  def transparent(color)
    return @image.transparent(color)
  end

  def rgb(color)
    return @image.rgb(color)
  end

  def color(red, green, blue)
    return color_resolve(red, green, blue)
  end

  def color_allocate(red, green, blue)
    return @image.colorAllocate(red, green, blue)
  end

  def color_closest(red, green, blue)
    return @image.colorClosest(red, green, blue)
  end

  def color_deallocate(color)
    return @image.colorDeallocate(color)
  end

  def color_exact(red, green, blue)
    return @image.colorExact
  end

  def color_resolve(red, green, blue)
    return @image.colorResolve(red, green, blue)
  end

  def colors_total
    return @image.colorsTotal
  end

  def png(io)
    @image.png(io)
  end

  def png_file(filepath)
    File.open(filepath, 'w') { |file|
      self.png(file)
    }
  end

  def jpeg(io, quality = -1)
    @image.jpeg(io, quality)
  end

  def jpeg_file(filepath, quality = -1)
    File.open(filepath, 'w') { |file|
      self.jpeg(file, quality)
    }
  end

  def set(obj)
    case obj
    when Color  then obj.set(self)
    when Stroke then obj.set(self)
    when Fill   then obj.set(self)
    else raise 'invalid object type'
    end
  end

  def set_brush(brush)
    @image.setBrush(brush)
  end

  def set_style(*style)
    @image.setStyle(*style)
  end

  def get_pixel(x, y)
    return @image.getPixel(x, y)
  end

  def set_pixel(x, y, color)
    @image.setPixel(x, y, color)
  end

  def pset(x, y, color)
    set_pixel(x, y, color)
  end

  def rect(x1, y1, x2, y2, line = nil, fill = nil)
    xx1, xx2 = [x1, x2].sort
    yy1, yy2 = [y1, y2].sort

    filler = Fill.new_from(fill)
    if filler.draw?
      layer = Layer.new(self, filler.opacity, filler.color.invert)
      layer.image.filledRectangle(xx1, yy1, xx2, yy2, layer.set(filler))
      layer.merge
    end

    stroke = Stroke.new_from(line)
    if stroke.draw?
      layer = Layer.new(self, stroke.opacity, stroke.color.invert)
      layer.image.rectangle(xx1, yy1, xx2, yy2, layer.set(stroke))
      layer.merge
    end
  end

  def line(x1, y1, x2, y2, line)
    stroke = Stroke.new_from(line)
    if stroke.draw?
      layer = Layer.new(self, stroke.opacity, stroke.color.invert)
      layer.image.line(x1, y1, x2, y2, layer.set(stroke))
      layer.merge
    end
  end

  def circle(cx, cy, rx, ry, line = nil, fill = nil, width = nil, pattern = nil)
    self.arc(cx, cy, rx, ry, 0, 360, line, fill, width, pattern)
  end

  def arc(cx, cy, rx, ry, deg1, deg2, line = nil, fill = nil, width = nil, pattern = nil)
    if fill
      @image.filledArc(cx, cy, rx, ry, deg1, deg2, fill, 0)
    end

    if line
      style = set_brush_and_pattern(line, width, pattern)
      @image.arc(cx, cy, rx, ry, deg1, deg2, style)
    end
  end

  def polygon(poly, line = nil, fill = nil, width = nil, pattern = nil)
    if fill
      @image.filledPolygon(poly.to_polygon, fill)
    end

    if line
      style = set_brush_and_pattern(line, width, pattern)
      @image.polygon(poly.to_polygon, style)
    end
  end

  def string(font, x, y, angle, color, text)
    font.draw(@image, x, y, angle, self.set(color), text)
  end

  def fill(x, y, fill, border = nil)
    if border
      @image.fillToBorder(x, y, border, fill)
    else
      @image.fill(x, y, fill)
    end
  end

  def copy(image, x, y, sx, sy, dx, dy)
    @image.copy(image2gd(image), x, y, sx, sy, dx, dy)
  end

  def copy_merge(image, x, y, sx, sy, dx, dy, percent)
    @image.copyMerge(image2gd(image), x, y, sx, sy, dx, dy, percent)
  end

  def copy_merge_gray(image, x, y, sx, sy, dx, dy, percent)
    @image.copyMergeGray(image2gd(image), x, y, sx, sy, dx, dy, percent)
  end

  def copy_resampled(image, x, y, sx, sy, dx, dy, sdx, sdy)
    @image.copyResampled(image2gd(image), x, y, sx, sy, dx, dy, sdx, sdy)
  end

  def copy_resized(image, x, y, sx, sy, dx, dy, sdx, sdy)
    @image.copyResized(image2gd(image), x, y, sx, sy, dx, dy, sdx, sdy)
  end

end # GRb

#==============================================================================#
#==============================================================================#
