
#==============================================================================#
# gruby/font.rb
# $Id: font.rb,v 1.2 2002/09/18 15:43:59 yuya Exp $
#==============================================================================#

#==============================================================================#
# GRb Class
class GRb

  #============================================================================#
  # Font Class
  class Font

    #==========================================================================#
    # Initialize Method

    def initialize(font)
      @font = font
    end

    #==========================================================================#
    # Class Methods

    def self.gd(name, bold = false, italic = false)
      return self.new(GDFont.new(name, bold, italic))
    end

    def self.truetype(filepath, size)
      return self.new(TrueTypeFont.new(filepath, size))
    end

    #==========================================================================#
    # Instance Methods

    def draw(image, x, y, angle, color, text)
      @font.draw(image, x, y, angle, color, text)
    end

    def width(text)
      return @font.width(text)
    end

    def height(text)
      return @font.height(text)
    end

    #==========================================================================#
    # GDFont Class
    class GDFont

      #========================================================================#
      # Class Constants

      R = 0.5

      #========================================================================#
      # Initialize Method

      def initialize(name, bold, italic)
        @font   = ::GD::Font.new(name)
        @bold   = bold
        @italic = italic
      end

      #========================================================================#
      # Private Instance Methods

      def draw_normal(image, x, y, angle, color, text)
        if angle % 360 == 90
          image.stringUp(@font, x, y, text, color)
        else
          image.string(@font, x, y, text, color)
        end
      end
      private :draw_normal

      def draw_bold(image, x, y, angle, color, text)
        draw_normal(image, x, y, angle, color, text)
        if angle % 360 == 90
          draw_normal(image, x, y + 1, angle, color, text)
        else
          draw_normal(image, x + 1, y, angle, color, text)
        end
      end
      private :draw_bold

      def draw_italic(image, x, y, angle, color, text)
        fdx = self.raw_width(text)
        fdy = self.raw_height(text)
        ndx = fdx + (fdy * R).floor
        ndy = fdy
        red, green, blue = image.rgb(color)

        if angle % 360 == 90
          fdx, fdy = fdy, fdx
          ndx, ndy = ndy, ndx
        end

        dummy1 = ::GD::Image.new(fdx, fdy)
        dummy1.colorAllocate(255, 255, 255)
        dummy1.transparent(0)

        if angle % 360 == 90
          dummy1.stringUp(@font, 0, fdy - 1, text, dummy1.colorAllocate(red, green, blue))
        else
          dummy1.string(@font, 0, 0, text, dummy1.colorAllocate(red, green, blue))
        end

        dummy2 = ::GD::Image.new(ndx, ndy)
        dummy2.colorAllocate(255, 255, 255)
        dummy2.colorAllocate(red, green, blue)
        dummy2.transparent(0)

        if angle % 360 == 90
          (0...fdx).each { |i|
            from_x = i
            from_y = 0
            to_x   = i
            to_y   = (i * R).floor
            width  = 1
            height = fdy
            dummy1.copy(dummy2, to_x, to_y, from_x, from_y, width, height)
          }
          dummy2.copy(image, x, y - fdy, 0, 0, ndx, ndy)
        else
          (0...fdy).each { |i|
            from_x = 0
            from_y = i
            to_x   = ((fdy - i) * R).floor
            to_y   = i
            width  = fdx
            height = 1
            dummy1.copy(dummy2, to_x, to_y, from_x, from_y, width, height)
          }
          dummy2.copy(image, x, y, 0, 0, ndx, ndy)
        end
      end
      private :draw_italic

      def draw_bolditalic(image, x, y, angle, color, text)
        draw_italic(image, x, y, angle, color, text)
        if angle % 360 == 90
          draw_italic(image, x, y + 1, angle, color, text)
        else
          draw_italic(image, x + 1, y, angle, color, text)
        end
      end
      private :draw_bolditalic

      #========================================================================#
      # Instance Methods

      def draw(image, x, y, angle, color, text)
        case [@bold, @italic]
        when [false, false] then draw_normal(image, x, y, angle, color, text)
        when [true, false]  then draw_bold(image, x, y, angle, color, text)
        when [false, true]  then draw_italic(image, x, y, angle, color, text)
        when [true, true]   then draw_bolditalic(image, x, y, angle, color, text)
        end
      end

      def width(text)
        return case [@bold, @italic]
               when [false, false] then @font.width * text.size
               when [true, false]  then @font.width * text.size + 1
               when [false, true]  then @font.width * text.size + @font.height * R
               when [true, true]   then @font.width * text.size + @font.height * R + 1
               end
      end

      def height(text)
        return @font.height
      end

      def raw_width(text)
        return @font.width * text.size
      end

      def raw_height(text)
        return @font.height
      end

    end # GDFont

    #==========================================================================#
    # TrueTypeFont Class
    class TrueTypeFont

      #========================================================================#
      # Initialize Method

      def initialize(filepath, size)
        @filepath = filepath
        @size     = size
      end

      #========================================================================#
      # Instance Method

      def draw(image, x, y, angle, color, text)
        image.stringFT(color, @filepath, @size, ::Math::PI / 180 * angle, x, y, text)
      end

      def width(text)
        raise NotImplementedError
      end

      def height(text)
        raise NotImplementedError
      end

    end # TrueTypeFont

    #==========================================================================#
    # Class Constants

    # Normal Font
    Giant  = self.gd('Giant')
    Large  = self.gd('Large')
    Medium = self.gd('Medium')
    Small  = self.gd('Small')
    Tiny   = self.gd('Tiny')

    # Bold Font
    BoldGiant  = self.gd('Giant',  true)
    BoldLarge  = self.gd('Large',  true)
    BoldMedium = self.gd('Medium', true)
    BoldSmall  = self.gd('Small',  true)
    BoldTiny   = self.gd('Tiny',   true)

    # Italic Font
    ItalicGiant  = self.gd('Giant',  false, true)
    ItalicLarge  = self.gd('Large',  false, true)
    ItalicMedium = self.gd('Medium', false, true)
    ItalicSmall  = self.gd('Small',  false, true)
    ItalicTiny   = self.gd('Tiny',   false, true)

    # Bold Italic Font
    BoldItalicGiant  = self.gd('Giant',  true, true)
    BoldItalicLarge  = self.gd('Large',  true, true)
    BoldItalicMedium = self.gd('Medium', true, true)
    BoldItalicSmall  = self.gd('Small',  true, true)
    BoldItalicTiny   = self.gd('Tiny',   true, true)

  end # Font

end # GRb

#==============================================================================#
#==============================================================================#
