#define BOOST_AUTO_TEST_MAIN
#include <boost/test/auto_unit_test.hpp>
#include <gslib/glsl_math/glsl_math.h>

using namespace gslib::glsl_math;

bool isNearEqual( float a, float b ) {
	return fabsf( a - b ) < 0.0001f;
}
template < size_t Size >
bool isNearEqual( const tuple< Size >& a, const tuple< Size >& b ) {
	return distance( a, b ) < 0.0001f;
}

BOOST_AUTO_UNIT_TEST( test_glsl_math ) {
	vec4 a( 1, 2, 3, 4 );
	vec4::iterator it = a.begin();
	float len = length( a );
	BOOST_CHECK( isNearEqual( sqrtf( 1 + 4 + 9 + 16 ), len ) );
	vec4 b( a );
	b.operator *= ( 2 );
	BOOST_CHECK( isNearEqual( vec4( 2, 4, 6, 8 ), b ) );
	a *= b;
	BOOST_CHECK( isNearEqual( vec4( 2, 8, 18, 32 ), a ) );

	//	test 'cross'
	BOOST_CHECK( isNearEqual( vec3( 0, 0, 1 ), cross( vec3( 1, 0, 0 ), vec3( 0, 1, 0 ) ) ) );
	BOOST_CHECK( isNearEqual( vec3( 1, 0, 0 ), cross( vec3( 0, 1, 0 ), vec3( 0, 0, 1 ) ) ) );
	BOOST_CHECK( isNearEqual( vec3( 0, 1, 0 ), cross( vec3( 0, 0, 1 ), vec3( 1, 0, 0 ) ) ) );
	
	mat2 m( 1, 2, 3, 4 );
	vec2 v( 1, 2 );
	
	//	m * v = ( 1, 3 ) * ( 1 )
	//          ( 2, 4 )   ( 2 )
	BOOST_CHECK( isNearEqual( vec2( 7, 10 ), m * v ) );

	//	v * m = ( 1, 2 ) * ( 1, 3 )
	//                     ( 2, 4 )
	BOOST_CHECK( isNearEqual( vec2( 5, 11 ), v * m ) );
	
	mat2 m2( 5, 6, 7, 8 );
	BOOST_CHECK( isNearEqual( mat2( 19, 43, 22, 50 ), m * m2 ) );
	BOOST_CHECK( isNearEqual( mat2( 23, 31, 34, 46 ), m2 * m ) );
	
	//	quat
	quat q1( 1, 0, 0, 0 );
	quat r( vec3( 0, 0, 1 ), radians( 90 ) );
	BOOST_CHECK( isNearEqual( quat( 0, 1, 0, 0 ), r * q1 * conj( r ) ) );
	quat q2( 1, 2, 3, 4 );
	BOOST_CHECK( isNearEqual( quat::identity(), inverse( q2 ) * q2 ) );
	
	//	inverse
	mat2 invM = inverse( m );
	BOOST_CHECK( isNearEqual( mat2::identity(), m * invM ) );
	mat3 m3( 1, 2, 3, 4, 0, 6, 7, 8, 9 );
	mat3 invM3 = inverse( m3 );
	BOOST_CHECK( isNearEqual( mat3::identity(), m3 * invM3 ) );
	mat4 m4( 1, 2, 3, 4, 0, 6, 7, 8, 9, 10, 11, 12, -13, 14, -15, 16 );
	mat4 invM4 = inverse( m4 );
	BOOST_CHECK( isNearEqual( mat4::identity(), m4 * invM4 ) );
	
	//	lerp
	BOOST_CHECK( isNearEqual( 100, lerp( 100, 200, 0 ) ) );
	BOOST_CHECK( isNearEqual( 150, lerp( 100, 200, 0.5 ) ) );
	BOOST_CHECK( isNearEqual( 200, lerp( 100, 200, 1 ) ) );
	{
		vec3 a( 1, 2, 3 ), b( 4, 5, 6 );
		BOOST_CHECK( isNearEqual( a, lerp( a, b, 0 ) ) );
		BOOST_CHECK( isNearEqual( b, lerp( a, b, 1 ) ) );
		BOOST_CHECK( isNearEqual( ( a + b ) * 0.5f, lerp( a, b, 0.5f ) ) );
	}
}
