/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * The original code is gtkimcontextxcim.c in GTK+. 
 * Copyright (C) 2000 Red Hat, Inc.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "prime-im-context.h"

#include <glib/gi18n.h>
#include <gdk/gdkkeysyms.h>

#include "prime.h"
#include "prime-candidate-window.h"

typedef struct _PrimeIMContextPrivate PrimeIMContextPrivate;

typedef enum
{
	PRIME_IM_MODE_NONE,
	PRIME_IM_MODE_FUNDAMENTAL,
	PRIME_IM_MODE_INPUT,
	PRIME_IM_MODE_CONVERSION,
	PRIME_IM_N_MODE
} PrimeIMMode;

struct _PrimeIMContextPrivate
{
	Prime *prime;

	GdkWindow *client_window;
	GtkWidget *widget;
	PrimeCandidateWindow *candidate_window;

	GString *preedit_str;
	gint preedit_cursor_pos; /* cursor position from the tail of input string */
				 /* 0 is the tail. */
	gboolean is_learning;
	GString *learning_str;
	gint learning_cursor_pos;
	GString *pronounce_str;

	PrimeIMMode mode;
};

#define PRIME_IM_CONTEXT_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), PRIME_TYPE_IM_CONTEXT, PrimeIMContextPrivate))

static void	prime_im_context_class_init    (PrimeIMContextClass *klass);
static void     prime_im_context_init          (PrimeIMContext      *context);
static void     prime_im_context_dispose       (GObject        *object);
static void     prime_im_context_finalize      (GObject        *object);

static void     im_context_set_client_window   (GtkIMContext   *context,
					        GdkWindow      *client_window);
static gboolean im_context_filter_keypress     (GtkIMContext   *context,
                                                GdkEventKey    *event);
static void     im_context_get_preedit_string  (GtkIMContext   *context,
				                gchar         **str,
				                PangoAttrList **attrs,
				                gint           *cursor_pos);
static void     im_context_focus_in            (GtkIMContext   *context);
static void     im_context_focus_out           (GtkIMContext   *context);
static void     im_context_set_cursor_location (GtkIMContext   *context,
					        GdkRectangle   *area);
static void     im_context_reset               (GtkIMContext   *context);

static PrimeIMMode prime_im_context_get_mode   (PrimeIMContext *context);
static void        prime_im_context_set_mode   (PrimeIMContext *context,
						PrimeIMMode     mode);

static GtkWidget *widget_for_window (GdkWindow *window);

static GtkIMContextClass *parent_class = NULL;

void
prime_im_context_register_type (GTypeModule *type_module)
{
	if (!prime_type_im_context)
	{
		static const GTypeInfo context_info =
		{
			sizeof (PrimeIMContextClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) prime_im_context_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (PrimeIMContext),
			0,		/* n_preallocs */
			(GInstanceInitFunc) prime_im_context_init,
		};

		prime_type_im_context = 
			g_type_module_register_type (type_module,
						     GTK_TYPE_IM_CONTEXT,
					             "PrimeIMContext",
					             &context_info, 0);
	}
}

static void
prime_im_context_class_init (PrimeIMContextClass *klass)
{
	GObjectClass *gobject_class;
  	GtkIMContextClass *imcontext_class = GTK_IM_CONTEXT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
  
	gobject_class->dispose  = prime_im_context_dispose;
	gobject_class->finalize = prime_im_context_finalize;

        imcontext_class->filter_keypress     = im_context_filter_keypress;
        imcontext_class->set_client_window   = im_context_set_client_window;
  	imcontext_class->get_preedit_string  = im_context_get_preedit_string;
	imcontext_class->focus_in            = im_context_focus_in;
	imcontext_class->focus_out           = im_context_focus_out;
	imcontext_class->set_cursor_location = im_context_set_cursor_location;
	imcontext_class->reset               = im_context_reset;

	g_type_class_add_private (gobject_class, sizeof(PrimeIMContextPrivate));
}

static void
prime_im_context_init (PrimeIMContext *context)
{
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	priv->client_window    = NULL;
	priv->candidate_window = prime_candidate_window_new();
	priv->preedit_str      = g_string_sized_new (0);
	priv->mode             = PRIME_IM_MODE_NONE;
	priv->prime	       = NULL;
	priv->is_learning      = FALSE;
	priv->learning_str     = NULL;
	priv->pronounce_str         = NULL;
	
	priv->preedit_cursor_pos = 0;
}

static void
prime_im_context_dispose (GObject *object)
{
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (object);

	if (priv->client_window)
		g_object_unref (priv->client_window);

	if (priv->candidate_window)
		gtk_widget_destroy (GTK_WIDGET (priv->candidate_window));

	if (priv->prime)
		g_object_unref (priv->prime);

	if (priv->preedit_str)
		g_string_free (priv->preedit_str, TRUE);
	
	if (priv->learning_str)
		g_string_free (priv->learning_str, TRUE);
	
	if (priv->pronounce_str)
		g_string_free (priv->pronounce_str, TRUE);
	
	priv->client_window    = NULL;
	priv->candidate_window = NULL;
	priv->preedit_str      = NULL;
	priv->learning_str     = NULL;
	priv->pronounce_str    = NULL;
	priv->prime            = NULL;

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
prime_im_context_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkIMContext *
prime_im_context_new (void)
{
	return g_object_new (PRIME_TYPE_IM_CONTEXT, NULL);
}


void
prime_im_context_shutdown (void)
{
}


static void
im_context_set_client_window  (GtkIMContext *context,
			       GdkWindow    *client_window)
{
	GtkWidget *widget;
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	if (priv->client_window)
		g_object_unref(priv->client_window);
	priv->client_window = NULL;

	if (client_window)
		priv->client_window = g_object_ref(client_window);

	priv->widget = widget_for_window (client_window);
}


static void
preedit_string_append_c (PrimeIMContext *context, guint32 c)
{
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);
 	guchar utf8[10];
	int len = g_unichar_to_utf8 (c, utf8);

	utf8[len] = 0;
	priv->preedit_str = g_string_insert (priv->preedit_str,
					     priv->preedit_str->len - priv->preedit_cursor_pos,
						     utf8);
	prime_im_context_set_mode (PRIME_IM_CONTEXT (context),
				   PRIME_IM_MODE_INPUT);
	g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");
}

static gboolean
fundamental_mode_process_key_event (PrimeIMContext *context,
			            GdkEventKey *event)
{
  	guint32 c;
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	if (priv->is_learning)
	{
		switch (event->keyval)
		{
		case GDK_Return:
		{
			gchar **input;
			gchar *buf;
			
			if (priv->learning_str->len == 0)
				return TRUE;
			
			priv->is_learning = FALSE;

			/* learning a word */
			input = prime_preedit_convert_input (priv->prime, 
							     priv->pronounce_str->str);
			buf = g_strconcat (input[0], input[1], NULL);
			prime_learn_word (priv->prime, 
					  buf,
					  priv->learning_str->str,
					  "",
					  "",
					  "",
					  "");
			g_free (buf);
			g_strfreev (input);

			/* commit a learning word */
			g_signal_emit_by_name (G_OBJECT (context), "commit",
					       priv->learning_str->str);

			priv->preedit_str = g_string_truncate (priv->preedit_str, 0);

			g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");
			return TRUE;
			break;
		}
		case GDK_Escape:
			if (priv->preedit_str)
				g_string_free (priv->preedit_str, TRUE);
			priv->preedit_str = g_string_new_len (priv->pronounce_str->str,
							      priv->pronounce_str->len);
			priv->is_learning = FALSE;
			prime_im_context_set_mode (PRIME_IM_CONTEXT (context),
						   PRIME_IM_MODE_INPUT);
			g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");
			return TRUE;
			break;
		case GDK_space:
			return TRUE;
			break;
		case GDK_BackSpace:
			if (priv->learning_str->len > 0)
			{
				glong len;
				gchar *dummy;
			
				len = g_utf8_strlen (priv->learning_str->str,
						     priv->learning_str->len);
				dummy = g_new0 (gchar, len);
				dummy = g_utf8_strncpy (dummy, 
							priv->learning_str->str,
							len - 1);

				priv->learning_str = g_string_assign (priv->learning_str,
								      dummy);
				g_free (dummy);
			
				g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");
				return TRUE;
			}
			return TRUE;
			break;
		default:
			break;
		}

		c = gdk_keyval_to_unicode (event->keyval);
	 	if (c)
		{
			preedit_string_append_c (context, c);
		}

		return TRUE;
	}

	if (event->keyval == GDK_space)
	{
		g_signal_emit_by_name (G_OBJECT (context), "commit", " ");
		return TRUE;
	}

	c = gdk_keyval_to_unicode (event->keyval);
 	if (c)
	{
		preedit_string_append_c (context, c);
		return TRUE;
	}

	return FALSE;
}

static gboolean
input_mode_process_key_event (PrimeIMContext *context,
			      GdkEventKey *event)
{
	gchar *string;
  	guint32 c;
	guint index;
	gboolean preedit_changed = FALSE;
	gchar **cand_list = NULL;
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);
	switch (event->keyval)
	{
	case GDK_Return:
		string = g_strndup (priv->preedit_str->str, 
				    priv->preedit_str->len - priv->preedit_cursor_pos);
		cand_list = prime_preedit_convert_input (priv->prime, string);
		g_free (string);

		if (cand_list && cand_list[0])
		{
			string = g_strconcat (cand_list[0],
					      cand_list[1],
					      NULL);
			g_strfreev (cand_list);
			cand_list = NULL;
		}
		else
		{
			string = g_strdup (priv->preedit_str->str);
		}
		
		if (priv->is_learning)
		{
			priv->learning_str = g_string_append (priv->learning_str,
							      string);

		}
		else
		{
			g_signal_emit_by_name (G_OBJECT (context), "commit",
					       string);
		}
		g_free (string);
		g_string_erase (priv->preedit_str, 
				0,
				priv->preedit_str->len - priv->preedit_cursor_pos);
		prime_im_context_set_mode (context,
					   PRIME_IM_MODE_FUNDAMENTAL);
		prime_candidate_window_set_candidates 
			(priv->candidate_window, NULL);
		priv->preedit_cursor_pos = 0;
		preedit_changed = TRUE;	
		break;
	case GDK_BackSpace:
		if (priv->preedit_cursor_pos >= priv->preedit_str->len)
			return TRUE;
		priv->preedit_str = g_string_erase (priv->preedit_str, 
						    priv->preedit_str->len - priv->preedit_cursor_pos - 1,
						    1);
		if (priv->preedit_str->len == 0)
		{
			prime_im_context_set_mode (context,
						   PRIME_IM_MODE_FUNDAMENTAL);
		}
		preedit_changed = TRUE;	
		break;
	case GDK_Left:
		if (priv->preedit_cursor_pos >= priv->preedit_str->len)
			return TRUE;
		/*priv->preedit_cursor_pos++;*/
		preedit_changed = TRUE;	
		break;
	case GDK_Right:
		if (priv->preedit_cursor_pos < 1)
			return TRUE;
		/*priv->preedit_cursor_pos--;*/
		preedit_changed = TRUE;	
		break;
	case GDK_space:
	case GDK_Up:
	case GDK_Down:
		prime_im_context_set_mode (context,
					   PRIME_IM_MODE_CONVERSION);
		string = g_strndup (priv->preedit_str->str,
				    priv->preedit_str->len - priv->preedit_cursor_pos);
		cand_list = prime_lookup (priv->prime, string);
		g_free (string);

		prime_candidate_window_set_candidates 
			(priv->candidate_window, cand_list);
		if (cand_list)
			g_strfreev (cand_list);
		preedit_changed = TRUE;	
		break;
	case GDK_Escape:
		if (priv->is_learning)
		{
			if (priv->preedit_str)
				g_string_free (priv->preedit_str, TRUE);
			priv->preedit_str = g_string_new_len (priv->pronounce_str->str,
							      priv->pronounce_str->len);
			priv->is_learning = FALSE;
			prime_im_context_set_mode (context,
						   PRIME_IM_MODE_INPUT);
			preedit_changed = TRUE;	
		}
		break;
	default:
		c = gdk_keyval_to_unicode (event->keyval);
	  	if (c)
 	   	{
 	     		guchar utf8[10];
	      		int len = g_unichar_to_utf8 (c, utf8);
	      		utf8[len] = 0;
#if 0
			if (g_unichar_isupper (c))
			{
				string = g_strndup (priv->preedit_str->str, 
						    priv->preedit_str->len - priv->preedit_cursor_pos);
				cand_list = prime_preedit_convert_input (priv->prime, string);
				g_free (string);

				if (cand_list && cand_list[0])
				{
					string = g_strconcat (cand_list[0],
							      cand_list[1],
							      NULL);
					g_strfreev (cand_list);
					cand_list = NULL;
				}
				else
				{
					string = g_strdup (priv->preedit_str->str);
				}

				if (priv->is_learning)
				{
					priv->learning_str = g_string_append (priv->learning_str,
									      string);
				}
				else
				{
					g_signal_emit_by_name (G_OBJECT (context), "commit",
							       string);
				}
				g_free (string);
				g_string_erase (priv->preedit_str, 
						0,
						priv->preedit_str->len - priv->preedit_cursor_pos);
			}
#endif
			priv->preedit_str = g_string_insert (priv->preedit_str,
							      priv->preedit_str->len - priv->preedit_cursor_pos,
							      utf8);
			preedit_changed = TRUE;	
    		}
		break;
	}
	if (preedit_changed)
		g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");

	return TRUE;
}

static gboolean
conversion_mode_process_key_event (PrimeIMContext *context,
				   GdkEventKey *event)
{
	gchar *string;
  	guint32 c;
	guint index;
	gboolean preedit_changed = FALSE;
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	switch (event->keyval)
	{
	case GDK_Home:
		/* learning a word */
		if (!priv->is_learning && event->state == GDK_CONTROL_MASK)
		{
			string = g_strndup (priv->preedit_str->str, 
				priv->preedit_str->len - priv->preedit_cursor_pos);
			priv->is_learning = TRUE;
			if (priv->learning_str)
				g_string_free (priv->learning_str, TRUE);
			priv->learning_str = g_string_sized_new (0);

			/* copy preedit string to read string */
			if (priv->pronounce_str)
				g_string_free (priv->pronounce_str, TRUE);
			priv->pronounce_str = g_string_new (priv->preedit_str->str);
			
			/* reinitialize preedit string */
			g_string_truncate (priv->preedit_str, 0);

			prime_im_context_set_mode (context,
					           PRIME_IM_MODE_FUNDAMENTAL);
			prime_candidate_window_set_candidates 
				(priv->candidate_window, NULL);
			priv->preedit_cursor_pos = 0;
			preedit_changed = TRUE;	
		}
		break;
	case GDK_Return:
		string = prime_candidate_window_get_candidate
				(priv->candidate_window);

		if (priv->is_learning)
		{
			priv->learning_str = g_string_append (priv->learning_str,
							      string);
		}
		else
		{
			g_signal_emit_by_name (G_OBJECT (context), "commit",
					       string);
		}
		g_free (string);
		g_string_erase (priv->preedit_str, 
				0,
				priv->preedit_str->len - priv->preedit_cursor_pos);
		prime_im_context_set_mode (context,
					   PRIME_IM_MODE_FUNDAMENTAL);
		prime_candidate_window_set_candidates 
			(priv->candidate_window, NULL);
		priv->preedit_cursor_pos = 0;
		preedit_changed = TRUE;	
		break;
	case GDK_Left:
	case GDK_Page_Up:
		/* go to the previous candidate page */
		prime_candidate_window_prev_page (priv->candidate_window);
		preedit_changed = TRUE;	
		break;
	case GDK_Right:
	case GDK_Page_Down:
		/* go to the next candidate page */
		prime_candidate_window_next_page (priv->candidate_window);
		preedit_changed = TRUE;	
		break;
	case GDK_space:
	case GDK_Down:
		prime_candidate_window_select_next_candidate
			(priv->candidate_window);
		preedit_changed = TRUE;	
		break;
	case GDK_Up:
		prime_candidate_window_select_prev_candidate
			(priv->candidate_window);
		preedit_changed = TRUE;	
		break;
	case GDK_BackSpace:
	case GDK_Escape:
		prime_im_context_set_mode (context,
					   PRIME_IM_MODE_INPUT);
		prime_candidate_window_set_candidates 
				(priv->candidate_window, NULL);
		preedit_changed = TRUE;	
		break;
	case GDK_0:
	case GDK_1:
	case GDK_2:
	case GDK_3:
	case GDK_4:
	case GDK_5:
	case GDK_6:
	case GDK_7:
	case GDK_8:
	case GDK_9:
		if (event->keyval != GDK_0)
			index = event->keyval - GDK_1;
		else
			index = event->keyval - GDK_1 + 10;

		string = prime_candidate_window_get_candidate_from_index
				(priv->candidate_window, index);

		if (!string)
			return TRUE;
		if (priv->is_learning)
		{
			priv->learning_str = g_string_append (priv->learning_str,
							      string);
		}
		else
		{
			g_signal_emit_by_name (G_OBJECT (context), "commit",
					       string);
		}
		g_free (string);
		g_string_erase (priv->preedit_str, 
				0,
				priv->preedit_str->len - priv->preedit_cursor_pos);
		prime_im_context_set_mode (context,
					   PRIME_IM_MODE_FUNDAMENTAL);
		prime_candidate_window_set_candidates 
			(priv->candidate_window, NULL);
		priv->preedit_cursor_pos = 0;
		preedit_changed = TRUE;	
		break;
	default:
		c = gdk_keyval_to_unicode (event->keyval);
	  	if (c)
	    	{
			string = prime_candidate_window_get_candidate
					(priv->candidate_window);
			if (priv->is_learning)
			{
				priv->learning_str = g_string_append (priv->learning_str,
								      string);
			}
			else
			{
				g_signal_emit_by_name (G_OBJECT (context), "commit",
						       string);
			}
			g_free (string);
			g_string_erase (priv->preedit_str, 
					0,
					priv->preedit_str->len - priv->preedit_cursor_pos);
		
			preedit_string_append_c (context, c);
			prime_candidate_window_set_candidates 
				(priv->candidate_window, NULL);
			preedit_changed = TRUE;	
    		}
		break;
	}
	if (preedit_changed)
		g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");

	return TRUE;
}


static gboolean 
im_context_filter_keypress (GtkIMContext *context,
                            GdkEventKey  *event)
{
  	guint32 c;
	gboolean ret;
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);
	gboolean preedit_changed = FALSE;

  	if (event->type == GDK_KEY_RELEASE)
    		return FALSE;

	/* toggle japanese input mode */
	if (event->state & GDK_SHIFT_MASK && event->keyval == GDK_space)
	{
		prime_candidate_window_set_candidates 
			(priv->candidate_window, NULL);
		priv->is_learning = FALSE;

		if (priv->mode == PRIME_IM_MODE_NONE)
		{
			if (!priv->prime)
				priv->prime = prime_new();
			gtk_widget_show (GTK_WIDGET (priv->candidate_window));
			prime_im_context_set_mode (PRIME_IM_CONTEXT (context),
						   PRIME_IM_MODE_FUNDAMENTAL);
		}
		else
		{
			if (priv->prime)
				g_object_unref (priv->prime);
			priv->prime = NULL;
			
			gtk_widget_hide (GTK_WIDGET (priv->candidate_window));
			prime_im_context_set_mode (PRIME_IM_CONTEXT (context),
						   PRIME_IM_MODE_NONE);
			g_string_truncate (priv->preedit_str, 0);
			g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");
		}
		return TRUE;
	}

	switch (priv->mode)
	{
	case PRIME_IM_MODE_NONE:
		{	
		guchar utf8[10];
    		gint len;
		c = gdk_keyval_to_unicode (event->keyval);

		if (!c) return FALSE;
    
		len = g_unichar_to_utf8 (c, utf8);
		utf8[len] = 0;
	  	g_signal_emit_by_name (G_OBJECT (context), "commit",
				       utf8);
		return TRUE;
		break;
		}
	case PRIME_IM_MODE_CONVERSION:
		conversion_mode_process_key_event (PRIME_IM_CONTEXT (context),
						   event);
		return TRUE;
		break;
	case PRIME_IM_MODE_INPUT:
		input_mode_process_key_event (PRIME_IM_CONTEXT (context),
					      event);
		return TRUE;
		break;
	case PRIME_IM_MODE_FUNDAMENTAL:
	default:
		ret = fundamental_mode_process_key_event (PRIME_IM_CONTEXT (context),
						          event);
		if (ret) return TRUE;
		break;
	}

	return FALSE;
}

static void
im_context_get_preedit_string (GtkIMContext   *context,
		               gchar         **str,
		               PangoAttrList **attrs,
		               gint           *cursor_pos)
{
	gchar *preedit_string, *whole_string;
	GdkColor fg, bg;

	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);
	
	switch (priv->mode)
	{
	case PRIME_IM_MODE_CONVERSION:
		preedit_string = prime_candidate_window_get_candidate 
					(priv->candidate_window);
		whole_string = g_strdup (preedit_string);
		break;
	case PRIME_IM_MODE_INPUT:
	{
		gchar **input;
		gchar *string;
		string = g_strndup (priv->preedit_str->str, 
				    priv->preedit_str->len - priv->preedit_cursor_pos);
		if (priv->preedit_cursor_pos == 0)
			input = prime_preedit_convert_input (priv->prime, string);
		else
		{
			input = prime_lookup_exact (priv->prime, string);
			if (!input)
				input = prime_get_label (priv->prime, string);
		}
		g_free (string);
	
		if (!input)
		{
			whole_string = g_strdup (priv->preedit_str->str);
			preedit_string = g_strdup ("");
		}
		else
		{
			if (priv->preedit_cursor_pos == 0)
			{
				preedit_string = g_strconcat (input[0], input[1], NULL);
				whole_string = g_strdup (preedit_string);
			}
			else
			{
				gchar *tail;
				tail = g_strndup (priv->preedit_str->str + priv->preedit_str->len - priv->preedit_cursor_pos,
						  priv->preedit_cursor_pos);
				preedit_string = g_strconcat (input[0], NULL);

				whole_string = g_strconcat (preedit_string, tail, NULL);
				g_free (tail);
			}
			g_strfreev (input);
		}
		break;
	}
	case PRIME_IM_MODE_FUNDAMENTAL:
	default:
		whole_string = g_strdup ("");
		preedit_string = g_strdup ("");
		break;
	}

	if (str)
		*str = whole_string;
	else
		g_free (whole_string);

  	if (attrs)
    	{
	    	PangoAttribute *attr;
		
		*attrs = pango_attr_list_new ();
			
		if (priv->is_learning)
		{
			gchar **input;
			gchar *buf;
			gchar *dummy;
			gint preedit_start, preedit_end;
			input = prime_preedit_convert_input (priv->prime, 
							     priv->pronounce_str->str);
			buf = g_strconcat (input[0], input[1], NULL);
			*str = g_strdup_printf (_("Learning[%s|%s%s]"), 
						buf,
						priv->learning_str->str,
						whole_string);
			g_strfreev (input);
	      		
			fg = priv->widget->style->fg [GTK_STATE_SELECTED];
			bg = priv->widget->style->bg [GTK_STATE_SELECTED];

      			attr = pango_attr_foreground_new (fg.red, fg.green, fg.blue);
			dummy = g_strdup_printf (_("Learning[%s|"), buf);
			attr->start_index = 0;
      			attr->end_index = strlen (dummy);
      			pango_attr_list_change (*attrs, attr);
      			
			attr = pango_attr_background_new (bg.red, bg.green, bg.blue);
      			attr->start_index = 0;
      			attr->end_index = strlen (dummy);
			pango_attr_list_change (*attrs, attr);

			/* preedit string */
			preedit_start = strlen (dummy) + priv->learning_str->len;
			preedit_end   = preedit_start + strlen (whole_string);
      			if (priv->mode == PRIME_IM_MODE_CONVERSION)
			{
				attr = pango_attr_foreground_new (0xffff, 0xffff, 0xffff);
				attr->start_index = preedit_start;
				attr->end_index   = preedit_end;
				pango_attr_list_change (*attrs, attr);

	      			attr = pango_attr_background_new (0, 0, 0);
				attr->start_index = preedit_start;
				attr->end_index   = preedit_end;
				pango_attr_list_change (*attrs, attr);
			}
			else
			{
				attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
				attr->start_index = preedit_start;
				attr->end_index   = preedit_end;
				pango_attr_list_change (*attrs, attr);
			}
			/**/
      			attr = pango_attr_foreground_new (fg.red, fg.green, fg.blue);
			attr->start_index = strlen (*str) - 1;
      			attr->end_index = strlen (*str);
      			pango_attr_list_change (*attrs, attr);
      			
			attr = pango_attr_background_new (bg.red, bg.green, bg.blue);
			attr->start_index = strlen (*str) - 1;
      			attr->end_index = strlen (*str);
			pango_attr_list_change (*attrs, attr);

			g_free (buf);
			g_free (whole_string);
			g_free (dummy);
		}
		else
		{
		switch (priv->mode)
		{
		case PRIME_IM_MODE_CONVERSION:
      			attr = pango_attr_foreground_new (0xffff, 0xffff, 0xffff);
			attr->start_index = 0;
      			attr->end_index = strlen (*str);
      			pango_attr_list_change (*attrs, attr);

      			attr = pango_attr_background_new (0, 0, 0);
      			attr->start_index = 0;
      			attr->end_index = strlen (*str);
			pango_attr_list_change (*attrs, attr);
			break;
		default:	
  	  		attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
    			attr->start_index = 0;
    			attr->end_index = strlen (*str);
			pango_attr_list_change (*attrs, attr);
			break;
		}
		}
	}

  	if (cursor_pos)
	{
		if (priv->is_learning)
			*cursor_pos = g_utf8_strlen (*str, -1) - 1; 
		else
			*cursor_pos = g_utf8_strlen (preedit_string, -1);
	}

	g_free (preedit_string);
}

static void
im_context_focus_in (GtkIMContext *context)
{
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	if (priv->mode != PRIME_IM_MODE_NONE)
		gtk_widget_show (GTK_WIDGET (priv->candidate_window));
}

static void
im_context_focus_out (GtkIMContext *context)
{
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	gtk_widget_hide (GTK_WIDGET (priv->candidate_window));
}


static void
im_context_set_cursor_location (GtkIMContext   *context,
				GdkRectangle   *area)
{
	gint x, y;
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	if (!priv->client_window) return;
	if (priv->mode == PRIME_IM_MODE_INPUT ||
	    priv->mode == PRIME_IM_MODE_CONVERSION) return;

	gdk_window_get_origin (priv->client_window, &x, &y);
	gtk_window_move (GTK_WINDOW (priv->candidate_window), 
			 x + area->x, y + area->y + area->height);
}


static void
im_context_reset (GtkIMContext *context)
{
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);
#if 0
	g_string_truncate (priv->preedit_str, 0);
	g_signal_emit_by_name (G_OBJECT (context), "preedit_changed");
#endif
}

static PrimeIMMode
prime_im_context_get_mode (PrimeIMContext *context)
{ 
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	return priv->mode;
}

static void
prime_im_context_set_mode (PrimeIMContext *context, PrimeIMMode mode)
{
	PrimeIMContextPrivate *priv = PRIME_IM_CONTEXT_GET_PRIVATE (context);

	priv->mode = mode;
}

/* Finds the GtkWidget that owns the window, or if none, the
 * widget owning the nearest parent that has a widget.
 */
static GtkWidget *
widget_for_window (GdkWindow *window)
{
	while (window)
	{
		gpointer user_data;
		gdk_window_get_user_data (window, &user_data);
		if (user_data)
			return user_data;

		window = gdk_window_get_parent (window);
    	}

	return NULL;
}
