/*
    disw32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "header.h"
#include "data.h"
#include "ordinal.h"
#include "misc/peimage.h"


/******************************************************************************
*                                                                             *
* ja:ヘッダ解析関数                                                           *
*                                                                             *
******************************************************************************/
static gboolean
dw32_header_resource_directory (Dw32Info    *info,
                                const gint   index,
                                const gchar *name,
                                const gint   depth)
{
  gint i, count;
  guint32 address;
  ImageResourceDirectory *ird;
  ImageResourceDirectoryEntry *irde;
  const static Dw32Data dataird[] = {{4, "Characteristics"},
                                     {4, "Time/Date Stamp"},
                                     {2, "Major Version"},
                                     {2, "Minor Version"},
                                     {2, "Number of Name Entries"},
                                     {2, "Number of ID Entries"},
                                     {0, NULL}};

  if (depth > 2)
    return FALSE;
  g_fprintf (stderr, "%08x/%08"G_GSIZE_MODIFIER"x Resource %s\n",
                                                    index, info->size, name);
  ird = (ImageResourceDirectory *)(info->image + index);
  irde = (ImageResourceDirectoryEntry *)(ird + 1);
  if (index < 0 || info->size <= index + sizeof (ImageResourceDirectory))
    return FALSE;
  count = ird_get_number_of_named_entries (ird)
                                        + ird_get_number_of_id_entries (ird);
  if (info->size <= index + sizeof (ImageResourceDirectory)
                                + count * sizeof (ImageResourceDirectoryEntry))
    return FALSE;
  info->separate[index] = g_strdup ("Resource Directory Table");
  dw32_data_make_struct (info, index, dataird, -1);
  address = pe_ioh_get_data_directory_virtual_address (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE);
  for (i = 0; i < count; i++)
    {
      gchar *new_name;
      gint offset;
      Dw32Data datairde[] = {{4, NULL},
                             {4, NULL},
                             {0, NULL}};

      if (irde_get_id (irde) & 0x80000000)
        {
          offset = address + (irde_get_id (irde) & 0x7fffffff);
          dw32_data_make_word (info, offset, 2, -1);
          dw32_data_make_stringw (info, offset + 2, -1);
          if (0 <= offset && offset + sizeof (guint16) <= info->size)
            {
              gchar *utf8str, *tmp;
              ImageResourceDirStringU *irdsu;

              irdsu = (ImageResourceDirStringU *)(info->image + offset);
              utf8str = g_utf16_to_utf8 (irdsu_get_name_string (irdsu),
                                            MIN (irdsu_get_length (irdsu),
                                            (info->size - (offset + 2)) / 2),
                                            NULL, NULL, NULL);
              tmp = g_ascii_strup (utf8str, -1);
              g_free (utf8str);
              new_name = g_strdup_printf ("%s/%s", name, tmp);
              g_free (tmp);
            }
          else
            {
              new_name = g_strdup_printf ("%s/", name);
            }
          datairde[0].name = "Name RVA";
        }
      else
        {
          new_name = g_strdup_printf ("%s/0x%x", name, irde_get_id (irde));
          datairde[0].name = "Integer ID";
        }
      if (irde_get_offset_to_data (irde) & 0x80000000)
        {
          if (!dw32_header_resource_directory (info,
                    address + (irde_get_offset_to_data (irde) & 0x7fffffff),
                                                        new_name, depth + 1))
            return FALSE;
          datairde[1].name = "Subdirectory RVA";
        }
      else
        {
          offset = address + irde_get_offset_to_data (irde);
          if (0 <= offset
                    && offset <= info->size - sizeof (ImageResourceDataEntry))
            {
              ImageResourceDataEntry *irda;
              const static Dw32Data d[] = {{4, "Data RVA"},
                                           {4, "Size"},
                                           {4, "Codepage"},
                                           {4, "Reserved"},
                                           {0, NULL}};

              info->separate[offset] = g_strdup ("Resource Data Entry");
              dw32_data_make_struct (info, offset, d, -1);
              irda = (ImageResourceDataEntry *)(info->image + offset);
              offset = irda_get_offset_to_data (irda);
              if (0 <= offset && offset < info->size)
                info->separate[offset] = g_strdup_printf ("Resource (%s) %08x",
                                            new_name, irda_get_size (irda));
              dw32_data_make_byte (info, offset, irda_get_size (irda), -1);
            }
          datairde[1].name = "Data Entry RVA";
        }
      g_free (new_name);
      offset = index + sizeof (ImageResourceDirectory)
                                    + i * sizeof (ImageResourceDirectoryEntry);
      info->separate[offset] = g_strdup ("Resource Directory Entry");
      dw32_data_make_struct (info, offset, datairde, -1);
      irde++;
    }
  return TRUE;
}


/*  ja:ヘッダ解析
    info,イメージ情報構造体
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
dw32_header_parse (Dw32Info *info)
{
  gint i, sections;
  const ImageSectionHeader *ish;

  if (!info)
    return FALSE;
  /* ja:エントリーポイント */
  if (pe_ioh_get_address_of_entry_point (info->image) != 0)
    {
      Dw32Label *label;

      label = g_malloc (sizeof (Dw32Label));
      label->name = g_strdup ("Entry Point");
      label->index = pe_ioh_get_address_of_entry_point (info->image);
      label->type = DW32_TYPE_ENTRY;
      label->bit = 0;
      label->size = 0;
      info->analyse = g_list_append (info->analyse, label);
    }
  /* ja:セクション */
  sections = pe_ifh_get_number_of_sections (info->image);
  ish = pe_image_section_header (info->image);
  for (i = 0; i < sections; i++)
    {
      gint j;

      for (j = 0; j < ish_get_virtual_size (ish); j++)
        info->status[j + ish_get_virtual_address (ish)] = DW32_STAT_UNKNOWN;
      ish++;
    }
  /* ja:エクスポートテーブル */
  if (pe_ioh_get_data_directory_size (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
    {
      guint32 address, size;
      const ImageExportDirectory *ied;

      address = pe_ioh_get_data_directory_virtual_address (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      size = pe_ioh_get_data_directory_size (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
      ied = (const ImageExportDirectory *)(info->image + address);
      if (address + size <= info->size && size >= sizeof (ImageExportDirectory)
        && ied_get_address_of_functions (ied)
        + ied_get_number_of_functions (ied) * sizeof (guint32) <= info->size
        && ied_get_address_of_names (ied)
        + ied_get_number_of_names (ied) * sizeof (guint32) <= info->size
        && ied_get_address_of_name_ordinals (ied)
        + ied_get_number_of_names (ied) * sizeof (guint16) <= info->size)
        {
          gint i;
          const guint16 *ordinals;
          const guint32 *functions, *names;
          const static Dw32Data dataied[] = {{4, "Export Flag"},
                                             {4, "Time/Date Stamp"},
                                             {2, "Major Version"},
                                             {2, "Minor Version"},
                                             {4, "Name RVA"},
                                             {4, "Ordinal Base"},
                                             {4, "Address Table Entries"},
                                             {4, "Number of Name Pointers"},
                                             {4, "Export Address Table RVA"},
                                             {4, "Name Pointer RVA"},
                                             {4, "Ordinal Table RVA"},
                                             {0, NULL}};

          info->separate[address] = g_strdup ("Export Table");
          dw32_data_make_struct (info, address, dataied, -1);
          dw32_data_make_dword (info, ied_get_address_of_functions (ied),
                    ied_get_number_of_functions (ied) * sizeof (guint32), -1);
          dw32_data_make_dword (info, ied_get_address_of_names (ied),
                    ied_get_number_of_names (ied) * sizeof (guint32), -1);
          dw32_data_make_word (info, ied_get_address_of_name_ordinals (ied),
                    ied_get_number_of_names (ied) * sizeof (guint16), -1);
          functions = (const guint32 *)(info->image
                                        + ied_get_address_of_functions (ied));
          names = (const guint32 *)
                                (info->image + ied_get_address_of_names (ied));
          ordinals = (const guint16 *)(info->image
                                    + ied_get_address_of_name_ordinals (ied));
          for (i = 0; i < ied_get_number_of_functions (ied); i++)
            {
              guint32 func;

              func = GUINT32_FROM_LE (functions[i]);
              if (0 <= func && func < info->size)
                {
                  if (address <= func && func < address + size)
                    {
                      dw32_data_make_string (info, func, -1);
                    }
                  else
                    {
                      gint j;
                      Dw32Label *label;

                      label = g_malloc (sizeof (Dw32Label));
                      for (j = 0; j < ied_get_number_of_names (ied); j++)
                        if (i == GUINT16_FROM_LE (ordinals[j]))
                          {
                            gint k;
                            guint32 name;

                            name = GUINT32_FROM_LE (names[j]);
                            for (k = name; 0 <= k && k < info->size; k++)
                              if (info->image[k] == '\0')
                                {
                                  label->name = g_strdup_printf ("%s (#%d)",
                                                    info->image + name,
                                                    i + ied_get_base (ied));
                                  break;
                                }
                            break;
                          }
                      if (!label->name)
                        label->name = g_strdup_printf ("(#%d)",
                                                    i + ied_get_base (ied));
                      label->index = func;
                      label->type = DW32_TYPE_EXPORT;
                      label->bit = 0;
                      label->size = 0;
                      info->analyse = g_list_append (info->analyse, label);
                    }
                }
            }
          for (i = 0; i < ied_get_number_of_names (ied); i++)
            dw32_data_make_string (info, GUINT32_FROM_LE (names[i]), -1);
        }
      for (i = 0; i < info->size; i++)
        if (info->status[i] == DW32_STAT_DATA)
          info->status[i] = DW32_STAT_EXPORT;
    }
  /* ja:インポートテーブル */
  if (pe_ioh_get_data_directory_size (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT))
    {
      guint16 sections;
      guint32 address;
      const static Dw32Data dataiid[] = {{4, "Import Lookup Table RVA"},
                                         {4, "Time/Date Stamp"},
                                         {4, "Fowarder Chain"},
                                         {4, "Name RVA"},
                                         {4, "Import Address Table RVA"},
                                         {0, NULL}};

      sections = pe_ifh_get_number_of_sections (info->image);
      for (address = pe_ioh_get_data_directory_virtual_address (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT);
                        address + sizeof (ImageImportDescriptor) <= info->size;
                                    address += sizeof (ImageImportDescriptor))
        {
          const gchar *file;
          guint32 lookup, thunk;
          gsize size = 4;
          const ImageSectionHeader *ish;
          const ImageImportDescriptor *iid;

          iid = (const ImageImportDescriptor *)(info->image + address);
          for (i = iid_get_name (iid); 0 < i && i < info->size; i++)
            if (info->image[i] == '\0')
              break;
          if (i <= 0 || info->size <= i)
            break;
          file = (const gchar *)(info->image + iid_get_name (iid));
          info->separate[address] = g_strdup_printf ("Import Table (%s)",
                                                                        file);
          dw32_data_make_struct (info, address, dataiid, -1);
          ish = pe_image_section_header (info->image);
          for (i = 0; i < sections; i++)
            {
              if (ish_get_virtual_address (ish)
                                        <= iid_get_original_first_thunk (iid)
                    && iid_get_original_first_thunk (iid)
                                                < ish_get_virtual_address (ish)
                                                + ish_get_virtual_size (ish))
                break;
              ish++;
            }
          lookup = i < sections ? iid_get_original_first_thunk (iid)
                                : iid_get_first_thunk (iid);
          thunk = iid_get_first_thunk (iid);
          while (lookup + sizeof (guint32) <= info->size
                                    && thunk + sizeof (guint32) <= info->size)
            {
              Dw32Label *label;
              const ImageThunkData *ilt, *iat;

              ilt = (const ImageThunkData *)(info->image + lookup);
              iat = (const ImageThunkData *)(info->image + thunk);
              if (!itd_get_address_of_data (ilt)
                                            || !itd_get_address_of_data (iat))
                break;
              label = g_malloc (sizeof (Dw32Label));
              if (itd_get_ordinal (ilt) & PEIMAGE_ORDINAL_FLAG)
                {
                  /* ja:オーディナル値 */
                  const gchar *name;
                  gint ordinal;

                  ordinal = itd_get_ordinal (ilt) & ~PEIMAGE_ORDINAL_FLAG;
                  name = ordinal_lookup_name (file, ordinal);
                  if (name)
                    label->name = g_strdup (name);
                  else
                    label->name = g_strdup_printf ("%s#%d", file, ordinal);
                }
              else
                {
                  /* ja:名前 */
                  gint index;

                  index = itd_get_address_of_data (ilt);
                  dw32_data_make_word (info, index, 2, -1);
                  dw32_data_make_string (info, index + 2, -1);
                  label->name = NULL;
                  for (i = index + 2; 0 <= i && i < info->size; i++)
                    if (info->image[i] == '\0')
                      {
                        label->name = g_strdup ((const gchar *)
                                                    (info->image + index + 2));
                        break;
                      }
                  index = itd_get_address_of_data (iat);
                  dw32_data_make_word (info, index, 2, -1);
                  dw32_data_make_string (info, index + 2, -1);
                 }
              label->index = thunk;
              label->type = DW32_TYPE_IMPORT;
              label->bit = 32;
              label->size = 4;
              info->import = g_list_append (info->import, label);
              lookup += sizeof (guint32);
              thunk += sizeof (guint32);
              size += 4;
            }
          dw32_data_make_string (info, iid_get_name (iid), -1);
          dw32_data_make_dword (info,
                                iid_get_original_first_thunk (iid), size, -1);
          dw32_data_make_dword (info, iid_get_first_thunk (iid), size, -1);
        }
      if (address < info->size)
        info->separate[address] = g_strdup ("Import Table (NULL)");
      dw32_data_make_struct (info, address, dataiid, -1);
      for (i = 0; i < info->size; i++)
        if (info->status[i] == DW32_STAT_DATA)
          info->status[i] = DW32_STAT_IMPORT;
    }
  /* ja:リソーステーブル */
  g_fprintf (stderr, "Resource Analyse\n");
  if (pe_ioh_get_data_directory_size (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_RESOURCE))
    {
      dw32_header_resource_directory (info,
                        pe_ioh_get_data_directory_virtual_address (info->image,
                                    PEIMAGE_DIRECTORY_ENTRY_RESOURCE), "", 0);
      for (i = 0; i < info->size; i++)
        if (info->status[i] == DW32_STAT_DATA)
          info->status[i] = DW32_STAT_RESOURCE;
    }
  /* ja:再配置 */
  if (pe_ioh_get_data_directory_size (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC))
    {
      gint index;
      guint32 address, size;

      address = pe_ioh_get_data_directory_virtual_address (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC);
      size = pe_ioh_get_data_directory_size (info->image,
                                            PEIMAGE_DIRECTORY_ENTRY_BASERELOC);
      index = address;
      while (index < address + size)
        {
          const ImageBaseRelocation *ibr;
          const static Dw32Data dataibr[] = {{4, "Page RVA"},
                                             {4, "Block Size"},
                                             {0, NULL}};

          ibr = (const ImageBaseRelocation *)(info->image + index);
          if (index < 0 || info->size <= index + sizeof (ImageBaseRelocation)
                                        || ibr_get_size_of_block (ibr) <= 0)
            break;
          info->separate[index] = g_strdup ("Relocation Table");
          dw32_data_make_struct (info, index, dataibr, -1);
          dw32_data_make_word (info, index + sizeof (guint32) * 2,
                    ibr_get_size_of_block (ibr) - sizeof (guint32) * 2, -1);
          index += ibr_get_size_of_block (ibr);
        }
      for (i = 0; i < info->size; i++)
        if (info->status[i] == DW32_STAT_DATA)
          info->status[i] = DW32_STAT_RELOC;
    }
  return TRUE;
}
