/*
    dumpw32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "misc/peimage.h"


/*  ja:ヘルプを表示する
      fp,ファイルポインタ
    file,ファイル名                                                         */
static void
help (FILE        *fp,
      const gchar *file)
{
  g_fprintf (fp,
"dumpw32 "VERSION" ("BUILD_ENVIRONMENT")\n"
"copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org\n"
"\n"
"Usage: %s command... exefile\n"
"\n"
"   h    headers          display header\n"
"   e    exports          display export table\n"
"   i    imports          display import table\n"
"   r    resource         display resource directory\n"
"   l    reloc            display relocation table\n"
"\n", file);
}


#define DP32_MODE_HEADER    1
#define DP32_MODE_EXPORT    2
#define DP32_MODE_IMPORT    4
#define DP32_MODE_RESOURCE  8
#define DP32_MODE_RELOC    16


int
main (int   argc,
      char *argv[])
{
  gint i;
  guint mode = 0;
  guint8 *image;

  /* ja:初期化 */
  gtk_set_locale ();
  gtk_init (&argc, &argv);

  for (i = 1; i < argc - 1; i++)
    if (g_ascii_strcasecmp (argv[i], "header") == 0
                            || g_ascii_strcasecmp (argv[i], "headers") == 0)
      {
        mode |= DP32_MODE_HEADER;
      }
    else if (g_ascii_strcasecmp (argv[i], "export") == 0
                            || g_ascii_strcasecmp (argv[i], "exports") == 0)
      {
        mode |= DP32_MODE_EXPORT;
      }
    else if (g_ascii_strcasecmp (argv[i], "import") == 0
                            || g_ascii_strcasecmp (argv[i], "imports") == 0)
      {
        mode |= DP32_MODE_IMPORT;
      }
    else if (g_ascii_strcasecmp (argv[i], "resource") == 0
                            || g_ascii_strcasecmp (argv[i], "resources") == 0)
      {
        mode |= DP32_MODE_RESOURCE;
      }
    else if (g_ascii_strcasecmp (argv[i], "reloc") == 0)
      {
        mode |= DP32_MODE_RELOC;
      }
    else if (g_ascii_strcasecmp (argv[i], "-h") == 0
                                || g_ascii_strcasecmp (argv[i], "-?") == 0
                                || g_ascii_strcasecmp (argv[i], "--help") == 0)
      {
        help (stdout, argv[0]);
        return 0;
      }
    else
      {
        gchar *p;

        for (p = argv[i]; *p != '\0'; p++)
          switch (*p)
            {
              case 'h': mode |= DP32_MODE_HEADER; break;
              case 'e': mode |= DP32_MODE_EXPORT; break;
              case 'i': mode |= DP32_MODE_IMPORT; break;
              case 'r': mode |= DP32_MODE_RESOURCE; break;
              case 'l': mode |= DP32_MODE_RELOC; break;
              default : help (stderr, argv[0]); return -1;
            }
      }
  if (mode == 0)
    {
      help (stderr, argv[0]);
      return -1;
    }
  /* ja:情報の表示 */
  image = peimage_file_load (argv[argc - 1]);
  if (!image)
    {
      g_fprintf (stderr, "\'%s\' load error.\n", argv[argc - 1]);
      return -1;
    }
  if (mode & DP32_MODE_HEADER)
    {
      gint sections;
      ImageFileHeader *ifh;
      ImageOptionalHeader *ioh;
      ImageSectionHeader *ish;

      ifh = pe_image_file_header (image);
      ioh = pe_image_optional_header (image);
      ish = pe_image_section_header (image);
      g_printf (
"File Header\n"
"Machine                         %04X\n"
"Number of Sections              %04X\n"
"Time Date Stamp                 %08X\n"
"File Pointer to Symbol Table    %08X\n"
"Number of Symbols               %08X\n"
"Size of Option Header           %04X\n"
"Characteristics                 %04X\n"
"\n"
"Optional Header\n"
"Magic                           %04X\n"
"Major Linker Version            %02X\n"
"Minor Linker Version            %02X\n"
"Size of Code                    %08X\n"
"Size of Initialized Data        %08X\n"
"Size of Uninitialized Data      %08X\n"
"Address of Entry Point          %08X\n"
"Base of Code                    %08X\n"
"Base of Data                    %08X\n"
"Image Base                      %08X\n"
"Section Alignment               %08X\n"
"File Alignment                  %08X\n"
"Major Operationg System Version %04X\n"
"Minor Operationg System Version %04X\n"
"Major Image Version             %04X\n"
"Minor Image Version             %04X\n"
"Major Subsystem Version         %04X\n"
"Minor Subsystem Version         %04X\n"
"Win32 Version                   %08X\n"
"Size of Image                   %08X\n"
"Size of Headers                 %08X\n"
"Check Sum                       %08X\n"
"Subsystem                       %04X\n"
"Dll Characteristics             %04X\n"
"Size of Stack Reserve           %08X\n"
"Size of Stack Commit            %08X\n"
"Size of Heap Reserve            %08X\n"
"Size of Heap Commit             %08X\n"
"Loader Flags                    %08X\n"
"Number of RVA and Sizes         %08X\n"
"Export Table RVA                %08X\n"
"Export Table Size               %08X\n"
"Import Table RVA                %08X\n"
"Import Table Size               %08X\n"
"Resource Table RVA              %08X\n"
"Resource Table Size             %08X\n"
"Exception Table RVA             %08X\n"
"Exception Table Size            %08X\n"
"Certificate Table RVA           %08X\n"
"Certificate Table Size          %08X\n"
"Base Relocation Table RVA       %08X\n"
"Base Relocation Table Size      %08X\n"
"Debug RVA                       %08X\n"
"Debug Size                      %08X\n"
"Architecture RVA                %08X\n"
"Architecture Size               %08X\n"
"Global Ptr RVA                  %08X\n"
"Global Ptr Size                 %08X\n"
"TLS Table RVA                   %08X\n"
"TLS Table Size                  %08X\n"
"Load Config Table RVA           %08X\n"
"Load Config Table Size          %08X\n"
"Bound Import RVA                %08X\n"
"Bound Import Size               %08X\n"
"IAT RVA                         %08X\n"
"IAT Size                        %08X\n"
"Delay Import Descriptor RVA     %08X\n"
"Delay Import Descriptor Size    %08X\n",
                            ifh_get_machine (ifh),
                            ifh_get_number_of_sections (ifh),
                            ifh_get_time_date_stamp (ifh),
                            ifh_get_pointer_to_symbol_table (ifh),
                            ifh_get_number_of_symbols (ifh),
                            ifh_get_size_of_optional_header (ifh),
                            ifh_get_characteristics (ifh),
                            ioh_get_magic (ioh),
                            ioh_get_major_linker_version (ioh),
                            ioh_get_minor_linker_version (ioh),
                            ioh_get_size_of_code (ioh),
                            ioh_get_size_of_initialized_data (ioh),
                            ioh_get_size_of_uninitialized_data (ioh),
                            ioh_get_address_of_entry_point (ioh),
                            ioh_get_base_of_code (ioh),
                            ioh_get_base_of_data (ioh),
                            ioh_get_image_base (ioh),
                            ioh_get_section_alignment (ioh),
                            ioh_get_file_alignment (ioh),
                            ioh_get_major_operating_system_version (ioh),
                            ioh_get_minor_operating_system_version (ioh),
                            ioh_get_major_image_version (ioh),
                            ioh_get_minor_image_version (ioh),
                            ioh_get_major_subsystem_version (ioh),
                            ioh_get_minor_subsystem_version (ioh),
                            ioh_get_win32_version_value (ioh),
                            ioh_get_size_of_image (ioh),
                            ioh_get_size_of_headers (ioh),
                            ioh_get_check_sum (ioh),
                            ioh_get_subsystem (ioh),
                            ioh_get_dll_characteristics (ioh),
                            ioh_get_size_of_stack_reserve (ioh),
                            ioh_get_size_of_stack_commit (ioh),
                            ioh_get_size_of_heap_reserve (ioh),
                            ioh_get_size_of_heap_commit (ioh),
                            ioh_get_loader_flags (ioh),
                            ioh_get_number_of_rva_and_sizes (ioh),
                            ioh_get_data_directory_virtual_address (ioh, 0),
                            ioh_get_data_directory_size (ioh, 0),
                            ioh_get_data_directory_virtual_address (ioh, 1),
                            ioh_get_data_directory_size (ioh, 1),
                            ioh_get_data_directory_virtual_address (ioh, 2),
                            ioh_get_data_directory_size (ioh, 2),
                            ioh_get_data_directory_virtual_address (ioh, 3),
                            ioh_get_data_directory_size (ioh, 3),
                            ioh_get_data_directory_virtual_address (ioh, 4),
                            ioh_get_data_directory_size (ioh, 4),
                            ioh_get_data_directory_virtual_address (ioh, 5),
                            ioh_get_data_directory_size (ioh, 5),
                            ioh_get_data_directory_virtual_address (ioh, 6),
                            ioh_get_data_directory_size (ioh, 6),
                            ioh_get_data_directory_virtual_address (ioh, 7),
                            ioh_get_data_directory_size (ioh, 7),
                            ioh_get_data_directory_virtual_address (ioh, 8),
                            ioh_get_data_directory_size (ioh, 8),
                            ioh_get_data_directory_virtual_address (ioh, 9),
                            ioh_get_data_directory_size (ioh, 9),
                            ioh_get_data_directory_virtual_address (ioh, 10),
                            ioh_get_data_directory_size (ioh, 10),
                            ioh_get_data_directory_virtual_address (ioh, 11),
                            ioh_get_data_directory_size (ioh, 11),
                            ioh_get_data_directory_virtual_address (ioh, 12),
                            ioh_get_data_directory_size (ioh, 12),
                            ioh_get_data_directory_virtual_address (ioh, 13),
                            ioh_get_data_directory_size (ioh, 13));
      sections = ifh_get_number_of_sections (ifh);
      for (i = 0; i < sections; i++)
        {
          gchar *name;
          gint j;

          g_printf (
"\n"
"Section Header #%d\n"
"Name                            ", i);
          name = ish_get_name (ish);
          for (j = 0; j < 8; j++)
            if (name[j] != '\0')
              break;
          if (j < 8)
            for (j = 0; j < 8; j++)
              if (name[j] == '\0')
                {
                  gint k;

                  for (k = j + 1; k < 8; k++)
                    if (name[k] != '\0')
                      break;
                  if (k >= 8)
                    break;
                  g_printf ("\\0");
                }
              else if (name[j] == ' ')
                {
                  gint k;

                  for (k = 0; k < j; k++)
                    if (name[k] != ' ')
                      break;
                  if (k < j)
                    {
                      for (k = j + 1; k < 8; k++)
                        if (name[k] != ' ')
                          break;
                      while (k < 8)
                        {
                          if (name[k] != '\0')
                            break;
                          k++;
                        }
                    }
                  else
                    {
                      k = 8;
                    }
                  if (k < 8)
                    g_printf (" ");
                  else
                    g_printf ("\\s");
                }
              else if (name[j] == '\a')
                g_printf ("\\a");
              else if (name[j] == '\b')
                g_printf ("\\b");
              else if (name[j] == '\t')
                g_printf ("\\t");
              else if (name[j] == '\n')
                g_printf ("\\n");
              else if (name[j] == '\v')
                g_printf ("\\v");
              else if (name[j] == '\r')
                g_printf ("\\r");
              else if (name[j] == '\\')
                g_printf ("\\\\");
              else if (g_ascii_isprint (name[j]))
                g_printf ("%c", name[j]);
              else
                g_printf ("\\x%02x", (guchar)name[j]);
          else
            g_printf ("\\0");
          g_printf ("\n");
          g_printf (
"Virtual Size                    %08X\n"
"Virtual Address                 %08X\n"
"Size of Raw Data                %08X\n"
"Pointer to Raw Data             %08X\n"
"Pointer to Relocations          %08X\n"
"Pointer to Linenumbers          %08X\n"
"Number of Relocations           %04X\n"
"Number of Linenumbers           %04X\n"
"Characteristics                 %08X\n",
                            ish_get_virtual_size (ish),
                            ish_get_virtual_address (ish),
                            ish_get_size_of_raw_data (ish),
                            ish_get_pointer_to_raw_data (ish),
                            ish_get_pointer_to_relocations (ish),
                            ish_get_pointer_to_linenumbers (ish),
                            ish_get_number_of_relocations (ish),
                            ish_get_number_of_linenumbers (ish),
                            ish_get_characteristics (ish));
          ish++;
        }
    }
  if (mode & DP32_MODE_EXPORT)
    {
      const gchar *fmt;
      guint32 ordinal = 0;
      GList *gl, *glist;

      glist = peimage_file_enum_export (image);
      for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
        {
          PeExport *peexp;

          peexp = gl->data;
          if (ordinal < peexp->ordinal)
            ordinal = peexp->ordinal;
        }
      if (ordinal < 10)
        fmt = "%08X %d";
      else if (ordinal < 100)
        fmt = "%08X %2d";
      else if (ordinal < 1000)
        fmt = "%08X %3d";
      else if (ordinal < 10000)
        fmt = "%08X %4d";
      else
        fmt = "%08X %5d";
      while ((glist = g_list_first (glist)))
        {
          PeExport *peexp;

          peexp = glist->data;
          g_printf (fmt, peexp->address, peexp->ordinal);
          if (peexp->name)
            g_printf (" %s", peexp->name);
          if (peexp->forward)
            g_printf (" %s", peexp->forward);
          g_printf ("\n");
          g_free (peexp);
          glist = g_list_delete_link (glist, glist);
        }
    }
  if (mode & DP32_MODE_IMPORT)
    {
      gchar *fmt;
      const gchar *file = NULL;
      gsize length = 0;
      GList *gl, *glist;

      glist = peimage_file_enum_import (image);
      for (gl = g_list_first (glist); gl; gl = g_list_next (gl))
        {
          gsize leng;
          PeImport *peimp;

          peimp = gl->data;
          leng = g_strlen (peimp->file);
          if (length < leng)
            length = leng;
        }
      fmt = g_strdup_printf ("%%-%ds ", length);
      while ((glist = g_list_first (glist)))
        {
          PeImport *peimp;

          peimp = glist->data;
          g_printf (fmt, file != peimp->file ? peimp->file : "");
          if (peimp->name)
            g_printf ("%s\n", peimp->name);
          else
            g_printf ("#%d\n", peimp->ordinal);
          file = peimp->file;
          g_free (peimp);
          glist = g_list_delete_link (glist, glist);
        }
      g_free (fmt);
    }
  if (mode & DP32_MODE_RESOURCE)
    {
      GList *glist;

      glist = peimage_file_enum_resource (image);
      if (glist)
        {
          g_printf ("Address  Size     CodePage Key\n");
          while ((glist = g_list_first (glist)))
            {
              PeResource *peres;

              peres = glist->data;
              g_printf ("%08X %08X %08X %s\n",
                    peres->address, peres->size, peres->codepage, peres->key);
              g_free (peres->key);
              g_free (peres);
              glist = g_list_delete_link (glist, glist);
            }
        }
    }
  if (mode & DP32_MODE_RELOC)
    {
      GList *glist;

      glist = peimage_file_enum_relocate (image);
      while ((glist = g_list_first (glist)))
        {
          g_printf ("%08X\n", GPOINTER_TO_UINT (glist->data));
          glist = g_list_delete_link (glist, glist);
        }
    }
  g_free (image);
  return 0;
}
