/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __CPU_H__
#define __CPU_H__


#include "ia32.h"


G_BEGIN_DECLS


/******************************************************************************
*                                                                             *
* ja:CPU構造体                                                                *
*                                                                             *
******************************************************************************/
#define IA32_CPU_DESC_SIZE 8


typedef struct _Ia32CpuDesc { guint8 dummy[IA32_CPU_DESC_SIZE]; } Ia32CpuDesc;


/******************************************************************************
*                                                                             *
* ja:CPU構造体マクロ                                                          *
*                                                                             *
******************************************************************************/
#define IA32_CPU_DESC_TYPE_READONLY  0
#define IA32_CPU_DESC_TYPE_READWRITE 1
#define IA32_CPU_DESC_TYPE_READSTACK 2
#define IA32_CPU_DESC_TYPE_STACK     3
#define IA32_CPU_DESC_TYPE_EXECONLY  4
#define IA32_CPU_DESC_TYPE_EXECREAD  5
#define IA32_CPU_DESC_TYPE_EXECCNFM  6
#define IA32_CPU_DESC_TYPE_EXEC      7


#define ia32_cpu_desc_get_limit(desc) ((((guint8 *)(desc))[0]+(((guint8 *)(desc))[1]<<8)+(((guint8 *)(desc))[6]<<16))&0xfff)
#define ia32_cpu_desc_get_base(desc) (((guint8 *)(desc))[2]+(((guint8 *)(desc))[3]<<8)+(((guint8 *)(desc))[4]<<16)+(((guint8 *)(desc))[7]<<24))
#define ia32_cpu_desc_get_a(desc) (((guint8 *)(desc))[5]&1)
#define ia32_cpu_desc_get_type(desc) ((((guint8 *)(desc))[5]>>1)&7)
#define ia32_cpu_desc_get_s(desc) ((((guint8 *)(desc))[5]>>4)&1)
#define ia32_cpu_desc_get_dpl(desc) ((((guint8 *)(desc))[5]>>5)&3)
#define ia32_cpu_desc_get_p(desc) (((guint8 *)(desc))[5]>>7)
#define ia32_cpu_desc_get_avl(desc) ((((guint8 *)(desc))[6]>>4)&1)
#define ia32_cpu_desc_get_db(desc) ((((guint8 *)(desc))[6]>>6)&1)
#define ia32_cpu_desc_get_g(desc) (((guint8 *)(desc))[6]>>7)
#define ia32_cpu_desc_set_limit(desc,limit) (((guint8 *)(desc))[0]=(guint8)(limit),((guint8 *)(desc))[1]=(limit)>>8,((guint8 *)(desc))[6]=(((limit)>>16)&0x0f)|(((guint8 *)(desc))[6]&0xf0),(limit))
#define ia32_cpu_desc_set_base(desc,base) (((guint8 *)(desc))[2]=(base),((guint8 *)(desc))[3]=(base)>>8,((guint8 *)(desc))[4]=(base)>>16,((guint8 *)(desc))[7]=(base)>>24,(base))
#define ia32_cpu_desc_set_a(desc,a) (((guint8 *)(desc))[5]=((a)&1)|(((guint8 *)(desc))[5]&0xfe),(a))
#define ia32_cpu_desc_set_type(desc,type) (((guint8 *)(desc))[5]=(((type)<<1)&0x0e)|(((guint8 *)(desc))[5]&0xf1),(type))
#define ia32_cpu_desc_set_s(desc,s) (((guint8 *)(desc))[5]=(((s)<<4)&0x10)|(((guint8 *)(desc))[5]&0xef),(s))
#define ia32_cpu_desc_set_dpl(desc,dpl) (((guint8 *)(desc))[5]=(((dpl)<<5)&0x60)|(((guint8 *)(desc))[5]&0x9f),(dpl))
#define ia32_cpu_desc_set_p(desc,p) (((guint8 *)(desc))[5]=((p)<<7)|(((guint8 *)(desc))[5]&0x7f),(p))
#define ia32_cpu_desc_set_avl(desc,avl) (((guint8 *)(desc))[6]=(((avl)<<4)&0x10)|(((guint8 *)(desc))[6]&0xef),(avl))
#define ia32_cpu_desc_set_db(desc,db) (((guint8 *)(desc))[6]=(((db)<<6)&0x40)|(((guint8 *)(desc))[6]&0xbf),(db))
#define ia32_cpu_desc_set_g(desc,g) (((guint8 *)(desc))[6]=((g)<<7)|(((guint8 *)(desc))[6]&0x7f),(g))


#define ia32_cpu_seg_get_rpl(seg) ((seg)&3)
#define ia32_cpu_seg_get_ti(seg) (((seg)>>2)&1)
#define ia32_cpu_seg_get_index(seg) (((seg)>>3)&0x1fff)
#define ia32_cpu_seg_set_rpl(seg,rpl) ((seg)=((rpl)&1)|((seg)&0xfffe))
#define ia32_cpu_seg_set_ti(seg,ti) ((seg)=(((ti)<<1)&4)|((seg)&0xfffb))
#define ia32_cpu_seg_set_index(seg,index) ((seg)=(((index)<<3)&0xfff8)|((seg)&7))


/******************************************************************************
*                                                                             *
* ja:CPU関数群                                                                *
*                                                                             *
******************************************************************************/
#define IA32_CPU_PSEG_NONE 0
#define IA32_CPU_PSEG_CS 0x2e
#define IA32_CPU_PSEG_DS 0x3e
#define IA32_CPU_PSEG_ES 0x26
#define IA32_CPU_PSEG_FS 0x64
#define IA32_CPU_PSEG_GS 0x65
#define IA32_CPU_PSEG_SS 0x36


/*  ja:セグメント変換
     thread,スレッド構造体
    address,アドレス
       pseg,セグメントレジスタプリフィックス
        RET,リニアアドレス                                                  */
guint32
ia32_cpu_seg (Ia32Thread    *thread,
              const guint32  address,
              const guint8   pseg);


/*  ja:ステップ実行
    thread,スレッド構造体
       RET,例外(IA32_ERR_NULL:正常終了)                                     */
gint
ia32_cpu_step (Ia32Thread *thread);


/*  ja:命令実行
       process,プロセス構造体
    breakpoint,ブレークポイント
          tail,記録するステップ数(0:随時出力)
         flags,フラグ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
ia32_cpu_run (Ia32Process *process,
              GList       *breakpoint,
              const gint   tail,
              const guint  flags);


/*  ja:プロセス実行
       process,プロセス構造体
    breakpoint,ブレークポイント
         timer,制限時間(秒)
          step,最大実行ステップ数(0:無制限)
          tail,記録するステップ数(0:随時出力)
         flags,フラグ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
ia32_cpu_loop (Ia32Process *process,
               GList       *breakpoint,
               const gint   step,
               const gint   timer,
               const gint   tail,
               const guint  flags);


G_END_DECLS


#endif /* __CPU_H__ */
