/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "kernel32.h"
#include "ia32/message.h"
#include "ia32/module.h"
#include "ia32/stack.h"
#include "misc/peimage.h"


/* CheckRemoteDebuggerPresent ordinal = 80, argument = 8 */
static gboolean kernel32_CheckRemoteDebuggerPresent (Ia32Thread *thread)
{
  guint32 address;

  address = ia32_stack_value (thread, 2);
  thread->reg.eax = ia32_memory_write_dword (thread->process,
                                                        address, FALSE, -1);
  ia32_message_record_api_with_argument (thread,
                                "CheckRemoteDebuggerPresent", IA32_MESSAGE_DEC,
                                "hProcess", IA32_MESSAGE_HEX,
                                "pbDebuggerPresent", IA32_MESSAGE_HEX,
                                NULL);
  return TRUE;
}


/* GetCurrentProcess ordinal = 450, argument = 0 */
static gboolean kernel32_GetCurrentProcess (Ia32Thread *thread)
{
  thread->reg.eax = thread->process->id;
  ia32_message_record_api_with_argument (thread,
                                        "GetCurrentProcess", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GetModuleFileNameA ordinal = 533, argument = 12 */
static gboolean kernel32_GetModuleFileNameA (Ia32Thread *thread)
{
  guint32 address, size;
  gchar *name;
  const gchar *filename;

  address = ia32_stack_value (thread, 1);
  name = ia32_stack_pointer (thread, 2);
  size = ia32_stack_value (thread, 3);
  if (address == 0)
    address = thread->process->address;
  filename = ia32_module_get_filename (thread->process, address);
  if (name && filename)
    {
      gint i;

      for (i = 0; i < size - 1 && filename[i]; i++)
        name[i] = filename[i];
      name[i] = '\0';
      thread->reg.eax = i;
    }
  else
    {
      thread->reg.eax = 0;
    }
  ia32_message_record_api_with_argument (thread,
                                        "GetModuleFileNameA", IA32_MESSAGE_DEC,
                                        "hModule", IA32_MESSAGE_HEX,
                                        "lpFilename", IA32_MESSAGE_STRING,
                                        "nSize", IA32_MESSAGE_DEC,
                                        NULL);
  return TRUE;
}


/* GetModuleFileNameW ordinal = 534, argument = 12 */
static gboolean kernel32_GetModuleFileNameW (Ia32Thread *thread)
{
  guint32 address, size;
  gunichar2 *name;
  const gchar *filename;

  address = ia32_stack_value (thread, 1);
  name = ia32_stack_pointer (thread, 2);
  size = ia32_stack_value (thread, 3);
  if (address == 0)
    address = thread->process->address;
  filename = ia32_module_get_filename (thread->process, address);
  if (name && filename)
    {
      gint i;

      for (i = 0; i < size - 1 && filename[i]; i++)
        name[i] = filename[i];
      name[i] = '\0';
      thread->reg.eax = i;
    }
  else
    {
      thread->reg.eax = 0;
    }
  ia32_message_record_api_with_argument (thread,
                                        "GetModuleFileNameW", IA32_MESSAGE_DEC,
                                        "hModule", IA32_MESSAGE_HEX,
                                        "lpFilename", IA32_MESSAGE_STRINGW,
                                        "nSize", IA32_MESSAGE_DEC,
                                        NULL);
  return TRUE;
}


/* GetModuleHandleA ordinal = 535, argument = 4 */
static gboolean kernel32_GetModuleHandleA (Ia32Thread *thread)
{
  guint32 address;

  address = ia32_module_get_library (thread->process,
                                                ia32_stack_string (thread, 1));
  thread->reg.eax = address != IA32_MEMORY_NULL ? address : 0;
  ia32_message_record_api_with_argument (thread,
                                        "GetModuleHandleA", IA32_MESSAGE_HEX,
                                        "lpFileName", IA32_MESSAGE_STRING,
                                        NULL);
  return TRUE;
}


/* GetModuleHandleW ordinal = 538, argument = 4 */
static gboolean kernel32_GetModuleHandleW (Ia32Thread *thread)
{
  gchar *utf8str;
  guint32 address;

  utf8str = g_utf16_to_utf8 (ia32_stack_stringw (thread, 1), -1,
                                                            NULL, NULL, NULL);
  address = ia32_module_get_library (thread->process, utf8str);
  g_free (utf8str);
  thread->reg.eax = address != IA32_MEMORY_NULL ? address : 0;
  ia32_message_record_api_with_argument (thread,
                                        "GetModuleHandleW", IA32_MESSAGE_HEX,
                                        "lpFileName", IA32_MESSAGE_STRINGW,
                                        NULL);
  return TRUE;
}


/* GetPriorityClass ordinal = 572, argument = 4 */
static gboolean kernel32_GetPriorityClass (Ia32Thread *thread)
{
  thread->reg.eax = 0;
  ia32_message_record_api_with_argument (thread,
                                        "GetPriorityClass", IA32_MESSAGE_HEX,
                                        "hProcess", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GetProcAddress ordinal = 583, argument = 8 */
static gboolean kernel32_GetProcAddress (Ia32Thread *thread)
{
  gconstpointer func;
  guint8 *image;
  guint32 name, retval;

  image = ia32_stack_pointer (thread, 1);
  name = ia32_stack_value (thread, 2);
  if (name < 0x10000)
    func = peimage_file_get_proc (image, name);
  else if (ia32_memory_is_string (thread->process, name))
    func = peimage_file_get_func (image, ia32_stack_string (thread, 2));
  else
    func = NULL;
  retval = ia32_memory_virtual_address (thread->process, func);
  thread->reg.eax = retval != IA32_MEMORY_NULL ? retval : 0;
  ia32_message_record_api_with_argument (thread,
                            "GetProcAddress", IA32_MESSAGE_HEX,
                            "hModule", IA32_MESSAGE_HEX,
                            "lpProcName", name < 0x10000 ? IA32_MESSAGE_DEC
                                                         : IA32_MESSAGE_STRING,
                            NULL);
  return TRUE;
}


/* GetTickCount ordinal = 662, argument = 0 */
static gboolean kernel32_GetTickCount (Ia32Thread *thread)
{
  thread->reg.eax = time (NULL) * 1000;
  ia32_message_record_api_with_argument (thread,
                                        "GetTickCount", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GetVersion ordinal = 676, argument = 0 */
static gboolean kernel32_GetVersion (Ia32Thread *thread)
{
  thread->reg.eax = 0x1db00106;
  ia32_message_record_api_with_argument (thread,
                                        "GetVersion", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GlobalAlloc ordinal = 693, argument = 8 */
static gboolean kernel32_GlobalAlloc (Ia32Thread *thread)
{
  guint32 address;

  address = ia32_memory_alloc (thread->process,
                              IA32_MEMORY_NULL,
                              ia32_stack_value (thread, 2),
                              IA32_MEMORY_ATTR_USER);
  thread->reg.eax = address != IA32_MEMORY_NULL ? address : 0;
  ia32_message_record_api_with_argument (thread,
                                        "GlobalAlloc", IA32_MESSAGE_HEX,
                                        "uFlags", IA32_MESSAGE_HEX,
                                        "dwBytes", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GlobalFree ordinal = 700, argument = 4 */
static gboolean kernel32_GlobalFree (Ia32Thread *thread)
{
  thread->reg.eax = 0;
  ia32_memory_free (thread->process, ia32_stack_value (thread, 1));
  ia32_message_record_api_with_argument (thread,
                                        "GlobalFree", IA32_MESSAGE_HEX,
                                        "hMem", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GlobalHandle ordinal = 703, argument = 4 */
static gboolean kernel32_GlobalHandle (Ia32Thread *thread)
{
  thread->reg.eax = ia32_stack_value (thread, 1);
  ia32_message_record_api_with_argument (thread,
                                        "GlobalHandle", IA32_MESSAGE_HEX,
                                        "pMem", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GlobalLock ordinal = 704, argument = 4 */
static gboolean kernel32_GlobalLock (Ia32Thread *thread)
{
  thread->reg.eax = ia32_stack_value (thread, 1);
  ia32_message_record_api_with_argument (thread,
                                        "GlobalLock", IA32_MESSAGE_HEX,
                                        "hMem", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* GlobalUnlock ordinal = 711, argument = 4 */
static gboolean kernel32_GlobalUnlock (Ia32Thread *thread)
{
  thread->reg.eax = FALSE;
  ia32_message_record_api_with_argument (thread,
                                        "GlobalUnlock", IA32_MESSAGE_DEC,
                                        "hMem", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* IsBadReadPtr ordinal = 760, argument = 8 */
static gboolean kernel32_IsBadReadPtr (Ia32Thread *thread)
{
  thread->reg.eax = FALSE;
  ia32_message_record_api_with_argument (thread,
                                        "IsBadReadPtr", IA32_MESSAGE_DEC,
                                        "lp", IA32_MESSAGE_HEX,
                                        "ucb", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* IsBadWritePtr ordinal = 763, argument = 8 */
static gboolean kernel32_IsBadWritePtr (Ia32Thread *thread)
{
  guint32 s, address, size;

  address = ia32_stack_value (thread, 1);
  size = ia32_stack_value (thread, 2);
  while (size > 0
                && (s = ia32_memory_real_size (thread->process, address)) > 0)
    {
      address += s;
      size = size > s ? size - s : 0;
    }
  thread->reg.eax = size <= 0;
  ia32_message_record_api_with_argument (thread,
                                        "IsBadWritePtr", IA32_MESSAGE_DEC,
                                        "lp", IA32_MESSAGE_HEX,
                                        "ucp", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* IsDebuggerPresent ordinal = 769, argument = 0 */
static gboolean kernel32_IsDebuggerPresent (Ia32Thread *thread)
{
  thread->reg.eax = FALSE;
  ia32_message_record_api_with_argument (thread,
                                        "IsDebuggerPresent", IA32_MESSAGE_DEC,
                                        NULL);
  return TRUE;
}


/* LoadLibraryA ordinal = 829, argument = 4 */
static gboolean kernel32_LoadLibraryA (Ia32Thread *thread)
{
  guint32 address;

  address = ia32_module_load_library (thread->process,
                                                ia32_stack_string (thread, 1));
  thread->reg.eax = address != IA32_MEMORY_NULL ? address : 0;
  ia32_message_record_api_with_argument (thread,
                                        "LoadLibraryA", IA32_MESSAGE_HEX,
                                        "lpFileName", IA32_MESSAGE_STRING,
                                        NULL);
  return TRUE;
}


/* LoadLibraryW ordinal = 832, argument = 4 */
static gboolean kernel32_LoadLibraryW (Ia32Thread *thread)
{
  gchar *utf8str;
  guint32 address;

  utf8str = g_utf16_to_utf8 (ia32_stack_stringw (thread, 1), -1,
                                                            NULL, NULL, NULL);
  address = ia32_module_load_library (thread->process, utf8str);
  g_free (utf8str);
  thread->reg.eax = address != IA32_MEMORY_NULL ? address : 0;
  ia32_message_record_api_with_argument (thread,
                                        "LoadLibraryW", IA32_MESSAGE_HEX,
                                        "lpFileName", IA32_MESSAGE_STRINGW,
                                        NULL);
  return TRUE;
}


/* SetPriorityClass ordinal = 1150, argument = 8 */
static gboolean kernel32_SetPriorityClass (Ia32Thread *thread)
{
  thread->reg.eax = TRUE;
  ia32_message_record_api_with_argument (thread,
                                        "SetPriorityClass", IA32_MESSAGE_DEC,
                                        "hProcess", IA32_MESSAGE_HEX,
                                        "dwPriorityClass", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualAlloc ordinal = 1258, argument = 16 */
static gboolean kernel32_VirtualAlloc (Ia32Thread *thread)
{
  guint32 address;

  address = ia32_stack_value (thread, 1);
  if (address == 0)
    address = IA32_MEMORY_NULL;
  address = ia32_memory_alloc (thread->process,
                              address,
                              ia32_stack_value (thread, 2),
                              IA32_MEMORY_ATTR_USER);
  thread->reg.eax = address != IA32_MEMORY_NULL ? address : 0;
  ia32_message_record_api_with_argument (thread,
                                        "VirtualAlloc", IA32_MESSAGE_HEX,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        "flAllocationType", IA32_MESSAGE_HEX,
                                        "flProtect", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualAllocEx ordinal = 1259, argument = 20 */
static gboolean kernel32_VirtualAllocEx (Ia32Thread *thread)
{
  guint32 address;

  address = ia32_memory_alloc (thread->process,
                              ia32_stack_value (thread, 2),
                              ia32_stack_value (thread, 3),
                              IA32_MEMORY_ATTR_USER);
  thread->reg.eax = address != IA32_MEMORY_NULL ? address : 0;
  ia32_message_record_api_with_argument (thread,
                                        "VirtualAllocEx", IA32_MESSAGE_HEX,
                                        "hProcess", IA32_MESSAGE_HEX,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        "flAllocationType", IA32_MESSAGE_HEX,
                                        "flProtect", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualFree ordinal = 1261, argument = 12 */
static gboolean kernel32_VirtualFree (Ia32Thread *thread)
{
  thread->reg.eax = ia32_memory_free (thread->process,
                                                ia32_stack_value (thread, 1));
  ia32_message_record_api_with_argument (thread,
                                        "VirtualFree", IA32_MESSAGE_DEC,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        "dwFreeType", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualFreeEx ordinal = 1262, argument = 16 */
static gboolean kernel32_VirtualFreeEx (Ia32Thread *thread)
{
  thread->reg.eax = ia32_memory_free (thread->process,
                                                ia32_stack_value (thread, 2));
  ia32_message_record_api_with_argument (thread,
                                        "VirtualFreeEx", IA32_MESSAGE_DEC,
                                        "hProcess", IA32_MESSAGE_HEX,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        "dwFreeType", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualLock ordinal = 1263, argument = 8 */
static gboolean kernel32_VirtualLock (Ia32Thread *thread)
{
  thread->reg.eax = TRUE;
  ia32_message_record_api_with_argument (thread,
                                        "VirtualLock", IA32_MESSAGE_DEC,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualProtect ordinal = 1264, argument = 16 */
static gboolean kernel32_VirtualProtect (Ia32Thread *thread)
{
  thread->reg.eax = TRUE;
  ia32_message_record_api_with_argument (thread,
                                        "VirtualProtect", IA32_MESSAGE_DEC,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        "flNewProtect", IA32_MESSAGE_HEX,
                                        "lpflOldProtect", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualProtectEx ordinal = 1265, argument = 20 */
static gboolean kernel32_VirtualProtectEx (Ia32Thread *thread)
{
  thread->reg.eax = TRUE;
  ia32_message_record_api_with_argument (thread,
                                        "VirtualProtectEx", IA32_MESSAGE_DEC,
                                        "hProcess", IA32_MESSAGE_HEX,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        "flNewProtect", IA32_MESSAGE_HEX,
                                        "lpflOldProtect", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


/* VirtualUnlock ordinal = 1268, argument = 8 */
static gboolean kernel32_VirtualUnlock (Ia32Thread *thread)
{
  thread->reg.eax = TRUE;
  ia32_message_record_api_with_argument (thread,
                                        "VirtualUnlock", IA32_MESSAGE_DEC,
                                        "lpAddress", IA32_MESSAGE_HEX,
                                        "dwSize", IA32_MESSAGE_HEX,
                                        NULL);
  return TRUE;
}


Ia32Export ia32_kernel32_exports[] = {
{ 4,    3, "AcquireSRWLockExclusive",                       NULL},
{ 4,    4, "AcquireSRWLockShared",                          NULL},
{ 8,    5, "ActivateActCtx",                                NULL},
{ 4,    6, "AddAtomA",                                      NULL},
{ 4,    7, "AddAtomW",                                      NULL},
{12,    8, "AddConsoleAliasA",                              NULL},
{12,    9, "AddConsoleAliasW",                              NULL},
{ 0,   10, "AddIntegrityLabelToBoundaryDescriptor",         NULL},
{ 8,   11, "AddLocalAlternateComputerNameA",                NULL},
{ 8,   12, "AddLocalAlternateComputerNameW",                NULL},
{ 4,   13, "AddRefActCtx",                                  NULL},
{ 8,   14, "AddSIDToBoundaryDescriptor",                    NULL},
{ 0,   15, "AddSecureMemoryCacheCallback",                  NULL},
{ 8,   16, "AddVectoredContinueHandler",                    NULL},
{ 8,   17, "AddVectoredExceptionHandler",                   NULL},
{ 0,   18, "AdjustCalendarDate",                            NULL},
{ 0,   19, "AllocConsole",                                  NULL},
{12,   20, "AllocateUserPhysicalPages",                     NULL},
{16,   21, "AllocateUserPhysicalPagesNuma",                 NULL},
{ 4,   22, "ApplicationRecoveryFinished",                   NULL},
{ 4,   23, "ApplicationRecoveryInProgress",                 NULL},
{ 0,   24, "AreFileApisANSI",                               NULL},
{ 8,   25, "AssignProcessToJobObject",                      NULL},
{ 4,   26, "AttachConsole",                                 NULL},
{28,   27, "BackupRead",                                    NULL},
{24,   28, "BackupSeek",                                    NULL},
{28,   29, "BackupWrite",                                   NULL},
{ 0,   30, "BaseCheckAppcompatCache",                       NULL},
{ 0,   31, "BaseCheckAppcompatCacheEx",                     NULL},
{ 0,   32, "BaseCheckRunApp",                               NULL},
{ 0,   33, "BaseCleanupAppcompatCacheSupport",              NULL},
{ 0,   34, "BaseDllReadWriteIniFile",                       NULL},
{ 0,   35, "BaseDumpAppcompatCache",                        NULL},
{ 0,   36, "BaseFlushAppcompatCache",                       NULL},
{ 0,   37, "BaseFormatObjectAttributes",                    NULL},
{ 0,   38, "BaseFormatTimeOut",                             NULL},
{ 0,   39, "BaseGenerateAppCompatData",                     NULL},
{ 0,   40, "BaseGetNamedObjectDirectory",                   NULL},
{ 0,   41, "BaseInitAppcompatCacheSupport",                 NULL},
{ 0,   42, "BaseIsAppcompatInfrastructureDisabled",         NULL},
{ 0,   43, "BaseQueryModuleData",                           NULL},
{ 0,   44, "BaseSetLastNTError",                            NULL},
{ 0,    1, "BaseThreadInitThunk",                           NULL},
{ 0,   45, "BaseUpdateAppcompatCache",                      NULL},
{ 0,   46, "BaseVerifyUnicodeString",                       NULL},
{ 0,   47, "Basep8BitStringToDynamicUnicodeString",         NULL},
{ 0,   48, "BasepAllocateActivationContextActivationBlock", NULL},
{ 0,   49, "BasepAnsiStringToDynamicUnicodeString",         NULL},
{ 0,   50, "BasepCheckAppCompat",                           NULL},
{ 0,   51, "BasepCheckBadapp",                              NULL},
{ 0,   52, "BasepCheckWinSaferRestrictions",                NULL},
{ 0,   53, "BasepFreeActivationContextActivationBlock",     NULL},
{ 0,   54, "BasepFreeAppCompatData",                        NULL},
{ 0,   55, "BasepMapModuleHandle",                          NULL},
{ 8,   56, "Beep",                                          NULL},
{ 8,   57, "BeginUpdateResourceA",                          NULL},
{ 8,   58, "BeginUpdateResourceW",                          NULL},
{12,   59, "BindIoCompletionCallback",                      NULL},
{ 8,   60, "BuildCommDCBA",                                 NULL},
{12,   61, "BuildCommDCBAndTimeoutsA",                      NULL},
{12,   62, "BuildCommDCBAndTimeoutsW",                      NULL},
{ 8,   63, "BuildCommDCBW",                                 NULL},
{28,   64, "CallNamedPipeA",                                NULL},
{28,   65, "CallNamedPipeW",                                NULL},
{ 4,   66, "CallbackMayRunLong",                            NULL},
{ 4,   67, "CancelDeviceWakeupRequest",                     NULL},
{ 4,   68, "CancelIo",                                      NULL},
{ 8,   69, "CancelIoEx",                                    NULL},
{ 4,   70, "CancelSynchronousIo",                           NULL},
{ 4,   71, "CancelThreadpoolIo",                            NULL},
{ 8,   72, "CancelTimerQueueTimer",                         NULL},
{ 4,   73, "CancelWaitableTimer",                           NULL},
{16,   74, "ChangeTimerQueueTimer",                         NULL},
{ 0,   75, "CheckElevation",                                NULL},
{ 0,   76, "CheckElevationEnabled",                         NULL},
{ 0,   77, "CheckForReadOnlyResource",                      NULL},
{20,   78, "CheckNameLegalDOS8Dot3A",                       NULL},
{20,   79, "CheckNameLegalDOS8Dot3W",                       NULL},
{ 8,   80, "CheckRemoteDebuggerPresent",                    kernel32_CheckRemoteDebuggerPresent},
{ 4,   81, "ClearCommBreak",                                NULL},
{12,   82, "ClearCommError",                                NULL},
{ 0,   83, "CloseConsoleHandle",                            NULL},
{ 4,   84, "CloseHandle",                                   NULL},
{ 8,   85, "ClosePrivateNamespace",                         NULL},
{ 0,   86, "CloseProfileUserMapping",                       NULL},
{ 4,   87, "CloseThreadpool",                               NULL},
{ 4,   88, "CloseThreadpoolCleanupGroup",                   NULL},
{12,   89, "CloseThreadpoolCleanupGroupMembers",            NULL},
{ 4,   90, "CloseThreadpoolIo",                             NULL},
{ 4,   91, "CloseThreadpoolTimer",                          NULL},
{ 4,   92, "CloseThreadpoolWait",                           NULL},
{ 4,   93, "CloseThreadpoolWork",                           NULL},
{ 0,   94, "CmdBatNotification",                            NULL},
{12,   95, "CommConfigDialogA",                             NULL},
{12,   96, "CommConfigDialogW",                             NULL},
{ 0,   97, "CompareCalendarDates",                          NULL},
{ 8,   98, "CompareFileTime",                               NULL},
{24,   99, "CompareStringA",                                NULL},
{36,  100, "CompareStringEx",                               NULL},
{20,  101, "CompareStringOrdinal",                          NULL},
{24,  102, "CompareStringW",                                NULL},
{ 8,  103, "ConnectNamedPipe",                              NULL},
{ 0,  104, "ConsoleMenuControl",                            NULL},
{12,  105, "ContinueDebugEvent",                            NULL},
{ 0,  106, "ConvertCalDateTimeToSystemTime",                NULL},
{ 4,  107, "ConvertDefaultLocale",                          NULL},
{ 0,  108, "ConvertFiberToThread",                          NULL},
{ 0,  109, "ConvertNLSDayOfWeekToWin32DayOfWeek",           NULL},
{ 0,  110, "ConvertSystemTimeToCalDateTime",                NULL},
{ 4,  111, "ConvertThreadToFiber",                          NULL},
{ 8,  112, "ConvertThreadToFiberEx",                        NULL},
{ 0,  113, "CopyExtendedContext",                           NULL},
{12,  114, "CopyFileA",                                     NULL},
{24,  115, "CopyFileExA",                                   NULL},
{24,  116, "CopyFileExW",                                   NULL},
{28,  117, "CopyFileTransactedA",                           NULL},
{28,  118, "CopyFileTransactedW",                           NULL},
{12,  119, "CopyFileW",                                     NULL},
{ 0,  120, "CopyLZFile",                                    NULL},
{ 4,  121, "CreateActCtxA",                                 NULL},
{ 4,  122, "CreateActCtxW",                                 NULL},
{ 8,  123, "CreateBoundaryDescriptorA",                     NULL},
{ 8,  124, "CreateBoundaryDescriptorW",                     NULL},
{20,  125, "CreateConsoleScreenBuffer",                     NULL},
{ 8,  126, "CreateDirectoryA",                              NULL},
{12,  127, "CreateDirectoryExA",                            NULL},
{12,  128, "CreateDirectoryExW",                            NULL},
{16,  129, "CreateDirectoryTransactedA",                    NULL},
{16,  130, "CreateDirectoryTransactedW",                    NULL},
{ 8,  131, "CreateDirectoryW",                              NULL},
{16,  132, "CreateEventA",                                  NULL},
{16,  133, "CreateEventExA",                                NULL},
{16,  134, "CreateEventExW",                                NULL},
{16,  135, "CreateEventW",                                  NULL},
{12,  136, "CreateFiber",                                   NULL},
{20,  137, "CreateFiberEx",                                 NULL},
{28,  138, "CreateFileA",                                   NULL},
{24,  139, "CreateFileMappingA",                            NULL},
{28,  140, "CreateFileMappingNumaA",                        NULL},
{28,  141, "CreateFileMappingNumaW",                        NULL},
{24,  142, "CreateFileMappingW",                            NULL},
{40,  143, "CreateFileTransactedA",                         NULL},
{40,  144, "CreateFileTransactedW",                         NULL},
{28,  145, "CreateFileW",                                   NULL},
{12,  146, "CreateHardLinkA",                               NULL},
{16,  147, "CreateHardLinkTransactedA",                     NULL},
{16,  148, "CreateHardLinkTransactedW",                     NULL},
{12,  149, "CreateHardLinkW",                               NULL},
{16,  150, "CreateIoCompletionPort",                        NULL},
{ 8,  151, "CreateJobObjectA",                              NULL},
{ 8,  152, "CreateJobObjectW",                              NULL},
{12,  153, "CreateJobSet",                                  NULL},
{16,  154, "CreateMailslotA",                               NULL},
{16,  155, "CreateMailslotW",                               NULL},
{ 4,  156, "CreateMemoryResourceNotification",              NULL},
{12,  157, "CreateMutexA",                                  NULL},
{16,  158, "CreateMutexExA",                                NULL},
{16,  159, "CreateMutexExW",                                NULL},
{12,  160, "CreateMutexW",                                  NULL},
{32,  161, "CreateNamedPipeA",                              NULL},
{32,  162, "CreateNamedPipeW",                              NULL},
{16,  163, "CreatePipe",                                    NULL},
{12,  164, "CreatePrivateNamespaceA",                       NULL},
{12,  165, "CreatePrivateNamespaceW",                       NULL},
{40,  166, "CreateProcessA",                                NULL},
{ 0,  167, "CreateProcessAsUserW",                          NULL},
{ 0,  168, "CreateProcessInternalA",                        NULL},
{ 0,  169, "CreateProcessInternalW",                        NULL},
{40,  170, "CreateProcessW",                                NULL},
{28,  171, "CreateRemoteThread",                            NULL},
{ 0,  172, "CreateRemoteThreadEx",                          NULL},
{16,  173, "CreateSemaphoreA",                              NULL},
{24,  174, "CreateSemaphoreExA",                            NULL},
{24,  175, "CreateSemaphoreExW",                            NULL},
{16,  176, "CreateSemaphoreW",                              NULL},
{ 0,  177, "CreateSocketHandle",                            NULL},
{12,  178, "CreateSymbolicLinkA",                           NULL},
{16,  179, "CreateSymbolicLinkTransactedA",                 NULL},
{16,  180, "CreateSymbolicLinkTransactedW",                 NULL},
{12,  181, "CreateSymbolicLinkW",                           NULL},
{16,  182, "CreateTapePartition",                           NULL},
{24,  183, "CreateThread",                                  NULL},
{ 4,  184, "CreateThreadpool",                              NULL},
{ 0,  185, "CreateThreadpoolCleanupGroup",                  NULL},
{16,  186, "CreateThreadpoolIo",                            NULL},
{12,  187, "CreateThreadpoolTimer",                         NULL},
{12,  188, "CreateThreadpoolWait",                          NULL},
{12,  189, "CreateThreadpoolWork",                          NULL},
{ 0,  190, "CreateTimerQueue",                              NULL},
{28,  191, "CreateTimerQueueTimer",                         NULL},
{ 8,  192, "CreateToolhelp32Snapshot",                      NULL},
{12,  193, "CreateWaitableTimerA",                          NULL},
{16,  194, "CreateWaitableTimerExA",                        NULL},
{16,  195, "CreateWaitableTimerExW",                        NULL},
{12,  196, "CreateWaitableTimerW",                          NULL},
{ 0,  197, "CtrlRoutine",                                   NULL},
{ 8,  198, "DeactivateActCtx",                              NULL},
{ 4,  199, "DebugActiveProcess",                            NULL},
{ 4,  200, "DebugActiveProcessStop",                        NULL},
{ 0,  201, "DebugBreak",                                    NULL},
{ 4,  202, "DebugBreakProcess",                             NULL},
{ 4,  203, "DebugSetProcessKillOnExit",                     NULL},
{ 4,  204, "DecodePointer",                                 NULL},
{ 4,  205, "DecodeSystemPointer",                           NULL},
{12,  206, "DefineDosDeviceA",                              NULL},
{12,  207, "DefineDosDeviceW",                              NULL},
{ 0,  208, "DelayLoadFailureHook",                          NULL},
{ 4,  209, "DeleteAtom",                                    NULL},
{ 4,  210, "DeleteBoundaryDescriptor",                      NULL},
{ 4,  211, "DeleteCriticalSection",                         NULL},
{ 4,  212, "DeleteFiber",                                   NULL},
{ 4,  213, "DeleteFileA",                                   NULL},
{ 8,  214, "DeleteFileTransactedA",                         NULL},
{ 8,  215, "DeleteFileTransactedW",                         NULL},
{ 4,  216, "DeleteFileW",                                   NULL},
{ 4,  217, "DeleteProcThreadAttributeList",                 NULL},
{ 4,  218, "DeleteTimerQueue",                              NULL},
{ 8,  219, "DeleteTimerQueueEx",                            NULL},
{12,  220, "DeleteTimerQueueTimer",                         NULL},
{ 4,  221, "DeleteVolumeMountPointA",                       NULL},
{ 4,  222, "DeleteVolumeMountPointW",                       NULL},
{32,  223, "DeviceIoControl",                               NULL},
{ 4,  224, "DisableThreadLibraryCalls",                     NULL},
{ 0,  225, "DisableThreadProfiling",                        NULL},
{ 4,  226, "DisassociateCurrentThreadFromCallback",         NULL},
{ 4,  227, "DisconnectNamedPipe",                           NULL},
{12,  228, "DnsHostnameToComputerNameA",                    NULL},
{12,  229, "DnsHostnameToComputerNameW",                    NULL},
{12,  230, "DosDateTimeToFileTime",                         NULL},
{ 0,  231, "DosPathToSessionPathA",                         NULL},
{ 0,  232, "DosPathToSessionPathW",                         NULL},
{ 0,  233, "DuplicateConsoleHandle",                        NULL},
{28,  234, "DuplicateHandle",                               NULL},
{ 0,  235, "EnableThreadProfiling",                         NULL},
{ 4,  236, "EncodePointer",                                 NULL},
{ 4,  237, "EncodeSystemPointer",                           NULL},
{ 8,  238, "EndUpdateResourceA",                            NULL},
{ 8,  239, "EndUpdateResourceW",                            NULL},
{ 4,  240, "EnterCriticalSection",                          NULL},
{16,  241, "EnumCalendarInfoA",                             NULL},
{16,  242, "EnumCalendarInfoExA",                           NULL},
{24,  243, "EnumCalendarInfoExEx",                          NULL},
{16,  244, "EnumCalendarInfoExW",                           NULL},
{16,  245, "EnumCalendarInfoW",                             NULL},
{12,  246, "EnumDateFormatsA",                              NULL},
{12,  247, "EnumDateFormatsExA",                            NULL},
{16,  248, "EnumDateFormatsExEx",                           NULL},
{12,  249, "EnumDateFormatsExW",                            NULL},
{12,  250, "EnumDateFormatsW",                              NULL},
{16,  251, "EnumLanguageGroupLocalesA",                     NULL},
{16,  252, "EnumLanguageGroupLocalesW",                     NULL},
{20,  253, "EnumResourceLanguagesA",                        NULL},
{28,  254, "EnumResourceLanguagesExA",                      NULL},
{28,  255, "EnumResourceLanguagesExW",                      NULL},
{20,  256, "EnumResourceLanguagesW",                        NULL},
{16,  257, "EnumResourceNamesA",                            NULL},
{24,  258, "EnumResourceNamesExA",                          NULL},
{24,  259, "EnumResourceNamesExW",                          NULL},
{16,  260, "EnumResourceNamesW",                            NULL},
{12,  261, "EnumResourceTypesA",                            NULL},
{20,  262, "EnumResourceTypesExA",                          NULL},
{20,  263, "EnumResourceTypesExW",                          NULL},
{12,  264, "EnumResourceTypesW",                            NULL},
{ 8,  265, "EnumSystemCodePagesA",                          NULL},
{ 8,  266, "EnumSystemCodePagesW",                          NULL},
{12,  267, "EnumSystemFirmwareTables",                      NULL},
{12,  268, "EnumSystemGeoID",                               NULL},
{12,  269, "EnumSystemLanguageGroupsA",                     NULL},
{12,  270, "EnumSystemLanguageGroupsW",                     NULL},
{ 8,  271, "EnumSystemLocalesA",                            NULL},
{16,  272, "EnumSystemLocalesEx",                           NULL},
{ 8,  273, "EnumSystemLocalesW",                            NULL},
{12,  274, "EnumTimeFormatsA",                              NULL},
{16,  275, "EnumTimeFormatsEx",                             NULL},
{12,  276, "EnumTimeFormatsW",                              NULL},
{12,  277, "EnumUILanguagesA",                              NULL},
{12,  278, "EnumUILanguagesW",                              NULL},
{16,  279, "EnumerateLocalComputerNamesA",                  NULL},
{16,  280, "EnumerateLocalComputerNamesW",                  NULL},
{12,  281, "EraseTape",                                     NULL},
{ 8,  282, "EscapeCommFunction",                            NULL},
{ 4,  283, "ExitProcess",                                   NULL},
{ 4,  284, "ExitThread",                                    NULL},
{ 0,  285, "ExitVDM",                                       NULL},
{12,  286, "ExpandEnvironmentStringsA",                     NULL},
{12,  287, "ExpandEnvironmentStringsW",                     NULL},
{ 0,  288, "ExpungeConsoleCommandHistoryA",                 NULL},
{ 0,  289, "ExpungeConsoleCommandHistoryW",                 NULL},
{ 8,  290, "FatalAppExitA",                                 NULL},
{ 8,  291, "FatalAppExitW",                                 NULL},
{ 4,  292, "FatalExit",                                     NULL},
{12,  293, "FileTimeToDosDateTime",                         NULL},
{ 8,  294, "FileTimeToLocalFileTime",                       NULL},
{ 8,  295, "FileTimeToSystemTime",                          NULL},
{20,  296, "FillConsoleOutputAttribute",                    NULL},
{20,  297, "FillConsoleOutputCharacterA",                   NULL},
{20,  298, "FillConsoleOutputCharacterW",                   NULL},
{20,  299, "FindActCtxSectionGuid",                         NULL},
{20,  300, "FindActCtxSectionStringA",                      NULL},
{20,  301, "FindActCtxSectionStringW",                      NULL},
{ 4,  302, "FindAtomA",                                     NULL},
{ 4,  303, "FindAtomW",                                     NULL},
{ 4,  304, "FindClose",                                     NULL},
{ 4,  305, "FindCloseChangeNotification",                   NULL},
{12,  306, "FindFirstChangeNotificationA",                  NULL},
{12,  307, "FindFirstChangeNotificationW",                  NULL},
{ 8,  308, "FindFirstFileA",                                NULL},
{24,  309, "FindFirstFileExA",                              NULL},
{24,  310, "FindFirstFileExW",                              NULL},
{20,  311, "FindFirstFileNameTransactedW",                  NULL},
{16,  312, "FindFirstFileNameW",                            NULL},
{28,  313, "FindFirstFileTransactedA",                      NULL},
{28,  314, "FindFirstFileTransactedW",                      NULL},
{ 8,  315, "FindFirstFileW",                                NULL},
{20,  316, "FindFirstStreamTransactedW",                    NULL},
{16,  317, "FindFirstStreamW",                              NULL},
{ 8,  318, "FindFirstVolumeA",                              NULL},
{12,  319, "FindFirstVolumeMountPointA",                    NULL},
{12,  320, "FindFirstVolumeMountPointW",                    NULL},
{ 8,  321, "FindFirstVolumeW",                              NULL},
{28,  322, "FindNLSString",                                 NULL},
{40,  323, "FindNLSStringEx",                               NULL},
{ 4,  324, "FindNextChangeNotification",                    NULL},
{ 8,  325, "FindNextFileA",                                 NULL},
{12,  326, "FindNextFileNameW",                             NULL},
{ 8,  327, "FindNextFileW",                                 NULL},
{ 8,  328, "FindNextStreamW",                               NULL},
{12,  329, "FindNextVolumeA",                               NULL},
{12,  330, "FindNextVolumeMountPointA",                     NULL},
{12,  331, "FindNextVolumeMountPointW",                     NULL},
{12,  332, "FindNextVolumeW",                               NULL},
{12,  333, "FindResourceA",                                 NULL},
{16,  334, "FindResourceExA",                               NULL},
{16,  335, "FindResourceExW",                               NULL},
{12,  336, "FindResourceW",                                 NULL},
{ 0,  337, "FindStringOrdinal",                             NULL},
{ 4,  338, "FindVolumeClose",                               NULL},
{ 4,  339, "FindVolumeMountPointClose",                     NULL},
{ 4,  340, "FlsAlloc",                                      NULL},
{ 4,  341, "FlsFree",                                       NULL},
{ 4,  342, "FlsGetValue",                                   NULL},
{ 8,  343, "FlsSetValue",                                   NULL},
{ 4,  344, "FlushConsoleInputBuffer",                       NULL},
{ 4,  345, "FlushFileBuffers",                              NULL},
{12,  346, "FlushInstructionCache",                         NULL},
{ 0,  347, "FlushProcessWriteBuffers",                      NULL},
{ 8,  348, "FlushViewOfFile",                               NULL},
{20,  349, "FoldStringA",                                   NULL},
{20,  350, "FoldStringW",                                   NULL},
{28,  351, "FormatMessageA",                                NULL},
{28,  352, "FormatMessageW",                                NULL},
{ 0,  353, "FreeConsole",                                   NULL},
{ 4,  354, "FreeEnvironmentStringsA",                       NULL},
{ 4,  355, "FreeEnvironmentStringsW",                       NULL},
{ 4,  356, "FreeLibrary",                                   NULL},
{ 8,  357, "FreeLibraryAndExitThread",                      NULL},
{ 8,  358, "FreeLibraryWhenCallbackReturns",                NULL},
{ 4,  359, "FreeResource",                                  NULL},
{12,  360, "FreeUserPhysicalPages",                         NULL},
{ 8,  361, "GenerateConsoleCtrlEvent",                      NULL},
{ 0,  362, "GetACP",                                        NULL},
{ 0,  363, "GetActiveProcessorCount",                       NULL},
{ 0,  364, "GetActiveProcessorGroupCount",                  NULL},
{20,  365, "GetApplicationRecoveryCallback",                NULL},
{16,  366, "GetApplicationRestartSettings",                 NULL},
{12,  367, "GetAtomNameA",                                  NULL},
{12,  368, "GetAtomNameW",                                  NULL},
{ 8,  369, "GetBinaryType",                                 NULL},
{ 8,  370, "GetBinaryTypeA",                                NULL},
{ 8,  371, "GetBinaryTypeW",                                NULL},
{ 8,  372, "GetCPInfo",                                     NULL},
{12,  373, "GetCPInfoExA",                                  NULL},
{12,  374, "GetCPInfoExW",                                  NULL},
{ 0,  375, "GetCalendarDateFormat",                         NULL},
{ 0,  376, "GetCalendarDateFormatEx",                       NULL},
{ 0,  377, "GetCalendarDaysInMonth",                        NULL},
{ 0,  378, "GetCalendarDifferenceInDays",                   NULL},
{24,  379, "GetCalendarInfoA",                              NULL},
{28,  380, "GetCalendarInfoEx",                             NULL},
{24,  381, "GetCalendarInfoW",                              NULL},
{ 0,  382, "GetCalendarMonthsInYear",                       NULL},
{ 0,  383, "GetCalendarSupportedDateRange",                 NULL},
{ 0,  384, "GetCalendarWeekNumber",                         NULL},
{ 0,  385, "GetComPlusPackageInstallStatus",                NULL},
{12,  386, "GetCommConfig",                                 NULL},
{ 8,  387, "GetCommMask",                                   NULL},
{ 8,  388, "GetCommModemStatus",                            NULL},
{ 8,  389, "GetCommProperties",                             NULL},
{ 8,  390, "GetCommState",                                  NULL},
{ 8,  391, "GetCommTimeouts",                               NULL},
{ 0,  392, "GetCommandLineA",                               NULL},
{ 0,  393, "GetCommandLineW",                               NULL},
{ 8,  394, "GetCompressedFileSizeA",                        NULL},
{12,  395, "GetCompressedFileSizeTransactedA",              NULL},
{12,  396, "GetCompressedFileSizeTransactedW",              NULL},
{ 8,  397, "GetCompressedFileSizeW",                        NULL},
{ 8,  398, "GetComputerNameA",                              NULL},
{12,  399, "GetComputerNameExA",                            NULL},
{12,  400, "GetComputerNameExW",                            NULL},
{ 8,  401, "GetComputerNameW",                              NULL},
{16,  402, "GetConsoleAliasA",                              NULL},
{ 8,  403, "GetConsoleAliasExesA",                          NULL},
{ 0,  404, "GetConsoleAliasExesLengthA",                    NULL},
{ 0,  405, "GetConsoleAliasExesLengthW",                    NULL},
{ 8,  406, "GetConsoleAliasExesW",                          NULL},
{16,  407, "GetConsoleAliasW",                              NULL},
{12,  408, "GetConsoleAliasesA",                            NULL},
{ 4,  409, "GetConsoleAliasesLengthA",                      NULL},
{ 4,  410, "GetConsoleAliasesLengthW",                      NULL},
{12,  411, "GetConsoleAliasesW",                            NULL},
{ 0,  412, "GetConsoleCP",                                  NULL},
{ 0,  413, "GetConsoleCharType",                            NULL},
{ 0,  414, "GetConsoleCommandHistoryA",                     NULL},
{ 0,  415, "GetConsoleCommandHistoryLengthA",               NULL},
{ 0,  416, "GetConsoleCommandHistoryLengthW",               NULL},
{ 0,  417, "GetConsoleCommandHistoryW",                     NULL},
{ 8,  418, "GetConsoleCursorInfo",                          NULL},
{ 0,  419, "GetConsoleCursorMode",                          NULL},
{ 4,  420, "GetConsoleDisplayMode",                         NULL},
{ 0,  421, "GetConsoleFontInfo",                            NULL},
{ 8,  422, "GetConsoleFontSize",                            NULL},
{ 0,  423, "GetConsoleHardwareState",                       NULL},
{ 4,  424, "GetConsoleHistoryInfo",                         NULL},
{ 0,  425, "GetConsoleInputExeNameA",                       NULL},
{ 0,  426, "GetConsoleInputExeNameW",                       NULL},
{ 0,  427, "GetConsoleInputWaitHandle",                     NULL},
{ 0,  428, "GetConsoleKeyboardLayoutNameA",                 NULL},
{ 0,  429, "GetConsoleKeyboardLayoutNameW",                 NULL},
{ 8,  430, "GetConsoleMode",                                NULL},
{ 0,  431, "GetConsoleNlsMode",                             NULL},
{ 8,  432, "GetConsoleOriginalTitleA",                      NULL},
{ 8,  433, "GetConsoleOriginalTitleW",                      NULL},
{ 0,  434, "GetConsoleOutputCP",                            NULL},
{ 8,  435, "GetConsoleProcessList",                         NULL},
{ 8,  436, "GetConsoleScreenBufferInfo",                    NULL},
{ 8,  437, "GetConsoleScreenBufferInfoEx",                  NULL},
{ 4,  438, "GetConsoleSelectionInfo",                       NULL},
{ 8,  439, "GetConsoleTitleA",                              NULL},
{ 8,  440, "GetConsoleTitleW",                              NULL},
{ 0,  441, "GetConsoleWindow",                              NULL},
{24,  442, "GetCurrencyFormatA",                            NULL},
{24,  443, "GetCurrencyFormatEx",                           NULL},
{24,  444, "GetCurrencyFormatW",                            NULL},
{ 4,  445, "GetCurrentActCtx",                              NULL},
{12,  446, "GetCurrentConsoleFont",                         NULL},
{12,  447, "GetCurrentConsoleFontEx",                       NULL},
{ 8,  448, "GetCurrentDirectoryA",                          NULL},
{ 8,  449, "GetCurrentDirectoryW",                          NULL},
{ 0,  450, "GetCurrentProcess",                             kernel32_GetCurrentProcess},
{ 0,  451, "GetCurrentProcessId",                           NULL},
{ 0,  452, "GetCurrentProcessorNumber",                     NULL},
{ 0,  453, "GetCurrentProcessorNumberEx",                   NULL},
{ 0,  454, "GetCurrentThread",                              NULL},
{ 0,  455, "GetCurrentThreadId",                            NULL},
{24,  456, "GetDateFormatA",                                NULL},
{28,  457, "GetDateFormatEx",                               NULL},
{24,  458, "GetDateFormatW",                                NULL},
{12,  459, "GetDefaultCommConfigA",                         NULL},
{12,  460, "GetDefaultCommConfigW",                         NULL},
{ 8,  461, "GetDevicePowerState",                           NULL},
{20,  462, "GetDiskFreeSpaceA",                             NULL},
{16,  463, "GetDiskFreeSpaceExA",                           NULL},
{16,  464, "GetDiskFreeSpaceExW",                           NULL},
{20,  465, "GetDiskFreeSpaceW",                             NULL},
{ 8,  466, "GetDllDirectoryA",                              NULL},
{ 8,  467, "GetDllDirectoryW",                              NULL},
{ 4,  468, "GetDriveTypeA",                                 NULL},
{ 4,  469, "GetDriveTypeW",                                 NULL},
{32,  470, "GetDurationFormat",                             NULL},
{32,  471, "GetDurationFormatEx",                           NULL},
{ 4,  472, "GetDynamicTimeZoneInformation",                 NULL},
{ 0,  473, "GetEnabledExtendedFeatures",                    NULL},
{ 0,  474, "GetEnvironmentStrings",                         NULL},
{ 0,  475, "GetEnvironmentStringsA",                        NULL},
{ 0,  476, "GetEnvironmentStringsW",                        NULL},
{12,  477, "GetEnvironmentVariableA",                       NULL},
{12,  478, "GetEnvironmentVariableW",                       NULL},
{ 0,  479, "GetEraNameCountedString",                       NULL},
{ 0,  480, "GetErrorMode",                                  NULL},
{ 8,  481, "GetExitCodeProcess",                            NULL},
{ 8,  482, "GetExitCodeThread",                             NULL},
{ 0,  483, "GetExpandedNameA",                              NULL},
{ 0,  484, "GetExpandedNameW",                              NULL},
{ 0,  485, "GetExtendedContextLength",                      NULL},
{ 0,  486, "GetExtendedFeaturesMask",                       NULL},
{ 4,  487, "GetFileAttributesA",                            NULL},
{12,  488, "GetFileAttributesExA",                          NULL},
{12,  489, "GetFileAttributesExW",                          NULL},
{16,  490, "GetFileAttributesTransactedA",                  NULL},
{16,  491, "GetFileAttributesTransactedW",                  NULL},
{ 4,  492, "GetFileAttributesW",                            NULL},
{24,  493, "GetFileBandwidthReservation",                   NULL},
{ 8,  494, "GetFileInformationByHandle",                    NULL},
{16,  495, "GetFileInformationByHandleEx",                  NULL},
{16,  496, "GetFileMUIInfo",                                NULL},
{28,  497, "GetFileMUIPath",                                NULL},
{ 8,  498, "GetFileSize",                                   NULL},
{ 8,  499, "GetFileSizeEx",                                 NULL},
{16,  500, "GetFileTime",                                   NULL},
{ 4,  501, "GetFileType",                                   NULL},
{16,  502, "GetFinalPathNameByHandleA",                     NULL},
{16,  503, "GetFinalPathNameByHandleW",                     NULL},
{16,  504, "GetFirmwareEnvironmentVariableA",               NULL},
{16,  505, "GetFirmwareEnvironmentVariableW",               NULL},
{16,  506, "GetFullPathNameA",                              NULL},
{20,  507, "GetFullPathNameTransactedA",                    NULL},
{20,  508, "GetFullPathNameTransactedW",                    NULL},
{16,  509, "GetFullPathNameW",                              NULL},
{20,  510, "GetGeoInfoA",                                   NULL},
{20,  511, "GetGeoInfoW",                                   NULL},
{ 0,  512, "GetHandleContext",                              NULL},
{ 8,  513, "GetHandleInformation",                          NULL},
{ 0,  514, "GetLargePageMinimum",                           NULL},
{ 4,  515, "GetLargestConsoleWindowSize",                   NULL},
{ 0,  516, "GetLastError",                                  NULL},
{ 4,  517, "GetLocalTime",                                  NULL},
{16,  518, "GetLocaleInfoA",                                NULL},
{16,  519, "GetLocaleInfoEx",                               NULL},
{16,  520, "GetLocaleInfoW",                                NULL},
{ 8,  521, "GetLogicalDriveStringsA",                       NULL},
{ 8,  522, "GetLogicalDriveStringsW",                       NULL},
{ 0,  523, "GetLogicalDrives",                              NULL},
{ 8,  524, "GetLogicalProcessorInformation",                NULL},
{ 0,  525, "GetLogicalProcessorInformationEx",              NULL},
{12,  526, "GetLongPathNameA",                              NULL},
{16,  527, "GetLongPathNameTransactedA",                    NULL},
{16,  528, "GetLongPathNameTransactedW",                    NULL},
{12,  529, "GetLongPathNameW",                              NULL},
{20,  530, "GetMailslotInfo",                               NULL},
{ 0,  531, "GetMaximumProcessorCount",                      NULL},
{ 0,  532, "GetMaximumProcessorGroupCount",                 NULL},
{12,  533, "GetModuleFileNameA",                            kernel32_GetModuleFileNameA},
{12,  534, "GetModuleFileNameW",                            kernel32_GetModuleFileNameW},
{ 4,  535, "GetModuleHandleA",                              kernel32_GetModuleHandleA},
{12,  536, "GetModuleHandleExA",                            NULL},
{12,  537, "GetModuleHandleExW",                            NULL},
{ 4,  538, "GetModuleHandleW",                              kernel32_GetModuleHandleW},
{12,  539, "GetNLSVersion",                                 NULL},
{12,  540, "GetNLSVersionEx",                               NULL},
{20,  541, "GetNamedPipeAttribute",                         NULL},
{12,  542, "GetNamedPipeClientComputerNameA",               NULL},
{12,  543, "GetNamedPipeClientComputerNameW",               NULL},
{ 8,  544, "GetNamedPipeClientProcessId",                   NULL},
{ 8,  545, "GetNamedPipeClientSessionId",                   NULL},
{28,  546, "GetNamedPipeHandleStateA",                      NULL},
{28,  547, "GetNamedPipeHandleStateW",                      NULL},
{20,  548, "GetNamedPipeInfo",                              NULL},
{ 8,  549, "GetNamedPipeServerProcessId",                   NULL},
{ 8,  550, "GetNamedPipeServerSessionId",                   NULL},
{ 4,  551, "GetNativeSystemInfo",                           NULL},
{ 0,  552, "GetNextVDMCommand",                             NULL},
{ 8,  553, "GetNumaAvailableMemoryNode",                    NULL},
{ 0,  554, "GetNumaAvailableMemoryNodeEx",                  NULL},
{ 4,  555, "GetNumaHighestNodeNumber",                      NULL},
{ 0,  556, "GetNumaNodeNumberFromHandle",                   NULL},
{ 8,  557, "GetNumaNodeProcessorMask",                      NULL},
{ 0,  558, "GetNumaNodeProcessorMaskEx",                    NULL},
{ 8,  559, "GetNumaProcessorNode",                          NULL},
{ 0,  560, "GetNumaProcessorNodeEx",                        NULL},
{ 8,  561, "GetNumaProximityNode",                          NULL},
{ 0,  562, "GetNumaProximityNodeEx",                        NULL},
{24,  563, "GetNumberFormatA",                              NULL},
{24,  564, "GetNumberFormatEx",                             NULL},
{24,  565, "GetNumberFormatW",                              NULL},
{ 0,  566, "GetNumberOfConsoleFonts",                       NULL},
{ 8,  567, "GetNumberOfConsoleInputEvents",                 NULL},
{ 4,  568, "GetNumberOfConsoleMouseButtons",                NULL},
{ 0,  569, "GetOEMCP",                                      NULL},
{16,  570, "GetOverlappedResult",                           NULL},
{ 0,  571, "GetPhysicallyInstalledSystemMemory",            NULL},
{ 4,  572, "GetPriorityClass",                              kernel32_GetPriorityClass},
{16,  573, "GetPrivateProfileIntA",                         NULL},
{16,  574, "GetPrivateProfileIntW",                         NULL},
{16,  575, "GetPrivateProfileSectionA",                     NULL},
{12,  576, "GetPrivateProfileSectionNamesA",                NULL},
{12,  577, "GetPrivateProfileSectionNamesW",                NULL},
{16,  578, "GetPrivateProfileSectionW",                     NULL},
{24,  579, "GetPrivateProfileStringA",                      NULL},
{24,  580, "GetPrivateProfileStringW",                      NULL},
{20,  581, "GetPrivateProfileStructA",                      NULL},
{20,  582, "GetPrivateProfileStructW",                      NULL},
{ 8,  583, "GetProcAddress",                                kernel32_GetProcAddress},
{12,  584, "GetProcessAffinityMask",                        NULL},
{ 0,  585, "GetProcessDEPPolicy",                           NULL},
{ 0,  586, "GetProcessGroupAffinity",                       NULL},
{ 8,  587, "GetProcessHandleCount",                         NULL},
{ 0,  588, "GetProcessHeap",                                NULL},
{ 8,  589, "GetProcessHeaps",                               NULL},
{ 4,  590, "GetProcessId",                                  NULL},
{ 4,  591, "GetProcessIdOfThread",                          NULL},
{ 8,  592, "GetProcessIoCounters",                          NULL},
{ 0,  593, "GetProcessPreferredUILanguages",                NULL},
{ 8,  594, "GetProcessPriorityBoost",                       NULL},
{ 8,  595, "GetProcessShutdownParameters",                  NULL},
{20,  596, "GetProcessTimes",                               NULL},
{ 4,  597, "GetProcessVersion",                             NULL},
{12,  598, "GetProcessWorkingSetSize",                      NULL},
{16,  599, "GetProcessWorkingSetSizeEx",                    NULL},
{ 0,  600, "GetProcessorSystemCycleTime",                   NULL},
{20,  601, "GetProductInfo",                                NULL},
{12,  602, "GetProfileIntA",                                NULL},
{12,  603, "GetProfileIntW",                                NULL},
{12,  604, "GetProfileSectionA",                            NULL},
{12,  605, "GetProfileSectionW",                            NULL},
{20,  606, "GetProfileStringA",                             NULL},
{20,  607, "GetProfileStringW",                             NULL},
{20,  608, "GetQueuedCompletionStatus",                     NULL},
{24,  609, "GetQueuedCompletionStatusEx",                   NULL},
{12,  610, "GetShortPathNameA",                             NULL},
{12,  611, "GetShortPathNameW",                             NULL},
{ 4,  612, "GetStartupInfoA",                               NULL},
{ 4,  613, "GetStartupInfoW",                               NULL},
{ 4,  614, "GetStdHandle",                                  NULL},
{20,  615, "GetStringScripts",                              NULL},
{20,  616, "GetStringTypeA",                                NULL},
{20,  617, "GetStringTypeExA",                              NULL},
{20,  618, "GetStringTypeExW",                              NULL},
{16,  619, "GetStringTypeW",                                NULL},
{ 0,  620, "GetSystemDEPPolicy",                            NULL},
{ 0,  621, "GetSystemDefaultLCID",                          NULL},
{ 0,  622, "GetSystemDefaultLangID",                        NULL},
{ 8,  623, "GetSystemDefaultLocaleName",                    NULL},
{ 0,  624, "GetSystemDefaultUILanguage",                    NULL},
{ 8,  625, "GetSystemDirectoryA",                           NULL},
{ 8,  626, "GetSystemDirectoryW",                           NULL},
{12,  627, "GetSystemFileCacheSize",                        NULL},
{16,  628, "GetSystemFirmwareTable",                        NULL},
{ 4,  629, "GetSystemInfo",                                 NULL},
{ 4,  630, "GetSystemPowerStatus",                          NULL},
{16,  631, "GetSystemPreferredUILanguages",                 NULL},
{ 8,  632, "GetSystemRegistryQuota",                        NULL},
{ 4,  633, "GetSystemTime",                                 NULL},
{12,  634, "GetSystemTimeAdjustment",                       NULL},
{ 4,  635, "GetSystemTimeAsFileTime",                       NULL},
{12,  636, "GetSystemTimes",                                NULL},
{ 8,  637, "GetSystemWindowsDirectoryA",                    NULL},
{ 8,  638, "GetSystemWindowsDirectoryW",                    NULL},
{ 8,  639, "GetSystemWow64DirectoryA",                      NULL},
{ 8,  640, "GetSystemWow64DirectoryW",                      NULL},
{16,  641, "GetTapeParameters",                             NULL},
{20,  642, "GetTapePosition",                               NULL},
{ 4,  643, "GetTapeStatus",                                 NULL},
{16,  644, "GetTempFileNameA",                              NULL},
{16,  645, "GetTempFileNameW",                              NULL},
{ 8,  646, "GetTempPathA",                                  NULL},
{ 8,  647, "GetTempPathW",                                  NULL},
{ 8,  648, "GetThreadContext",                              NULL},
{ 0,  649, "GetThreadErrorMode",                            NULL},
{ 0,  650, "GetThreadGroupAffinity",                        NULL},
{ 8,  651, "GetThreadIOPendingFlag",                        NULL},
{ 4,  652, "GetThreadId",                                   NULL},
{ 0,  653, "GetThreadIdealProcessorEx",                     NULL},
{ 0,  654, "GetThreadLocale",                               NULL},
{16,  655, "GetThreadPreferredUILanguages",                 NULL},
{ 4,  656, "GetThreadPriority",                             NULL},
{ 8,  657, "GetThreadPriorityBoost",                        NULL},
{12,  658, "GetThreadSelectorEntry",                        NULL},
{20,  659, "GetThreadTimes",                                NULL},
{ 0,  660, "GetThreadUILanguage",                           NULL},
{ 0,  662, "GetTickCount",                                  kernel32_GetTickCount},
{ 0,  661, "GetTickCount64",                                NULL},
{24,  663, "GetTimeFormatA",                                NULL},
{24,  664, "GetTimeFormatEx",                               NULL},
{24,  665, "GetTimeFormatW",                                NULL},
{ 4,  666, "GetTimeZoneInformation",                        NULL},
{ 0,  667, "GetTimeZoneInformationForYear",                 NULL},
{20,  668, "GetUILanguageInfo",                             NULL},
{ 0,  669, "GetUserDefaultLCID",                            NULL},
{ 0,  670, "GetUserDefaultLangID",                          NULL},
{ 8,  671, "GetUserDefaultLocaleName",                      NULL},
{ 0,  672, "GetUserDefaultUILanguage",                      NULL},
{ 4,  673, "GetUserGeoID",                                  NULL},
{16,  674, "GetUserPreferredUILanguages",                   NULL},
{ 0,  675, "GetVDMCurrentDirectories",                      NULL},
{ 0,  676, "GetVersion",                                    kernel32_GetVersion},
{ 4,  677, "GetVersionExA",                                 NULL},
{ 4,  678, "GetVersionExW",                                 NULL},
{32,  679, "GetVolumeInformationA",                         NULL},
{32,  680, "GetVolumeInformationByHandleW",                 NULL},
{32,  681, "GetVolumeInformationW",                         NULL},
{12,  682, "GetVolumeNameForVolumeMountPointA",             NULL},
{12,  683, "GetVolumeNameForVolumeMountPointW",             NULL},
{12,  684, "GetVolumePathNameA",                            NULL},
{12,  685, "GetVolumePathNameW",                            NULL},
{16,  686, "GetVolumePathNamesForVolumeNameA",              NULL},
{16,  687, "GetVolumePathNamesForVolumeNameW",              NULL},
{ 8,  688, "GetWindowsDirectoryA",                          NULL},
{ 8,  689, "GetWindowsDirectoryW",                          NULL},
{24,  690, "GetWriteWatch",                                 NULL},
{ 4,  691, "GlobalAddAtomA",                                NULL},
{ 4,  692, "GlobalAddAtomW",                                NULL},
{ 8,  693, "GlobalAlloc",                                   kernel32_GlobalAlloc},
{ 4,  694, "GlobalCompact",                                 NULL},
{ 4,  695, "GlobalDeleteAtom",                              NULL},
{ 4,  696, "GlobalFindAtomA",                               NULL},
{ 4,  697, "GlobalFindAtomW",                               NULL},
{ 4,  698, "GlobalFix",                                     NULL},
{ 4,  699, "GlobalFlags",                                   NULL},
{ 4,  700, "GlobalFree",                                    kernel32_GlobalFree},
{12,  701, "GlobalGetAtomNameA",                            NULL},
{12,  702, "GlobalGetAtomNameW",                            NULL},
{ 4,  703, "GlobalHandle",                                  kernel32_GlobalHandle},
{ 4,  704, "GlobalLock",                                    kernel32_GlobalLock},
{ 4,  705, "GlobalMemoryStatus",                            NULL},
{ 4,  706, "GlobalMemoryStatusEx",                          NULL},
{12,  707, "GlobalReAlloc",                                 NULL},
{ 4,  708, "GlobalSize",                                    NULL},
{ 4,  709, "GlobalUnWire",                                  NULL},
{ 4,  710, "GlobalUnfix",                                   NULL},
{ 4,  711, "GlobalUnlock",                                  kernel32_GlobalUnlock},
{ 4,  712, "GlobalWire",                                    NULL},
{12,  713, "Heap32First",                                   NULL},
{ 8,  714, "Heap32ListFirst",                               NULL},
{ 8,  715, "Heap32ListNext",                                NULL},
{ 4,  716, "Heap32Next",                                    NULL},
{12,  717, "HeapAlloc",                                     NULL},
{ 8,  718, "HeapCompact",                                   NULL},
{12,  719, "HeapCreate",                                    NULL},
{ 4,  720, "HeapDestroy",                                   NULL},
{12,  721, "HeapFree",                                      NULL},
{ 4,  722, "HeapLock",                                      NULL},
{20,  723, "HeapQueryInformation",                          NULL},
{16,  724, "HeapReAlloc",                                   NULL},
{16,  725, "HeapSetInformation",                            NULL},
{12,  726, "HeapSize",                                      NULL},
{ 0,  727, "HeapSummary",                                   NULL},
{ 4,  728, "HeapUnlock",                                    NULL},
{12,  729, "HeapValidate",                                  NULL},
{ 8,  730, "HeapWalk",                                      NULL},
{ 0,  731, "IdnToAscii",                                    NULL},
{ 0,  732, "IdnToNameprepUnicode",                          NULL},
{ 0,  733, "IdnToUnicode",                                  NULL},
{ 4,  734, "InitAtomTable",                                 NULL},
{16,  735, "InitOnceBeginInitialize",                       NULL},
{12,  736, "InitOnceComplete",                              NULL},
{16,  737, "InitOnceExecuteOnce",                           NULL},
{ 4,  738, "InitOnceInitialize",                            NULL},
{ 4,  739, "InitializeConditionVariable",                   NULL},
{ 4,  740, "InitializeCriticalSection",                     NULL},
{ 8,  741, "InitializeCriticalSectionAndSpinCount",         NULL},
{12,  742, "InitializeCriticalSectionEx",                   NULL},
{ 0,  743, "InitializeExtendedContext",                     NULL},
{16,  744, "InitializeProcThreadAttributeList",             NULL},
{ 4,  745, "InitializeSListHead",                           NULL},
{ 4,  746, "InitializeSRWLock",                             NULL},
{12,  748, "InterlockedCompareExchange",                    NULL},
{20,  747, "InterlockedCompareExchange64",                  NULL},
{ 4,  749, "InterlockedDecrement",                          NULL},
{ 8,  750, "InterlockedExchange",                           NULL},
{ 8,  751, "InterlockedExchangeAdd",                        NULL},
{ 4,  752, "InterlockedFlushSList",                         NULL},
{ 4,  753, "InterlockedIncrement",                          NULL},
{ 4,  754, "InterlockedPopEntrySList",                      NULL},
{ 8,  755, "InterlockedPushEntrySList",                     NULL},
{ 0,    2, "InterlockedPushListSList",                      NULL},
{ 0,  756, "InvalidateConsoleDIBits",                       NULL},
{ 4,  757, "IsBadCodePtr",                                  NULL},
{ 8,  758, "IsBadHugeReadPtr",                              NULL},
{ 8,  759, "IsBadHugeWritePtr",                             NULL},
{ 8,  760, "IsBadReadPtr",                                  kernel32_IsBadReadPtr},
{ 8,  761, "IsBadStringPtrA",                               NULL},
{ 8,  762, "IsBadStringPtrW",                               NULL},
{ 8,  763, "IsBadWritePtr",                                 kernel32_IsBadWritePtr},
{ 0,  764, "IsCalendarLeapDay",                             NULL},
{ 0,  765, "IsCalendarLeapMonth",                           NULL},
{ 0,  766, "IsCalendarLeapYear",                            NULL},
{ 4,  767, "IsDBCSLeadByte",                                NULL},
{ 8,  768, "IsDBCSLeadByteEx",                              NULL},
{ 0,  769, "IsDebuggerPresent",                             kernel32_IsDebuggerPresent},
{20,  770, "IsNLSDefinedString",                            NULL},
{ 0,  771, "IsNormalizedString",                            NULL},
{12,  772, "IsProcessInJob",                                NULL},
{ 4,  773, "IsProcessorFeaturePresent",                     NULL},
{ 0,  774, "IsSystemResumeAutomatic",                       NULL},
{ 0,  775, "IsThreadAFiber",                                NULL},
{ 4,  776, "IsThreadpoolTimerSet",                          NULL},
{ 0,  777, "IsTimeZoneRedirectionEnabled",                  NULL},
{ 0,  778, "IsValidCalDateTime",                            NULL},
{ 4,  779, "IsValidCodePage",                               NULL},
{ 8,  780, "IsValidLanguageGroup",                          NULL},
{ 8,  781, "IsValidLocale",                                 NULL},
{ 4,  782, "IsValidLocaleName",                             NULL},
{ 8,  783, "IsWow64Process",                                NULL},
{ 0,  784, "K32EmptyWorkingSet",                            NULL},
{ 0,  785, "K32EnumDeviceDrivers",                          NULL},
{ 0,  786, "K32EnumPageFilesA",                             NULL},
{ 0,  787, "K32EnumPageFilesW",                             NULL},
{ 0,  788, "K32EnumProcessModules",                         NULL},
{ 0,  789, "K32EnumProcessModulesEx",                       NULL},
{ 0,  790, "K32EnumProcesses",                              NULL},
{ 0,  791, "K32GetDeviceDriverBaseNameA",                   NULL},
{ 0,  792, "K32GetDeviceDriverBaseNameW",                   NULL},
{ 0,  793, "K32GetDeviceDriverFileNameA",                   NULL},
{ 0,  794, "K32GetDeviceDriverFileNameW",                   NULL},
{ 0,  795, "K32GetMappedFileNameA",                         NULL},
{ 0,  796, "K32GetMappedFileNameW",                         NULL},
{ 0,  797, "K32GetModuleBaseNameA",                         NULL},
{ 0,  798, "K32GetModuleBaseNameW",                         NULL},
{ 0,  799, "K32GetModuleFileNameExA",                       NULL},
{ 0,  800, "K32GetModuleFileNameExW",                       NULL},
{ 0,  801, "K32GetModuleInformation",                       NULL},
{ 0,  802, "K32GetPerformanceInfo",                         NULL},
{ 0,  803, "K32GetProcessImageFileNameA",                   NULL},
{ 0,  804, "K32GetProcessImageFileNameW",                   NULL},
{ 0,  805, "K32GetProcessMemoryInfo",                       NULL},
{ 0,  806, "K32GetWsChanges",                               NULL},
{ 0,  807, "K32GetWsChangesEx",                             NULL},
{ 0,  808, "K32InitializeProcessForWsWatch",                NULL},
{ 0,  809, "K32QueryWorkingSet",                            NULL},
{ 0,  810, "K32QueryWorkingSetEx",                          NULL},
{16,  811, "LCIDToLocaleName",                              NULL},
{24,  812, "LCMapStringA",                                  NULL},
{36,  813, "LCMapStringEx",                                 NULL},
{24,  814, "LCMapStringW",                                  NULL},
{ 0,  815, "LZClose",                                       NULL},
{ 0,  816, "LZCloseFile",                                   NULL},
{ 0,  817, "LZCopy",                                        NULL},
{ 0,  818, "LZCreateFileW",                                 NULL},
{ 0,  819, "LZDone",                                        NULL},
{ 0,  820, "LZInit",                                        NULL},
{ 0,  821, "LZOpenFileA",                                   NULL},
{ 0,  822, "LZOpenFileW",                                   NULL},
{ 0,  823, "LZRead",                                        NULL},
{ 0,  824, "LZSeek",                                        NULL},
{ 0,  825, "LZStart",                                       NULL},
{ 4,  826, "LeaveCriticalSection",                          NULL},
{ 8,  827, "LeaveCriticalSectionWhenCallbackReturns",       NULL},
{ 0,  828, "LoadAppInitDlls",                               NULL},
{ 4,  829, "LoadLibraryA",                                  kernel32_LoadLibraryA},
{12,  830, "LoadLibraryExA",                                NULL},
{12,  831, "LoadLibraryExW",                                NULL},
{ 4,  832, "LoadLibraryW",                                  kernel32_LoadLibraryW},
{ 8,  833, "LoadModule",                                    NULL},
{ 8,  834, "LoadResource",                                  NULL},
{20,  835, "LoadStringBaseExW",                             NULL},
{16,  836, "LoadStringBaseW",                               NULL},
{ 8,  837, "LocalAlloc",                                    NULL},
{ 4,  838, "LocalCompact",                                  NULL},
{ 8,  839, "LocalFileTimeToFileTime",                       NULL},
{ 4,  840, "LocalFlags",                                    NULL},
{ 4,  841, "LocalFree",                                     NULL},
{ 4,  842, "LocalHandle",                                   NULL},
{ 4,  843, "LocalLock",                                     NULL},
{12,  844, "LocalReAlloc",                                  NULL},
{ 8,  845, "LocalShrink",                                   NULL},
{ 4,  846, "LocalSize",                                     NULL},
{ 4,  847, "LocalUnlock",                                   NULL},
{ 8,  848, "LocaleNameToLCID",                              NULL},
{ 0,  849, "LocateExtendedFeature",                         NULL},
{ 0,  850, "LocateLegacyContext",                           NULL},
{20,  851, "LockFile",                                      NULL},
{24,  852, "LockFileEx",                                    NULL},
{ 4,  853, "LockResource",                                  NULL},
{12,  854, "MapUserPhysicalPages",                          NULL},
{12,  855, "MapUserPhysicalPagesScatter",                   NULL},
{20,  856, "MapViewOfFile",                                 NULL},
{24,  857, "MapViewOfFileEx",                               NULL},
{28,  858, "MapViewOfFileExNuma",                           NULL},
{ 8,  859, "Module32First",                                 NULL},
{ 8,  860, "Module32FirstW",                                NULL},
{ 8,  861, "Module32Next",                                  NULL},
{ 8,  862, "Module32NextW",                                 NULL},
{ 8,  863, "MoveFileA",                                     NULL},
{12,  864, "MoveFileExA",                                   NULL},
{12,  865, "MoveFileExW",                                   NULL},
{24,  866, "MoveFileTransactedA",                           NULL},
{24,  867, "MoveFileTransactedW",                           NULL},
{ 8,  868, "MoveFileW",                                     NULL},
{20,  869, "MoveFileWithProgressA",                         NULL},
{20,  870, "MoveFileWithProgressW",                         NULL},
{12,  871, "MulDiv",                                        NULL},
{24,  872, "MultiByteToWideChar",                           NULL},
{ 4,  873, "NeedCurrentDirectoryForExePathA",               NULL},
{ 4,  874, "NeedCurrentDirectoryForExePathW",               NULL},
{ 0,  875, "NlsCheckPolicy",                                NULL},
{ 0,  876, "NlsEventDataDescCreate",                        NULL},
{ 0,  877, "NlsGetCacheUpdateCount",                        NULL},
{ 0,  878, "NlsUpdateLocale",                               NULL},
{ 0,  879, "NlsUpdateSystemLocale",                         NULL},
{ 0,  880, "NlsWriteEtwEvent",                              NULL},
{ 0,  881, "NormalizeString",                               NULL},
{ 0,  882, "NotifyMountMgr",                                NULL},
{20,  883, "NotifyUILanguageChange",                        NULL},
{ 0,  884, "OpenConsoleW",                                  NULL},
{12,  885, "OpenEventA",                                    NULL},
{12,  886, "OpenEventW",                                    NULL},
{12,  887, "OpenFile",                                      NULL},
{24,  888, "OpenFileById",                                  NULL},
{12,  889, "OpenFileMappingA",                              NULL},
{12,  890, "OpenFileMappingW",                              NULL},
{12,  891, "OpenJobObjectA",                                NULL},
{12,  892, "OpenJobObjectW",                                NULL},
{12,  893, "OpenMutexA",                                    NULL},
{12,  894, "OpenMutexW",                                    NULL},
{ 8,  895, "OpenPrivateNamespaceA",                         NULL},
{ 8,  896, "OpenPrivateNamespaceW",                         NULL},
{12,  897, "OpenProcess",                                   NULL},
{ 0,  898, "OpenProcessToken",                              NULL},
{ 0,  899, "OpenProfileUserMapping",                        NULL},
{12,  900, "OpenSemaphoreA",                                NULL},
{12,  901, "OpenSemaphoreW",                                NULL},
{12,  902, "OpenThread",                                    NULL},
{ 0,  903, "OpenThreadToken",                               NULL},
{12,  904, "OpenWaitableTimerA",                            NULL},
{12,  905, "OpenWaitableTimerW",                            NULL},
{ 4,  906, "OutputDebugStringA",                            NULL},
{ 4,  907, "OutputDebugStringW",                            NULL},
{16,  908, "PeekConsoleInputA",                             NULL},
{16,  909, "PeekConsoleInputW",                             NULL},
{24,  910, "PeekNamedPipe",                                 NULL},
{16,  911, "PostQueuedCompletionStatus",                    NULL},
{ 0,  912, "PowerClearRequest",                             NULL},
{ 0,  913, "PowerCreateRequest",                            NULL},
{ 0,  914, "PowerSetRequest",                               NULL},
{12,  915, "PrepareTape",                                   NULL},
{ 0,  916, "PrivCopyFileExW",                               NULL},
{ 0,  917, "PrivMoveFileIdentityW",                         NULL},
{ 8,  918, "Process32First",                                NULL},
{ 8,  919, "Process32FirstW",                               NULL},
{ 8,  920, "Process32Next",                                 NULL},
{ 8,  921, "Process32NextW",                                NULL},
{ 8,  922, "ProcessIdToSessionId",                          NULL},
{ 4,  923, "PulseEvent",                                    NULL},
{ 8,  924, "PurgeComm",                                     NULL},
{28,  925, "QueryActCtxSettingsW",                          NULL},
{28,  926, "QueryActCtxW",                                  NULL},
{ 4,  927, "QueryDepthSList",                               NULL},
{12,  928, "QueryDosDeviceA",                               NULL},
{12,  929, "QueryDosDeviceW",                               NULL},
{16,  930, "QueryFullProcessImageNameA",                    NULL},
{16,  931, "QueryFullProcessImageNameW",                    NULL},
{ 8,  932, "QueryIdleProcessorCycleTime",                   NULL},
{ 0,  933, "QueryIdleProcessorCycleTimeEx",                 NULL},
{20,  934, "QueryInformationJobObject",                     NULL},
{ 8,  935, "QueryMemoryResourceNotification",               NULL},
{ 4,  936, "QueryPerformanceCounter",                       NULL},
{ 4,  937, "QueryPerformanceFrequency",                     NULL},
{ 0,  938, "QueryProcessAffinityUpdateMode",                NULL},
{ 8,  939, "QueryProcessCycleTime",                         NULL},
{ 8,  940, "QueryThreadCycleTime",                          NULL},
{ 0,  941, "QueryThreadProfiling",                          NULL},
{ 0,  942, "QueryThreadpoolStackInformation",               NULL},
{ 0,  943, "QueryUnbiasedInterruptTime",                    NULL},
{12,  944, "QueueUserAPC",                                  NULL},
{12,  945, "QueueUserWorkItem",                             NULL},
{16,  946, "RaiseException",                                NULL},
{ 0,  947, "RaiseFailFastException",                        NULL},
{16,  948, "ReOpenFile",                                    NULL},
{20,  949, "ReadConsoleA",                                  NULL},
{16,  950, "ReadConsoleInputA",                             NULL},
{ 0,  951, "ReadConsoleInputExA",                           NULL},
{ 0,  952, "ReadConsoleInputExW",                           NULL},
{16,  953, "ReadConsoleInputW",                             NULL},
{20,  954, "ReadConsoleOutputA",                            NULL},
{20,  955, "ReadConsoleOutputAttribute",                    NULL},
{20,  956, "ReadConsoleOutputCharacterA",                   NULL},
{20,  957, "ReadConsoleOutputCharacterW",                   NULL},
{20,  958, "ReadConsoleOutputW",                            NULL},
{20,  959, "ReadConsoleW",                                  NULL},
{32,  960, "ReadDirectoryChangesW",                         NULL},
{20,  961, "ReadFile",                                      NULL},
{20,  962, "ReadFileEx",                                    NULL},
{20,  963, "ReadFileScatter",                               NULL},
{20,  964, "ReadProcessMemory",                             NULL},
{ 0,  965, "ReadThreadProfilingData",                       NULL},
{ 0,  966, "RegCloseKey",                                   NULL},
{ 0,  967, "RegCreateKeyExA",                               NULL},
{ 0,  968, "RegCreateKeyExW",                               NULL},
{ 0,  969, "RegDeleteKeyExA",                               NULL},
{ 0,  970, "RegDeleteKeyExW",                               NULL},
{ 0,  971, "RegDeleteTreeA",                                NULL},
{ 0,  972, "RegDeleteTreeW",                                NULL},
{ 0,  973, "RegDeleteValueA",                               NULL},
{ 0,  974, "RegDeleteValueW",                               NULL},
{ 0,  975, "RegDisablePredefinedCacheEx",                   NULL},
{ 0,  976, "RegEnumKeyExA",                                 NULL},
{ 0,  977, "RegEnumKeyExW",                                 NULL},
{ 0,  978, "RegEnumValueA",                                 NULL},
{ 0,  979, "RegEnumValueW",                                 NULL},
{ 0,  980, "RegFlushKey",                                   NULL},
{ 0,  981, "RegGetKeySecurity",                             NULL},
{ 0,  982, "RegGetValueA",                                  NULL},
{ 0,  983, "RegGetValueW",                                  NULL},
{ 0,  984, "RegKrnGetGlobalState",                          NULL},
{ 0,  985, "RegKrnInitialize",                              NULL},
{ 0,  986, "RegLoadKeyA",                                   NULL},
{ 0,  987, "RegLoadKeyW",                                   NULL},
{ 0,  988, "RegLoadMUIStringA",                             NULL},
{ 0,  989, "RegLoadMUIStringW",                             NULL},
{ 0,  990, "RegNotifyChangeKeyValue",                       NULL},
{ 0,  991, "RegOpenCurrentUser",                            NULL},
{ 0,  992, "RegOpenKeyExA",                                 NULL},
{ 0,  993, "RegOpenKeyExW",                                 NULL},
{ 0,  994, "RegOpenUserClassesRoot",                        NULL},
{ 0,  995, "RegQueryInfoKeyA",                              NULL},
{ 0,  996, "RegQueryInfoKeyW",                              NULL},
{ 0,  997, "RegQueryValueExA",                              NULL},
{ 0,  998, "RegQueryValueExW",                              NULL},
{ 0,  999, "RegRestoreKeyA",                                NULL},
{ 0, 1000, "RegRestoreKeyW",                                NULL},
{ 0, 1001, "RegSaveKeyExA",                                 NULL},
{ 0, 1002, "RegSaveKeyExW",                                 NULL},
{ 0, 1003, "RegSetKeySecurity",                             NULL},
{ 0, 1004, "RegSetValueExA",                                NULL},
{ 0, 1005, "RegSetValueExW",                                NULL},
{ 0, 1006, "RegUnLoadKeyA",                                 NULL},
{ 0, 1007, "RegUnLoadKeyW",                                 NULL},
{16, 1008, "RegisterApplicationRecoveryCallback",           NULL},
{ 8, 1009, "RegisterApplicationRestart",                    NULL},
{ 0, 1010, "RegisterConsoleIME",                            NULL},
{ 0, 1011, "RegisterConsoleOS2",                            NULL},
{ 0, 1012, "RegisterConsoleVDM",                            NULL},
{ 0, 1013, "RegisterWaitForInputIdle",                      NULL},
{24, 1014, "RegisterWaitForSingleObject",                   NULL},
{20, 1015, "RegisterWaitForSingleObjectEx",                 NULL},
{ 0, 1016, "RegisterWowBaseHandlers",                       NULL},
{ 0, 1017, "RegisterWowExec",                               NULL},
{ 4, 1018, "ReleaseActCtx",                                 NULL},
{ 4, 1019, "ReleaseMutex",                                  NULL},
{ 8, 1020, "ReleaseMutexWhenCallbackReturns",               NULL},
{ 4, 1021, "ReleaseSRWLockExclusive",                       NULL},
{ 4, 1022, "ReleaseSRWLockShared",                          NULL},
{12, 1023, "ReleaseSemaphore",                              NULL},
{12, 1024, "ReleaseSemaphoreWhenCallbackReturns",           NULL},
{ 4, 1025, "RemoveDirectoryA",                              NULL},
{ 8, 1026, "RemoveDirectoryTransactedA",                    NULL},
{ 8, 1027, "RemoveDirectoryTransactedW",                    NULL},
{ 4, 1028, "RemoveDirectoryW",                              NULL},
{ 8, 1029, "RemoveLocalAlternateComputerNameA",             NULL},
{ 8, 1030, "RemoveLocalAlternateComputerNameW",             NULL},
{ 0, 1031, "RemoveSecureMemoryCacheCallback",               NULL},
{ 4, 1032, "RemoveVectoredContinueHandler",                 NULL},
{ 4, 1033, "RemoveVectoredExceptionHandler",                NULL},
{24, 1034, "ReplaceFile",                                   NULL},
{24, 1035, "ReplaceFileA",                                  NULL},
{24, 1036, "ReplaceFileW",                                  NULL},
{ 0, 1037, "ReplacePartitionUnit",                          NULL},
{ 4, 1038, "RequestDeviceWakeup",                           NULL},
{ 4, 1039, "RequestWakeupLatency",                          NULL},
{ 4, 1040, "ResetEvent",                                    NULL},
{ 8, 1041, "ResetWriteWatch",                               NULL},
{ 0, 1042, "ResolveLocaleName",                             NULL},
{ 4, 1043, "RestoreLastError",                              NULL},
{ 4, 1044, "ResumeThread",                                  NULL},
{ 4, 1045, "RtlCaptureContext",                             NULL},
{16, 1046, "RtlCaptureStackBackTrace",                      NULL},
{12, 1047, "RtlFillMemory",                                 NULL},
{12, 1048, "RtlMoveMemory",                                 NULL},
{16, 1049, "RtlUnwind",                                     NULL},
{ 8, 1050, "RtlZeroMemory",                                 NULL},
{20, 1051, "ScrollConsoleScreenBufferA",                    NULL},
{20, 1052, "ScrollConsoleScreenBufferW",                    NULL},
{24, 1053, "SearchPathA",                                   NULL},
{24, 1054, "SearchPathW",                                   NULL},
{16, 1055, "SetCalendarInfoA",                              NULL},
{16, 1056, "SetCalendarInfoW",                              NULL},
{ 0, 1057, "SetClientTimeZoneInformation",                  NULL},
{ 0, 1058, "SetComPlusPackageInstallStatus",                NULL},
{ 4, 1059, "SetCommBreak",                                  NULL},
{12, 1060, "SetCommConfig",                                 NULL},
{ 8, 1061, "SetCommMask",                                   NULL},
{ 8, 1062, "SetCommState",                                  NULL},
{ 8, 1063, "SetCommTimeouts",                               NULL},
{ 4, 1064, "SetComputerNameA",                              NULL},
{ 8, 1065, "SetComputerNameExA",                            NULL},
{ 8, 1066, "SetComputerNameExW",                            NULL},
{ 4, 1067, "SetComputerNameW",                              NULL},
{ 4, 1068, "SetConsoleActiveScreenBuffer",                  NULL},
{ 4, 1069, "SetConsoleCP",                                  NULL},
{ 8, 1070, "SetConsoleCtrlHandler",                         NULL},
{ 8, 1071, "SetConsoleCursor",                              NULL},
{ 8, 1072, "SetConsoleCursorInfo",                          NULL},
{ 0, 1073, "SetConsoleCursorMode",                          NULL},
{ 8, 1074, "SetConsoleCursorPosition",                      NULL},
{12, 1075, "SetConsoleDisplayMode",                         NULL},
{ 0, 1076, "SetConsoleFont",                                NULL},
{ 0, 1077, "SetConsoleHardwareState",                       NULL},
{ 4, 1078, "SetConsoleHistoryInfo",                         NULL},
{ 0, 1079, "SetConsoleIcon",                                NULL},
{ 0, 1080, "SetConsoleInputExeNameA",                       NULL},
{ 0, 1081, "SetConsoleInputExeNameW",                       NULL},
{ 0, 1082, "SetConsoleKeyShortcuts",                        NULL},
{ 0, 1083, "SetConsoleLocalEUDC",                           NULL},
{ 0, 1084, "SetConsoleMaximumWindowSize",                   NULL},
{ 0, 1085, "SetConsoleMenuClose",                           NULL},
{ 8, 1086, "SetConsoleMode",                                NULL},
{ 0, 1087, "SetConsoleNlsMode",                             NULL},
{ 0, 1088, "SetConsoleNumberOfCommandsA",                   NULL},
{ 0, 1089, "SetConsoleNumberOfCommandsW",                   NULL},
{ 0, 1090, "SetConsoleOS2OemFormat",                        NULL},
{ 4, 1091, "SetConsoleOutputCP",                            NULL},
{ 0, 1092, "SetConsolePalette",                             NULL},
{ 8, 1093, "SetConsoleScreenBufferInfoEx",                  NULL},
{ 8, 1094, "SetConsoleScreenBufferSize",                    NULL},
{ 8, 1095, "SetConsoleTextAttribute",                       NULL},
{ 4, 1096, "SetConsoleTitleA",                              NULL},
{ 4, 1097, "SetConsoleTitleW",                              NULL},
{12, 1098, "SetConsoleWindowInfo",                          NULL},
{ 8, 1099, "SetCriticalSectionSpinCount",                   NULL},
{12, 1100, "SetCurrentConsoleFontEx",                       NULL},
{ 4, 1101, "SetCurrentDirectoryA",                          NULL},
{ 4, 1102, "SetCurrentDirectoryW",                          NULL},
{12, 1103, "SetDefaultCommConfigA",                         NULL},
{12, 1104, "SetDefaultCommConfigW",                         NULL},
{ 4, 1105, "SetDllDirectoryA",                              NULL},
{ 4, 1106, "SetDllDirectoryW",                              NULL},
{ 4, 1107, "SetDynamicTimeZoneInformation",                 NULL},
{ 4, 1108, "SetEndOfFile",                                  NULL},
{ 4, 1109, "SetEnvironmentStringsA",                        NULL},
{ 4, 1110, "SetEnvironmentStringsW",                        NULL},
{ 8, 1111, "SetEnvironmentVariableA",                       NULL},
{ 8, 1112, "SetEnvironmentVariableW",                       NULL},
{ 4, 1113, "SetErrorMode",                                  NULL},
{ 4, 1114, "SetEvent",                                      NULL},
{ 8, 1115, "SetEventWhenCallbackReturns",                   NULL},
{ 0, 1116, "SetExtendedFeaturesMask",                       NULL},
{ 0, 1117, "SetFileApisToANSI",                             NULL},
{ 0, 1118, "SetFileApisToOEM",                              NULL},
{ 8, 1119, "SetFileAttributesA",                            NULL},
{12, 1120, "SetFileAttributesTransactedA",                  NULL},
{12, 1121, "SetFileAttributesTransactedW",                  NULL},
{ 8, 1122, "SetFileAttributesW",                            NULL},
{24, 1123, "SetFileBandwidthReservation",                   NULL},
{ 8, 1124, "SetFileCompletionNotificationModes",            NULL},
{16, 1125, "SetFileInformationByHandle",                    NULL},
{12, 1126, "SetFileIoOverlappedRange",                      NULL},
{16, 1127, "SetFilePointer",                                NULL},
{20, 1128, "SetFilePointerEx",                              NULL},
{ 8, 1129, "SetFileShortNameA",                             NULL},
{ 8, 1130, "SetFileShortNameW",                             NULL},
{16, 1131, "SetFileTime",                                   NULL},
{12, 1132, "SetFileValidData",                              NULL},
{16, 1133, "SetFirmwareEnvironmentVariableA",               NULL},
{16, 1134, "SetFirmwareEnvironmentVariableW",               NULL},
{ 0, 1135, "SetHandleContext",                              NULL},
{ 4, 1136, "SetHandleCount",                                NULL},
{12, 1137, "SetHandleInformation",                          NULL},
{16, 1138, "SetInformationJobObject",                       NULL},
{ 0, 1139, "SetLastConsoleEventActive",                     NULL},
{ 4, 1140, "SetLastError",                                  NULL},
{ 8, 1141, "SetLocalPrimaryComputerNameA",                  NULL},
{ 8, 1142, "SetLocalPrimaryComputerNameW",                  NULL},
{ 4, 1143, "SetLocalTime",                                  NULL},
{12, 1144, "SetLocaleInfoA",                                NULL},
{12, 1145, "SetLocaleInfoW",                                NULL},
{ 8, 1146, "SetMailslotInfo",                               NULL},
{ 8, 1147, "SetMessageWaitingIndicator",                    NULL},
{20, 1148, "SetNamedPipeAttribute",                         NULL},
{16, 1149, "SetNamedPipeHandleState",                       NULL},
{ 8, 1150, "SetPriorityClass",                              kernel32_SetPriorityClass},
{ 8, 1151, "SetProcessAffinityMask",                        NULL},
{ 0, 1152, "SetProcessAffinityUpdateMode",                  NULL},
{ 0, 1153, "SetProcessDEPPolicy",                           NULL},
{ 0, 1154, "SetProcessPreferredUILanguages",                NULL},
{ 8, 1155, "SetProcessPriorityBoost",                       NULL},
{ 8, 1156, "SetProcessShutdownParameters",                  NULL},
{12, 1157, "SetProcessWorkingSetSize",                      NULL},
{16, 1158, "SetProcessWorkingSetSizeEx",                    NULL},
{ 0, 1159, "SetSearchPathMode",                             NULL},
{ 8, 1160, "SetStdHandle",                                  NULL},
{12, 1161, "SetStdHandleEx",                                NULL},
{12, 1162, "SetSystemFileCacheSize",                        NULL},
{ 8, 1163, "SetSystemPowerState",                           NULL},
{ 4, 1164, "SetSystemTime",                                 NULL},
{ 8, 1165, "SetSystemTimeAdjustment",                       NULL},
{12, 1166, "SetTapeParameters",                             NULL},
{24, 1167, "SetTapePosition",                               NULL},
{ 0, 1168, "SetTermsrvAppInstallMode",                      NULL},
{ 8, 1169, "SetThreadAffinityMask",                         NULL},
{ 8, 1170, "SetThreadContext",                              NULL},
{ 0, 1171, "SetThreadErrorMode",                            NULL},
{ 4, 1172, "SetThreadExecutionState",                       NULL},
{ 0, 1173, "SetThreadGroupAffinity",                        NULL},
{ 8, 1174, "SetThreadIdealProcessor",                       NULL},
{ 0, 1175, "SetThreadIdealProcessorEx",                     NULL},
{ 4, 1176, "SetThreadLocale",                               NULL},
{12, 1177, "SetThreadPreferredUILanguages",                 NULL},
{ 8, 1178, "SetThreadPriority",                             NULL},
{ 8, 1179, "SetThreadPriorityBoost",                        NULL},
{ 4, 1180, "SetThreadStackGuarantee",                       NULL},
{ 0, 1181, "SetThreadToken",                                NULL},
{ 4, 1182, "SetThreadUILanguage",                           NULL},
{ 0, 1183, "SetThreadpoolStackInformation",                 NULL},
{ 8, 1184, "SetThreadpoolThreadMaximum",                    NULL},
{ 8, 1185, "SetThreadpoolThreadMinimum",                    NULL},
{16, 1186, "SetThreadpoolTimer",                            NULL},
{12, 1187, "SetThreadpoolWait",                             NULL},
{ 4, 1188, "SetTimeZoneInformation",                        NULL},
{24, 1189, "SetTimerQueueTimer",                            NULL},
{ 4, 1190, "SetUnhandledExceptionFilter",                   NULL},
{ 4, 1191, "SetUserGeoID",                                  NULL},
{ 0, 1192, "SetVDMCurrentDirectories",                      NULL},
{ 8, 1193, "SetVolumeLabelA",                               NULL},
{ 8, 1194, "SetVolumeLabelW",                               NULL},
{ 8, 1195, "SetVolumeMountPointA",                          NULL},
{ 8, 1196, "SetVolumeMountPointW",                          NULL},
{24, 1197, "SetWaitableTimer",                              NULL},
{ 0, 1198, "SetWaitableTimerEx",                            NULL},
{12, 1199, "SetupComm",                                     NULL},
{ 0, 1200, "ShowConsoleCursor",                             NULL},
{16, 1201, "SignalObjectAndWait",                           NULL},
{ 8, 1202, "SizeofResource",                                NULL},
{ 4, 1203, "Sleep",                                         NULL},
{12, 1204, "SleepConditionVariableCS",                      NULL},
{16, 1205, "SleepConditionVariableSRW",                     NULL},
{ 8, 1206, "SleepEx",                                       NULL},
{ 0, 1207, "SortCloseHandle",                               NULL},
{ 0, 1208, "SortGetHandle",                                 NULL},
{ 4, 1209, "StartThreadpoolIo",                             NULL},
{ 4, 1210, "SubmitThreadpoolWork",                          NULL},
{ 4, 1211, "SuspendThread",                                 NULL},
{ 4, 1212, "SwitchToFiber",                                 NULL},
{ 0, 1213, "SwitchToThread",                                NULL},
{ 8, 1214, "SystemTimeToFileTime",                          NULL},
{12, 1215, "SystemTimeToTzSpecificLocalTime",               NULL},
{ 8, 1216, "TerminateJobObject",                            NULL},
{ 8, 1217, "TerminateProcess",                              NULL},
{ 8, 1218, "TerminateThread",                               NULL},
{ 0, 1219, "TermsrvAppInstallMode",                         NULL},
{ 8, 1220, "Thread32First",                                 NULL},
{ 8, 1221, "Thread32Next",                                  NULL},
{ 0, 1222, "TlsAlloc",                                      NULL},
{ 4, 1223, "TlsFree",                                       NULL},
{ 4, 1224, "TlsGetValue",                                   NULL},
{ 8, 1225, "TlsSetValue",                                   NULL},
{20, 1226, "Toolhelp32ReadProcessMemory",                   NULL},
{28, 1227, "TransactNamedPipe",                             NULL},
{ 8, 1228, "TransmitCommChar",                              NULL},
{ 0, 1229, "TryAcquireSRWLockExclusive",                    NULL},
{ 0, 1230, "TryAcquireSRWLockShared",                       NULL},
{ 4, 1231, "TryEnterCriticalSection",                       NULL},
{12, 1232, "TrySubmitThreadpoolCallback",                   NULL},
{12, 1233, "TzSpecificLocalTimeToSystemTime",               NULL},
{ 0, 1234, "UTRegister",                                    NULL},
{ 0, 1235, "UTUnRegister",                                  NULL},
{ 4, 1236, "UnhandledExceptionFilter",                      NULL},
{20, 1237, "UnlockFile",                                    NULL},
{20, 1238, "UnlockFileEx",                                  NULL},
{ 4, 1239, "UnmapViewOfFile",                               NULL},
{ 0, 1240, "UnregisterApplicationRecoveryCallback",         NULL},
{ 0, 1241, "UnregisterApplicationRestart",                  NULL},
{ 0, 1242, "UnregisterConsoleIME",                          NULL},
{ 4, 1243, "UnregisterWait",                                NULL},
{ 8, 1244, "UnregisterWaitEx",                              NULL},
{ 0, 1245, "UpdateCalendarDayOfWeek",                       NULL},
{28, 1246, "UpdateProcThreadAttribute",                     NULL},
{24, 1247, "UpdateResourceA",                               NULL},
{24, 1248, "UpdateResourceW",                               NULL},
{ 0, 1249, "VDMConsoleOperation",                           NULL},
{ 0, 1250, "VDMOperationStarted",                           NULL},
{12, 1251, "VerLanguageNameA",                              NULL},
{12, 1252, "VerLanguageNameW",                              NULL},
{16, 1253, "VerSetConditionMask",                           NULL},
{ 0, 1254, "VerifyConsoleIoHandle",                         NULL},
{20, 1255, "VerifyScripts",                                 NULL},
{16, 1256, "VerifyVersionInfoA",                            NULL},
{16, 1257, "VerifyVersionInfoW",                            NULL},
{16, 1258, "VirtualAlloc",                                  kernel32_VirtualAlloc},
{20, 1259, "VirtualAllocEx",                                kernel32_VirtualAllocEx},
{24, 1260, "VirtualAllocExNuma",                            NULL},
{12, 1261, "VirtualFree",                                   kernel32_VirtualFree},
{16, 1262, "VirtualFreeEx",                                 kernel32_VirtualFreeEx},
{ 8, 1263, "VirtualLock",                                   kernel32_VirtualLock},
{16, 1264, "VirtualProtect",                                kernel32_VirtualProtect},
{20, 1265, "VirtualProtectEx",                              kernel32_VirtualProtectEx},
{12, 1266, "VirtualQuery",                                  NULL},
{16, 1267, "VirtualQueryEx",                                NULL},
{ 8, 1268, "VirtualUnlock",                                 kernel32_VirtualUnlock},
{ 0, 1269, "WTSGetActiveConsoleSessionId",                  NULL},
{12, 1270, "WaitCommEvent",                                 NULL},
{ 8, 1271, "WaitForDebugEvent",                             NULL},
{16, 1272, "WaitForMultipleObjects",                        NULL},
{20, 1273, "WaitForMultipleObjectsEx",                      NULL},
{ 8, 1274, "WaitForSingleObject",                           NULL},
{12, 1275, "WaitForSingleObjectEx",                         NULL},
{ 8, 1276, "WaitForThreadpoolIoCallbacks",                  NULL},
{ 8, 1277, "WaitForThreadpoolTimerCallbacks",               NULL},
{ 8, 1278, "WaitForThreadpoolWaitCallbacks",                NULL},
{ 8, 1279, "WaitForThreadpoolWorkCallbacks",                NULL},
{ 8, 1280, "WaitNamedPipeA",                                NULL},
{ 8, 1281, "WaitNamedPipeW",                                NULL},
{ 4, 1282, "WakeAllConditionVariable",                      NULL},
{ 4, 1283, "WakeConditionVariable",                         NULL},
{ 8, 1284, "WerGetFlags",                                   NULL},
{12, 1285, "WerRegisterFile",                               NULL},
{ 8, 1286, "WerRegisterMemoryBlock",                        NULL},
{ 0, 1287, "WerRegisterRuntimeExceptionModule",             NULL},
{ 4, 1288, "WerSetFlags",                                   NULL},
{ 4, 1289, "WerUnregisterFile",                             NULL},
{ 4, 1290, "WerUnregisterMemoryBlock",                      NULL},
{ 0, 1291, "WerUnregisterRuntimeExceptionModule",           NULL},
{ 0, 1292, "WerpCleanupMessageMapping",                     NULL},
{ 4, 1293, "WerpInitiateRemoteRecovery",                    NULL},
{ 0, 1294, "WerpNotifyLoadStringResource",                  NULL},
{ 0, 1295, "WerpNotifyLoadStringResourceEx",                NULL},
{ 0, 1296, "WerpNotifyUseStringResource",                   NULL},
{ 0, 1297, "WerpStringLookup",                              NULL},
{32, 1298, "WideCharToMultiByte",                           NULL},
{ 8, 1299, "WinExec",                                       NULL},
{ 4, 1300, "Wow64DisableWow64FsRedirection",                NULL},
{ 4, 1301, "Wow64EnableWow64FsRedirection",                 NULL},
{ 8, 1302, "Wow64GetThreadContext",                         NULL},
{ 0, 1303, "Wow64GetThreadSelectorEntry",                   NULL},
{ 4, 1304, "Wow64RevertWow64FsRedirection",                 NULL},
{ 8, 1305, "Wow64SetThreadContext",                         NULL},
{ 4, 1306, "Wow64SuspendThread",                            NULL},
{20, 1307, "WriteConsoleA",                                 NULL},
{16, 1308, "WriteConsoleInputA",                            NULL},
{ 0, 1309, "WriteConsoleInputVDMA",                         NULL},
{ 0, 1310, "WriteConsoleInputVDMW",                         NULL},
{16, 1311, "WriteConsoleInputW",                            NULL},
{20, 1312, "WriteConsoleOutputA",                           NULL},
{20, 1313, "WriteConsoleOutputAttribute",                   NULL},
{20, 1314, "WriteConsoleOutputCharacterA",                  NULL},
{20, 1315, "WriteConsoleOutputCharacterW",                  NULL},
{20, 1316, "WriteConsoleOutputW",                           NULL},
{20, 1317, "WriteConsoleW",                                 NULL},
{20, 1318, "WriteFile",                                     NULL},
{20, 1319, "WriteFileEx",                                   NULL},
{20, 1320, "WriteFileGather",                               NULL},
{12, 1321, "WritePrivateProfileSectionA",                   NULL},
{12, 1322, "WritePrivateProfileSectionW",                   NULL},
{16, 1323, "WritePrivateProfileStringA",                    NULL},
{16, 1324, "WritePrivateProfileStringW",                    NULL},
{20, 1325, "WritePrivateProfileStructA",                    NULL},
{20, 1326, "WritePrivateProfileStructW",                    NULL},
{20, 1327, "WriteProcessMemory",                            NULL},
{ 8, 1328, "WriteProfileSectionA",                          NULL},
{ 8, 1329, "WriteProfileSectionW",                          NULL},
{12, 1330, "WriteProfileStringA",                           NULL},
{12, 1331, "WriteProfileStringW",                           NULL},
{16, 1332, "WriteTapemark",                                 NULL},
{ 4, 1333, "ZombifyActCtx",                                 NULL},
{12, 1334, "_hread",                                        NULL},
{12, 1335, "_hwrite",                                       NULL},
{ 4, 1336, "_lclose",                                       NULL},
{ 8, 1337, "_lcreat",                                       NULL},
{12, 1338, "_llseek",                                       NULL},
{ 8, 1339, "_lopen",                                        NULL},
{12, 1340, "_lread",                                        NULL},
{12, 1341, "_lwrite",                                       NULL},
{ 8, 1342, "lstrcat",                                       NULL},
{ 8, 1343, "lstrcatA",                                      NULL},
{ 8, 1344, "lstrcatW",                                      NULL},
{ 8, 1345, "lstrcmp",                                       NULL},
{ 8, 1346, "lstrcmpA",                                      NULL},
{ 8, 1347, "lstrcmpW",                                      NULL},
{ 8, 1348, "lstrcmpi",                                      NULL},
{ 8, 1349, "lstrcmpiA",                                     NULL},
{ 8, 1350, "lstrcmpiW",                                     NULL},
{ 8, 1351, "lstrcpy",                                       NULL},
{ 8, 1352, "lstrcpyA",                                      NULL},
{ 8, 1353, "lstrcpyW",                                      NULL},
{12, 1354, "lstrcpyn",                                      NULL},
{12, 1355, "lstrcpynA",                                     NULL},
{12, 1356, "lstrcpynW",                                     NULL},
{ 4, 1357, "lstrlen",                                       NULL},
{ 4, 1358, "lstrlenA",                                      NULL},
{ 4, 1359, "lstrlenW",                                      NULL},
{G_MAXINT, 0, NULL, NULL}};
